//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_SynchronizationFence_h)
#define Candera_SynchronizationFence_h

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {

/** @addtogroup CommonBase
 *  @{
 */

    /**
     * @brief Class for creating fence commands and waiting for their completion 
     *  in the RenderDevice command queue.
     *  
     * Each SynchronizationFence object is associated to exactly one fence
     *  command in the command queue. The fence command is added in the queue
     *  when the object is created, and its completion can be queried at any 
     *  time or waited upon.
     * SynchronizationFence objects are not reusable.
     */
    class SynchronizationFence
    {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(SynchronizationFence);

            FEATSTD_LINT_SYMBOL(578, Candera::SynchronizationFence::Handle, "Handle is a WIN32 specific type. Hiding it is acceptable.")
            typedef void* Handle;

            /**
             * Result of a wait on the fence command to be processed.
             */
            enum WaitResult {
                TimeoutExpired,      ///< Timeout expired waiting for the command to be processed.
                Processed,           ///< The command was processed before timeout expiration.
                AlreadyProcessed,    ///< Command already processed, thus no waiting occurs.
                Fail                 ///< Failure to wait for the command to be processed.
            };

            /**
             * Constant value that should be sent as ClientWait timeout parameter
             *  to ignore the timeout.
             */
            static const UInt64 c_IgnoreTimeout;

            /**
             * Create a SynchronizationFence object.
             * 
             * Upon creation, a fence command is added in the RenderDevice command
             *  queue.
             * 
             * @return SharedPointer to a shared SynchronizationFence object.
             */
            static SynchronizationFence::SharedPointer Create();

            /**
             * Check if fence command has already been processed.
             * 
             * @return true if command has been processed, false otherwise.
             */
            bool IsProcessed();

            /**
             * Block the current thread waiting for the fence command to be precessed.
             * 
             * @param timeout Optional timeout value in nanoseconds. If set to 0,
             *   the method will return the result without waiting. If set to the
             *   c_IgnoreTimeout constant, the timeout will be ignored and the 
             *   method will return only when the fence command is processed 
             *   or an error occurs.
             * @param flushBufferedCommands Optional flag that specifies if
             *   buffered commands should be flushed, forcing the fence command
             *   to be processed in finite time. If flush flag is set to false,
             *   the timeout should be set to a finite value.
             * @return Result of the wait command.
             * 
             * @see WaitResult
             */
            WaitResult ClientWait(UInt64 timeout = c_IgnoreTimeout, bool flushBufferedCommands = true);

            ~SynchronizationFence();
        private:
            SynchronizationFence();
            Handle m_fenceHandle;

            CANDERA_SHARED_POINTER_DECLARATION();
    };

 /** @} */ // end of CommonBase

} // namespace Candera

#endif
