//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(StereoProjection_H)
#define StereoProjection_H

#include <Candera/Engine3D/Core/PerspectiveProjection.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
class Projection;

/**
 *  @brief Stereo projection.
 *         This class represents basically a perspective projection, where an offset with respect 
 *         to the eye separation and convergence distance parameters is added to the left and right planes
 *         in order to realize an asymmetric frustum needed for stereoscopic 3D.
 */
 
class StereoProjection : public PerspectiveProjection
{
    FEATSTD_TYPEDEF_BASE(PerspectiveProjection);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(StereoProjection);
        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~StereoProjection() override;

        // Overrides Projection::Clone.
        virtual Projection::SharedPointer Clone() const override;

        FEATSTD_RTTI_DECLARATION();

        /**
         * Sets the eye separation used to calculate the offset for the asymmetric frustum.
         * If the projection is applied on the left camera, separation shall be negative, positive for right cameras.
         * @param separation Eye separation to set.
         */
        void SetEyeSeparation(Float separation);

        /**
         * Gets the eye separation used to calculate the offset for the asymmetric frustum.
         * @return Eye separation of this StereoProjection.
         */
        Float GetEyeSeparation() const { return m_eyeSeparation; }

         /**
         * Sets the convergence distance used to calculate the offset for the asymmetric frustum.
         * The convergence distance is described in detail in StereoCamera.
         * @param distance Convergence distance to set.
         */
        void SetConvergenceDistance(Float distance);

         /**
         * Gets the convergence distance used to calculate the offset for the asymmetric frustum.
         * The convergence distance is described in detail in StereoCamera.
         * @return Convergence distance of this StereoProjection.
         */
         Float GetConvergenceDistance() const { return m_convergenceDistance; }

    protected:
        // override (Projection)
        virtual void UpdateProjectionMatrix() const override;

    private:
        Float m_eyeSeparation;
        Float m_convergenceDistance;

        // private because only ::Create() should be used to create an instance of this class
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::StereoProjection::StereoProjection, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        StereoProjection();
        StereoProjection(const StereoProjection& other);
        StereoProjection& operator=(const StereoProjection& other);
};
 
/** @} */ // end of Core3D
 
} // namespace Candera

#endif

