//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_STEREO_CAMERA_H)
#define CANDERA_STEREO_CAMERA_H

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/SceneListener.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

   class Camera;

/**
 * @brief A StereoCamera is a controller of two assigned Camera objects, used to realize the effect of stereoscopic 3D.
 *        The two cameras serve as left and right eye, which record the same scene from two slightly different
 *        viewing positions, both having the same look-at direction, but different (overlapping) viewing frustums.
 *
 *        Note: A different, but incorrect, approach for stereoscopic 3D, called "toed-in" where both cameras
 *        look at a defined point in the scene also exists. This approach is not implemented here.
 *        This behavior results in wrong differences at the peripheral region of the screen.
 *
 *        The StereoCamera uses the eye separation and convergence distance parameters
 *        to set up both cameras and their projection to produce the stereoscopic 3D effect.
 *
 *        Eye separation controls the interaxial offset of the two Camera objects positions relative to the
 *        StereoCamera's position. Here for both camera's the position is set to half of the separation in +X and -X direction.
 *
 *        Convergence distance is the distance from the camera to the plane on which both projections coincide. Compared to mono projection two different, overlapping viewing frustums are applied,
 *        at convergence distance those overlapping rectangles would be the same.
 *        Convergence distance has a big effect on the perception of stereoscopic images, as it influences where we
 *        perceive the rendered object (in view space):
 *        - Objects which are closer than convergence distance are perceived in front of the screen.
 *        - Objects which are at convergence distance appear to be on the screen. (Hint: Render HUD, Interface, GUI elements here).
 *        - Objects which are farther away than convergence distance appear behind the screen.
 *
 *        To use the StereoCamera object two Camera objects have to be assigned as left and right eye. Only cameras that have no parent,
 *        or their parent is the StereoCamera itself, are allowed to be set as eyes.
 *         If they have no parent,
 *        they'll automatically get assigned as children of this StereoCamera. On these cameras
 *        position, lookAtVector, upVector and the projection get set automatically with respect
 *        to the StereoCamera objects setting.
 *
 *        The render targets associated to the StereoCamera are typically configured by application directly in order to 
 *        support specific output formats and obey application dependencies.
 *        Nevertheless, following very common render target setups are already prepared or can be used directly:
 *        - Display with two inputs: Render left and right camera into different render targets of different displays.
 *        - Side by side: Render both cameras to the same render targets, but apply viewports with half the width.
 *        - Interlaced modes: Render both cameras to the same render targets, use full viewports. Use stencil test and
 *                            a preconfigured stencil mask to render one image into the even and the other image into
 *                            the odd lines.
 *        - Frame packing (HDMI 1.4): Render both images into one render target that has double the height of
 *                                    one camera + 30 black pixels that are rendered in between.
 *
 *        Note: There are no general applicable eye separation and convergence distance factors, as it strongly
 *        depends on your application and the units used for the frustum. Also different screen
 *        sizes have an influence on this effect.
 *
 */
class StereoCamera:  public Node, public ProjectionListener
{
    FEATSTD_TYPEDEF_BASE(Node);

    public:

        /**
        * Creates an instance of this class.
        * Use Dispose() to delete the instance and possible children, if any.
        * @return Pointer to the created StereoCamera node.
        */
        static StereoCamera* Create();

        /**
        *  Clones this Node only.
        *  The clone doesn't have reference to the cameras serving as left and
        *  right eyes. They have to be attached after cloning.
        *  Attached Node resources like Appearance are not deep-copied but
        *  referenced.
        *  @return  The pointer to the cloned Node if successful, otherwise NULL.
        */
        virtual StereoCamera* Clone() const override;

        /**
         * Creates an instance of StereoCamera, using the data provided by
         * the passed (mono) camera.
         * Basically the camera is cloned twice and applied to the new
         * StereoCamera as left and right eye. Then the Cameras transformation and other node properties are applied to the new
         * created StereoCamera node.
         * @param camera Camera to extract data from.
         * @return Pointer to the created StereoCamera node. If the provided camera is 0, a simple created StereoCamera
         *         without any settinghs made is returned.
         */
        static StereoCamera* Create(const Camera* camera);

        /**
         * Destructor.
         */
        virtual ~StereoCamera() override;

        /**
         * Sets a Camera object as left eye of this StereoCamera.
         * Camera will automatically be added as child of this StereoCamera, if it is not already.
         * StereoCamera will be marked to update its left and right cameras (see also function StereoCamera::Update).
         * @param leftEye Camera to be set as left eye.
         * @return True if this camera was successfully added as scene graph child of this StereoCamera.
         *         False if camera has not been set.
         */
        bool SetLeftEye(Camera* leftEye);

        /**
         * Gets the Camera object used as left eye of this StereoCamera.
         * @return Camera that represents the left eye.
         */
        const Camera* GetLeftEye() const { return m_leftEye; }

         /**
         * Sets a Camera object as right eye of this StereoCamera.
         * Camera will automatically be added as child of this StereoCamera, if it is not already.
         * StereoCamera will be marked to update its left and right cameras (see also function StereoCamera::Update).
         * @param rightEye Camera to be set as right eye.
         * @return True if this camera was successfully added as scene graph child of this StereoCamera.
         *         False if camera has not been set.
         */
        bool SetRightEye(Camera* rightEye);

         /**
         * Gets the Camera object used as right eye of this StereoCamera.
         * @return Camera that represents the right eye.
         */
        const Camera* GetRightEye() const { return m_rightEye; }

        /**
         * Sets the StereoCamera's eye separation that controls the interaxial offset of the two left and right Camera objects.
         * @param separation Eye separation to set.
         */
        void SetEyeSeparation(Float separation);
         /**
         * Gets the StereoCamera's eye separation that controls the interaxial offset of the two left and right Camera objects.
         * @return Eye separation of this StereoCamera.
         */
        Float GetEyeSeparation() const { return m_eyeSeparation; }

        /**
         * Sets the convergence distance used to describe at which distance zero parallax shall occur.
         * @param distance The convergence distance to set.
         */
        void SetConvergenceDistance(Float distance);
         /**
         * Gets the convergence distance used to describe at which distance zero parallax shall occur.
         * @return The convergence distance of this StereoCamera..
         */
        Float GetConvergenceDistance() const { return m_convergenceDistance; }

        virtual void OnProjectionChanged(Projection* projection) override;

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        StereoCamera();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(StereoCamera);

         /**
         * Overrides Render from base class to allow instantiation.
         */
        virtual void Render() override {}

        /**
        * Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;

        /**
         * Updates position, lookAtVector, upVector and the projection of left and right eye,
         * if this StereoCamera is marked for update. Resets update flag afterwards.
         * If one of those cameras couldn't be updated (e.g. if it's 0) the
         * update flag remains true until two valid cameras are set.
         */
        void Update();

        /**
         *  Overrides Node::OnAncestorAdded
         *  @param scene Scene to which StereoCameraSceneListener shall be added.
         */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
         *  Overrides Node::OnAncestorRemoved
         *  @param scene Scene to remove StereoCameraSceneListener from.
         */
        virtual void OnAncestorRemoved(Scene* scene) override;

    private:

    /**
     *  @brief The inner class StereoCameraSceneListener notifies the StereoCamera when the associated scene is activated
     *  and it's camera's settings shall be updated due to the StereoCamera's settings.
     */
    class StereoCameraSceneListener : public SceneListener
    {
        public:
            /**
             *  Constructor
             */
            StereoCameraSceneListener(): m_stereoCam(0) {}

            /**
             *  Destructor
             */
            virtual ~StereoCameraSceneListener() override { m_stereoCam = 0; }

            /**
             *  Sets the StereoCamera for the StereoCameraSceneListener.
             *  @param stereoCam The StereoCamera that is set.
             */
            void SetStereoCamera(StereoCamera* stereoCam) { m_stereoCam = stereoCam; }

            /**
             *  Retrieves the StereoCamera from the StereoCameraSceneListener.
             *  @return The StereoCamera from the StereoCameraSceneListener.
             */
            const StereoCamera* GetStereoCamera() const { return m_stereoCam; }

            /**
             *  Overrides OnSceneActivated from SceneListener. Fires Update of the associated
             *  StereoCamera.
             */
            virtual void OnSceneActivated(Scene* scene) override;

        private:
            StereoCamera* m_stereoCam;
    };

        bool m_isCameraUpdateNeeded;
        Camera* m_leftEye;
        Camera* m_rightEye;
        Float m_eyeSeparation;
        Float m_convergenceDistance;

        StereoCameraSceneListener m_listener;

        CdaDynamicProperties(Candera::StereoCamera, Candera::Group);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_GROUP_H
