//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SKYBOX_H)
#define CANDERA_SKYBOX_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief A SkyBox is an axis aligned cube in world-space, which is used to display the background environment of a scene. The skybox cube is centered
 *        on the current camera's position, which creates the illusion that the background image is infinitely far away.
 *
 *        By default, a fixed-size cube mesh is generated, which is rendered at the camera's current position, before any other node is rendered. 
 *        Depth writing and testing is turned off while rendering the skybox, to ensure that it appears behind all other nodes.
 *
 *        Internally, the cube is represented by a Mesh, which can be set and retrieved by functions SetSkyBoxMesh and GetSkyBoxMesh.
 *        The SkyBox is initialized per default, with:
 *         - A mesh that owns a vertex buffer representing a cube,
 *         - RenderMode that has depth writing and testing disabled,
 *         - Default CubeMap Texture.
 *        For a custom skybox definition, typically only a CubeMapTextureImage and a matching shader (See reference shader RefTransCubeMapTexturing_RefCubeMapTex)
 *        has to be set, explicitly.
 */
class SkyBox: public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        /**
        *  Creates an instance of this class.
        *  @return A MemoryManagement::SharedPointer that manages the lifetime of the instance.
        */
        static MemoryManagement::SharedPointer<SkyBox> Create();

        /**
         *  Destructor
         */
        virtual ~SkyBox() override;

        /**
         * Updates the world position of the skybox mesh and renders it.
         * @return True if mesh was rendered, false otherwise.
         */
        bool Activate();

        /**
         * Uploads the skybox mesh to VRAM.
         * @return True if mesh was uploaded to VRAM, false otherwise.
         */
        virtual bool Upload();

        /**
         * Unloads the skybox mesh from VRAM.
         * @return True if mesh was unloaded from VRAM, false otherwise.
         */
        virtual bool Unload();

        /**
         * Sets the mesh that will be rendered. Basically this mesh is rendered before scene graph traversal, it's
         * rendering is triggered by Camera::Activate. Per default it contains a cube and all necessary
         * settings to use the mesh as SkyBox.
         * Note: Only meshes which are not part of a SceneGraph shall be used, otherwise it's possible
         * that it can be disposed multiple times, which would result in an error. 
         * @param skybox Mesh to use for rendering the SkyBox.
         */
        void SetSkyBoxMesh(Mesh* skybox);

        /**
         * Gets the mesh that will be rendered. Basically this mesh is rendered before scene graph traversal, it's
         * rendering is triggered by Camera::Activate. Per default it contains a cube and all necessary
         * settings to use the mesh as SkyBox.
         * Note: Only meshes which are not part of a SceneGraph shall be used, otherwise it's possible
         * that it can be disposed multiple times, which would result in an error.
         * @return Mesh used for rendering the SkyBox.
         */
        Mesh* GetSkyBoxMesh() { return m_skyBoxMesh; }

    private:

        Mesh* m_skyBoxMesh;

        void CreateSkyBoxMesh();

        // Private because only ::Create() should be used to create an instance of this class.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::SkyBox::SkyBox, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        SkyBox(): m_skyBoxMesh(0) {}
        SkyBox(const SkyBox& appearance);
        SkyBox& operator = (const SkyBox& appearance);

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

};

/** @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_SKYBOX_H

