//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_ShaderParamNames_h)
#define Candera_ShaderParamNames_h

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */
class ShaderParamNames {
    public:

        /**
         *  Enumerator UniformSemantic enfolds all indices for the uniform names used by Candera shaders.
         */
        enum UniformSemantic {
            ModelMatrix3,                   ///< 3x3 model matrix in world space used to transform three component vectors.
            ModelViewMatrix3,               ///< 3x3 model-view matrix in eye space used to transform three component vectors.
            NormalModelMatrix3,             ///< 3x3 model matrix to transform normal directions in world space.
            NormalModelViewMatrix3,         ///< 3x3 model-view matrix to transform normal directions in camera space.
            ModelMatrix4,                   ///< 4x4 model matrix in world space used to transform vertices.
            ModelViewMatrix4,               ///< 4x4 model-view matrix in eye space used to transform vertices.
            ProjectionMatrix4,              ///< 4x4 projection matrix.
            ModelViewProjectionMatrix4,     ///< 4x4 model-view-projection matrix used to transform vertices.
            CameraLookAtVector,             ///< Look-at vector of camera.
            CameraPosition,                 ///< Position of camera in world space.
            PointSpriteSize,                ///< Size of point sprite.

            MaterialAmbient,                ///< Material's ambient color.
            MaterialDiffuse,                ///< Material's diffuse color.
            MaterialEmissive,               ///< Material's emissive color.
            MaterialSpecular,               ///< Material's specular color.
            MaterialSpecularPower,          ///< Material's shininess factor.

            Texture,                        ///< Texture at unit 0
            Texture1,                       ///< Texture at unit 1
            Texture2,                       ///< Texture at unit 2
            Texture3,                       ///< Texture at unit 3
            Texture4,                       ///< Texture at unit 4
            Texture5,                       ///< Texture at unit 5
            Texture6,                       ///< Texture at unit 6
            Texture7,                       ///< Texture at unit 7

            CubeMapTexture,                 ///< CubeMapTexture at unit 0
            CubeMapTexture1,                ///< CubeMapTexture at unit 1
            CubeMapTexture2,                ///< CubeMapTexture at unit 2
            CubeMapTexture3,                ///< CubeMapTexture at unit 3
            CubeMapTexture4,                ///< CubeMapTexture at unit 4
            CubeMapTexture5,                ///< CubeMapTexture at unit 5
            CubeMapTexture6,                ///< CubeMapTexture at unit 6
            CubeMapTexture7,                ///< CubeMapTexture at unit 7

            LightType,                      ///< Type of light 0
            LightType1,                     ///< Type of light 1
            LightType2,                     ///< Type of light 2
            LightType3,                     ///< Type of light 3
            LightType4,                     ///< Type of light 4
            LightType5,                     ///< Type of light 5
            LightType6,                     ///< Type of light 6
            LightType7,                     ///< Type of light 7

            LightAmbient,                   ///< Ambient color of light 0
            LightAmbient1,                  ///< Ambient color of light 1
            LightAmbient2,                  ///< Ambient color of light 2
            LightAmbient3,                  ///< Ambient color of light 3
            LightAmbient4,                  ///< Ambient color of light 4
            LightAmbient5,                  ///< Ambient color of light 5
            LightAmbient6,                  ///< Ambient color of light 6
            LightAmbient7,                  ///< Ambient color of light 7

            LightDiffuse,                   ///< Diffuse color of light 0
            LightDiffuse1,                  ///< Diffuse color of light 1
            LightDiffuse2,                  ///< Diffuse color of light 2
            LightDiffuse3,                  ///< Diffuse color of light 3
            LightDiffuse4,                  ///< Diffuse color of light 4
            LightDiffuse5,                  ///< Diffuse color of light 5
            LightDiffuse6,                  ///< Diffuse color of light 6
            LightDiffuse7,                  ///< Diffuse color of light 7

            LightSpecular,                  ///< Specular color of light 0
            LightSpecular1,                 ///< Specular color of light 1
            LightSpecular2,                 ///< Specular color of light 2
            LightSpecular3,                 ///< Specular color of light 3
            LightSpecular4,                 ///< Specular color of light 4
            LightSpecular5,                 ///< Specular color of light 5
            LightSpecular6,                 ///< Specular color of light 6
            LightSpecular7,                 ///< Specular color of light 7

            LightPosition,                  ///< Position of light 0
            LightPosition1,                 ///< Position of light 1
            LightPosition2,                 ///< Position of light 2
            LightPosition3,                 ///< Position of light 3
            LightPosition4,                 ///< Position of light 4
            LightPosition5,                 ///< Position of light 5
            LightPosition6,                 ///< Position of light 6
            LightPosition7,                 ///< Position of light 7

            LightDirection,                 ///< Direction of light 0
            LightDirection1,                ///< Direction of light 1
            LightDirection2,                ///< Direction of light 2
            LightDirection3,                ///< Direction of light 3
            LightDirection4,                ///< Direction of light 4
            LightDirection5,                ///< Direction of light 5
            LightDirection6,                ///< Direction of light 6
            LightDirection7,                ///< Direction of light 7

            LightHalfplane,                 ///< Halfplane of light 0
            LightHalfplane1,                ///< Halfplane of light 1
            LightHalfplane2,                ///< Halfplane of light 2
            LightHalfplane3,                ///< Halfplane of light 3
            LightHalfplane4,                ///< Halfplane of light 4
            LightHalfplane5,                ///< Halfplane of light 5
            LightHalfplane6,                ///< Halfplane of light 6
            LightHalfplane7,                ///< Halfplane of light 7

            LightAttenuation,               ///< Attenuation of light 0
            LightAttenuation1,              ///< Attenuation of light 1
            LightAttenuation2,              ///< Attenuation of light 2
            LightAttenuation3,              ///< Attenuation of light 3
            LightAttenuation4,              ///< Attenuation of light 4
            LightAttenuation5,              ///< Attenuation of light 5
            LightAttenuation6,              ///< Attenuation of light 6
            LightAttenuation7,              ///< Attenuation of light 7

            LightSpotCosCutoff,             ///< Spot's cut-off angle of light 0
            LightSpotCosCutoff1,            ///< Spot's cut-off angle of light 1
            LightSpotCosCutoff2,            ///< Spot's cut-off angle of light 2
            LightSpotCosCutoff3,            ///< Spot's cut-off angle of light 3
            LightSpotCosCutoff4,            ///< Spot's cut-off angle of light 4
            LightSpotCosCutoff5,            ///< Spot's cut-off angle of light 5
            LightSpotCosCutoff6,            ///< Spot's cut-off angle of light 6
            LightSpotCosCutoff7,            ///< Spot's cut-off angle of light 7

            LightSpotExponent,              ///< Spot's exponent of light 0
            LightSpotExponent1,             ///< Spot's exponent of light 1
            LightSpotExponent2,             ///< Spot's exponent of light 2
            LightSpotExponent3,             ///< Spot's exponent of light 3
            LightSpotExponent4,             ///< Spot's exponent of light 4
            LightSpotExponent5,             ///< Spot's exponent of light 5
            LightSpotExponent6,             ///< Spot's exponent of light 6
            LightSpotExponent7,             ///< Spot's exponent of light 7

            LightRange,                     ///< Range of light 0
            LightRange1,                    ///< Range of light 1
            LightRange2,                    ///< Range of light 2
            LightRange3,                    ///< Range of light 3
            LightRange4,                    ///< Range of light 4
            LightRange5,                    ///< Range of light 5
            LightRange6,                    ///< Range of light 6
            LightRange7,                    ///< Range of light 7

            LightEnabled,                   ///< Specifies whether light 0 is enabled or not.
            LightEnabled1,                  ///< Specifies whether light 1 is enabled or not.
            LightEnabled2,                  ///< Specifies whether light 2 is enabled or not.
            LightEnabled3,                  ///< Specifies whether light 3 is enabled or not.
            LightEnabled4,                  ///< Specifies whether light 4 is enabled or not.
            LightEnabled5,                  ///< Specifies whether light 5 is enabled or not.
            LightEnabled6,                  ///< Specifies whether light 6 is enabled or not.
            LightEnabled7,                  ///< Specifies whether light 7 is enabled or not.

            LightCameraLookAtVector,        ///< Camera look-at vector of Light 0
            LightCameraLookAtVector1,       ///< Camera look-at vector of Light 1
            LightCameraLookAtVector2,       ///< Camera look-at vector of Light 2
            LightCameraLookAtVector3,       ///< Camera look-at vector of Light 3 
            LightCameraLookAtVector4,       ///< Camera look-at vector of Light 4
            LightCameraLookAtVector5,       ///< Camera look-at vector of Light 5
            LightCameraLookAtVector6,       ///< Camera look-at vector of Light 6
            LightCameraLookAtVector7,       ///< Camera look-at vector of Light 7

            MorphWeightArray,               ///< Specifies an array of morph weights (see class MorphingMesh).

            CanvasPivot,                    ///< Specifies the pivot of a Canvas vertex buffer. The vertex is translated before mvp transformation using this pivot as an offset.
            CanvasSize,                     ///< Specifies the size of a Canvas vertex buffer. The vertex is scaled before mvp transformation using this size.

            MaterialBlock,                  ///< Material properties (see above) as a uniform block.
            LightsBlock,                    ///< Light properties from Light0-7 (see above) as a uniform block.

            UniformSemanticCount            ///< Number of uniform semantics.
        };

        /**
         *  Enumerator AttributeSemantic enfolds all indices for the default attribute names used by Candera shaders. Typically there are 8 for each type,
         *  which are named according to the following pattern: SomeAttribute, SomeAttribute1, ... SomeAttribute7. Only the first of each type is documented 
         *  here.
         */
        enum AttributeSemantic {
            PositionAttribute,              ///< Position attribute 0 to 7.
            PositionAttribute1,
            PositionAttribute2,
            PositionAttribute3,
            PositionAttribute4,
            PositionAttribute5,
            PositionAttribute6,
            PositionAttribute7,

            PositionTransformedAttribute,   ///< Transformed position attribute 0 to 7.
            PositionTransformedAttribute1,
            PositionTransformedAttribute2,
            PositionTransformedAttribute3,
            PositionTransformedAttribute4,
            PositionTransformedAttribute5,
            PositionTransformedAttribute6,
            PositionTransformedAttribute7,

            NormalAttribute,                ///< Normal vector attribute 0 to 7.
            NormalAttribute1,
            NormalAttribute2,
            NormalAttribute3,
            NormalAttribute4,
            NormalAttribute5,
            NormalAttribute6,
            NormalAttribute7,

            TextureCoordinateAttribute,     ///< Texture UV coordinate attribute 0 to 7.
            TextureCoordinateAttribute1,
            TextureCoordinateAttribute2,
            TextureCoordinateAttribute3,
            TextureCoordinateAttribute4,
            TextureCoordinateAttribute5,
            TextureCoordinateAttribute6,
            TextureCoordinateAttribute7,

            ColorAttribute,                 ///< Color attribute 0 to 7.
            ColorAttribute1,
            ColorAttribute2,
            ColorAttribute3,
            ColorAttribute4,
            ColorAttribute5,
            ColorAttribute6,
            ColorAttribute7,

            BlendWeightAttribute,           ///< Blend weight attributes 0 to 7.
            BlendWeightAttribute1,
            BlendWeightAttribute2,
            BlendWeightAttribute3,
            BlendWeightAttribute4,
            BlendWeightAttribute5,
            BlendWeightAttribute6,
            BlendWeightAttribute7,

            BlendIndexAttribute,            ///< Blend index attributes 0 to 7.
            BlendIndexAttribute1,
            BlendIndexAttribute2,
            BlendIndexAttribute3,
            BlendIndexAttribute4,
            BlendIndexAttribute5,
            BlendIndexAttribute6,
            BlendIndexAttribute7,


            PointSizeAttribute,            ///< Point size attributes 0 to 7.
            PointSizeAttribute1,
            PointSizeAttribute2,
            PointSizeAttribute3,
            PointSizeAttribute4,
            PointSizeAttribute5,
            PointSizeAttribute6,
            PointSizeAttribute7,

            TangentAttribute,               ///< Tangent attributes 0 to 7.
            TangentAttribute1,
            TangentAttribute2,
            TangentAttribute3,
            TangentAttribute4,
            TangentAttribute5,
            TangentAttribute6,
            TangentAttribute7,

            BiNormalAttribute,              ///< Bi-Normal attributes 0 to 7.
            BiNormalAttribute1,
            BiNormalAttribute2,
            BiNormalAttribute3,
            BiNormalAttribute4,
            BiNormalAttribute5,
            BiNormalAttribute6,
            BiNormalAttribute7,

            TesselationFactorAttribute,     ///< Tessellation factor attributes 0 to 7.
            TesselationFactorAttribute1,
            TesselationFactorAttribute2,
            TesselationFactorAttribute3,
            TesselationFactorAttribute4,
            TesselationFactorAttribute5,
            TesselationFactorAttribute6,
            TesselationFactorAttribute7,

            FogAttribute,                   ///< Fog attributes 0 to 7.
            FogAttribute1,
            FogAttribute2,
            FogAttribute3,
            FogAttribute4,
            FogAttribute5,
            FogAttribute6,
            FogAttribute7,

            DepthAttribute,                 ///< Depth attributes 0 to 7.
            DepthAttribute1,
            DepthAttribute2,
            DepthAttribute3,
            DepthAttribute4,
            DepthAttribute5,
            DepthAttribute6,
            DepthAttribute7,

            SampleAttribute,                ///< Sample attributes 0 to 7.
            SampleAttribute1,
            SampleAttribute2,
            SampleAttribute3,
            SampleAttribute4,
            SampleAttribute5,
            SampleAttribute6,
            SampleAttribute7,

            CustomAttribute,                ///< Custom attributes 0 to 7.
            CustomAttribute1,
            CustomAttribute2,
            CustomAttribute3,
            CustomAttribute4,
            CustomAttribute5,
            CustomAttribute6,
            CustomAttribute7,

            AttributeSemanticCount          ///< Total number of attribute semantics.
        } ;


        /**
         *  Sets the uniform name for a certain uniform semantic that is used in Candera shaders.
         *  @param semantic UniformSemantic for which a name shall be assigned.
         *  @param name     Name to be assigned to a uniform semantic.
         */
        static void SetUniformName(UniformSemantic semantic, const Char* name);

        /**
         *  Returns the uniform name for a certain uniform semantic.
         *  @param semantic UniformSemantic to which name shall be retrieved.
         *  @return         Name assigned to semantic. 
         */
        static const Char* GetUniformName(UniformSemantic semantic) { return m_uniformNames[semantic]; }

        /**
         *  Returns the hash of the uniform name of the given uniform semantic.
         *  @param semantic UniformSemantic for which the hash of the uniform name is returned.
         *  @return         The hash of the uniform name of the given uniform semantic.
         */
        static UInt32 GetUniformNameHash(UniformSemantic semantic) { return m_uniformNamesHashes[semantic]; }

        /**
         *  Returns if the given uniform semantic refers to a uniform block.
         *  @param semantic UniformSemantic for which to return if it refers to a uniform block.
         *  @return         True, if the given uniform semantic refers to a uniform block. False, otherwise.
         */
        static bool IsUniformBlock(UniformSemantic semantic) { return m_isUniformBlock[semantic]; }

        /**
         *  Sets the attribute name for a certain attribute semantic that is used in Candera shaders.
         *  @param semantic AttributeSemantic for which a name shall be assigned.
         *  @param name     Name to be assigned to an attribute semantic.
         */
        static void SetAttributeName(AttributeSemantic semantic, const Char* name);

        /**
         *  Returns the attribute name for a certain attribute semantic.
         *  @param semantic AttributeSemantic to which name shall be retrieved.
         *  @return         Name assigned to semantic. 
         */
        static const Char* GetAttributeName(AttributeSemantic semantic) { return m_attributeNames[semantic]; }

        /**
         *  Returns the hash of the attribute name of the given attribute semantic.
         *  @param semantic UniformSemantic for which the hash of the attribute name is returned.
         *  @return         The hash of the attribute name of the given attribute semantic.
         */
        static UInt32 GetAttributeNameHash(AttributeSemantic semantic) { return m_attributeNamesHashes[semantic]; }

        /**
         * AttributeSemantic represents the ShaderParamNames::AttributeSemantic enum value that points to correct attribute 
         * string in ShaderParamNames::m_attributeNames[]. 
         * attributeSemantic = (vertexElement.Usage minus VertexGeometry::Position offset) 
         *          multiplied by an attribute group offset of 8 plus the attribute index.
         */
        static ShaderParamNames::AttributeSemantic GetAttributeSemantic(VertexGeometry::VertexUsage usage, UInt8 usageIndex);

        /**
         *  Initialize the array of hashes for the uniform names.
         */
        static void InitUniformNamesHashes();

        /**
         *  Initialize the array of hashes for the attribute names.
         */
        static void InitAttributeNamesHashes();

    private:
        static const Char* m_uniformNames[];
        static UInt32 m_uniformNamesHashes[];
        static bool m_areUniformNamesHashesInitialized;
        static const bool m_isUniformBlock[];

        static const Char* m_attributeNames[];
        static UInt32 m_attributeNamesHashes[];
        static bool m_areAttributeNamesHashesInitialized;

        static void CheckSizes();
};

/**  @} */ // end of Core3D

} // namespace Candera

#endif    // Candera_ShaderParamNames_h
