//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Shader.h"
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/UniformNode.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
#include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageProvider.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageInterface.h>
#endif //
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <FeatStd/Util/Hash.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    FEATSTD_RTTI_DEFINITION(Shader, Base)

    Shader::Shader() :
        Base(),
        m_vertexShaderResourceHandle(ResourceDataHandle::InvalidHandle()),
        m_fragmentShaderResourceHandle(ResourceDataHandle::InvalidHandle()),
        m_maxInstanceCount(1),
        m_currentBindingPoint(0),
        m_transformVaryings(0),
        m_transformVaryingsCount(0)
    {
        MemoryPlatform::Set(m_programMemoryHandle, 0, sizeof(m_programMemoryHandle));
        MemoryPlatform::Set(m_vertexShaderMemoryHandle, 0, sizeof(m_vertexShaderMemoryHandle));
        MemoryPlatform::Set(m_fragmentShaderMemoryHandle, 0, sizeof(m_fragmentShaderMemoryHandle));

        // Using LinearIncreasePolicy<1>, so we don't want to increase by c_MemoryInitialCapacity on first Add.
        static_cast<void>(m_uniformAccessorIdCache.Reserve(1));

        RegisterInstance();
    }

    SharedPointer<Shader> Shader::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        Shader* ptr = FEATSTD_NEW(Shader)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("Shader create failed, out of memory.");
        }
        SharedPointer<Shader> sharedPointer(ptr);
        return sharedPointer;
    }

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1579, "shader source code is disposed by the disposers; the disposers have external lifetime")
    Shader::~Shader()
    {
        static_cast<void>(Unload(ForceAll));
        Shader::DisposeInternal();

        DeregisterInstance();
        ClearUniformAccessorIdCache();
        ClearCustomUniformCache();
    }

    bool Shader::Activate() const
    {
        m_currentBindingPoint = 0;
        return Renderer::ActivateShader(*this, m_programMemoryHandle[ContextResourcePool::GetActive().GetIndex()]);
    }

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Shader::BindAttributes, "function has side effects and is therefore considered non-const")
    bool Shader::BindAttributes(const SharedPointer<VertexBuffer>& vertexBuffer, UInt divisor)
    {
        if (vertexBuffer == 0) {
            return false;
        }

        const VertexGeometry* vertexGeometry = vertexBuffer->GetVertexGeometry();
        if (vertexGeometry == 0) {
            return false;
        }

        UInt16 attribute_count = vertexGeometry->GetVertexElementCount();
        VertexGeometry::FormatArrayResource formatArrayResource(vertexGeometry->GetFormatArrayResourceHandle());
        const VertexGeometry::VertexElementFormat* vertexElements = formatArrayResource.GetData();

        if ((m_programMemoryHandle[ContextResourcePool::GetActive().GetIndex()] == 0) || (vertexElements == 0)) {
            return false;
        }

        bool result = true;
        const UInt maxVertexAttribs = RenderDevice::GetMaxVertexAttribsSupportedByDevice();
        UInt vertexAttribsEnabled = 0;
        const UInt vertexAttribsEnabledSize = static_cast<UInt>(sizeof(vertexAttribsEnabled) * 8);
        if (maxVertexAttribs > vertexAttribsEnabledSize) {
            FEATSTD_LOG_ERROR("Maximum vertex attibutes supported by Candera is %d, but the device has %d.", vertexAttribsEnabledSize, maxVertexAttribs);
            return false;
        }

        for (UInt16 i = 0; i < attribute_count; ++i) {
            VertexGeometry::VertexElementFormat vertexElement = vertexElements[i];

            ShaderParamNames::AttributeSemantic attributeSemantic =
                ShaderParamNames::GetAttributeSemantic(vertexElement.Usage, vertexElement.UsageIndex);
            Int attributeIndex = GetAttributeLocation(attributeSemantic);

            if (attributeIndex >= 0) {
                vertexAttribsEnabled = vertexAttribsEnabled | (static_cast<UInt>(1) << static_cast<UInt>(attributeIndex));
                if (divisor == 0) {
                    switch (vertexGeometry->GetMemoryPool()) {
                        case VertexGeometry::VideoMemory: {
                            if (!RenderDevice::BindAttribute(attributeIndex,
                                                             vertexElement,
                                                             vertexGeometry->GetVertexStride())) {
                                result = false;
                            }
                            break;
                        }

                        case VertexGeometry::SystemMemory: {
                            if (!RenderDevice::BindClientAttribute(attributeIndex,
                                                                   vertexElement,
                                                                   vertexGeometry->GetVertexStride(),
                                                                   VertexGeometry::VertexArrayResource(vertexGeometry->GetVertexArrayResourceHandle()).GetData())) {
                                result = false;
                            }
                            break;
                        }

                        default:
                            break;
                    }
                }
                else {
                    switch (vertexGeometry->GetMemoryPool()) {
                        case VertexGeometry::VideoMemory: {
                            if (!RenderDevice::BindInstanceAttribute(attributeIndex,
                                vertexElement,
                                vertexGeometry->GetVertexStride(),
                                divisor)) {
                                result = false;
                            }
                            break;
                        }

                        case VertexGeometry::SystemMemory: {
                            if (!RenderDevice::BindInstanceClientAttribute(attributeIndex,
                                vertexElement,
                                vertexGeometry->GetVertexStride(),
                                VertexGeometry::VertexArrayResource(vertexGeometry->GetVertexArrayResourceHandle()).GetData(),
                                divisor)) {
                                result = false;
                            }
                            break;
                        }

                        default:
                            break;
                    }
                }
            }
        }

        UInt vertexAttribsEnabledBit = 1;
        for (UInt i = 0; i < maxVertexAttribs; ++i) {
            if ((vertexAttribsEnabled & vertexAttribsEnabledBit) == 0) {
                result = RenderDevice::DeactivateAttribute(i) && result;
            }

            vertexAttribsEnabledBit = vertexAttribsEnabledBit << 1;
        }

        return result;
    }

    bool Shader::GetActiveUniformCount(Int& count) const
    {
        return RenderDevice::GetActiveUniformCount(GetProgramHandle(), count);
    }

    bool Shader::GetActiveUniformMaxLength(Int& maxLength) const
    {
        return RenderDevice::GetActiveUniformMaxLength(GetProgramHandle(), maxLength);
    }

    bool Shader::GetActiveUniformInfo(Int index, Int nameBufferSize, Char* name, Int& nameLength, Int& size, Shader::UniformType& type) const
    {
        return RenderDevice::GetActiveUniform(GetProgramHandle(), index, nameBufferSize, name, nameLength, size, type);
    }

    SizeType Shader::GetSize(UniformType uniformType)
    {
        SizeType size = 0;
        switch (uniformType) {
        case Shader::Float:     size = sizeof(Float); break;
        case Shader::FloatVec2: size = sizeof(Float) * 2; break;
        case Shader::FloatVec3: size = sizeof(Float) * 3; break;
        case Shader::FloatVec4: size = sizeof(Float) * 4; break;

        case Shader::Integer:     size = sizeof(Int32); break;
        case Shader::IntegerVec2: size = sizeof(Int32) * 2; break;
        case Shader::IntegerVec3: size = sizeof(Int32) * 3; break;
        case Shader::IntegerVec4: size = sizeof(Int32) * 4; break;

        case Shader::Bool:     size = sizeof(Int32); break;
        case Shader::BoolVec2: size = sizeof(Int32) * 2; break;
        case Shader::BoolVec3: size = sizeof(Int32) * 3; break;
        case Shader::BoolVec4: size = sizeof(Int32) * 4; break;

        case Shader::FloatMat2: size = sizeof(Float) * 4; break;
        case Shader::FloatMat3: size = sizeof(Float) * 9; break;
        case Shader::FloatMat4: size = sizeof(Float) * 16; break;

        case Shader::Sampler2D:   size = sizeof(Int32); break;
        case Shader::SamplerCube: size = sizeof(Int32); break;

        default:
            // If this hits, either a unknown uniformType has been passed, or
            // a new type has been introduced and needs to be handled above.
            FEATSTD_LOG_ERROR("Unknown UniformType.");
            FEATSTD_DEBUG_ASSERT(false);
            break;
        }

        return size;
    }

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Shader::SetUniform, CANDERA_LINT_REASON_NONCONSTMETHOD)
    bool Shader::SetUniform(const void* data, const Char* uniformName, UniformType type, UInt count)
    {
        return Renderer::SetUniformInRenderDevice(GetProgramHandle(), uniformName, data, type, count);
    }

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Shader::SetUniform, CANDERA_LINT_REASON_NONCONSTMETHOD)
    bool Shader::SetUniform(const void* data, Int uniformLocation, UniformType type, UInt count, Int instanceIndex)
    {
        return Renderer::SetUniform(uniformLocation, data, type, count, instanceIndex);
    }

    bool Shader::SetConstantVertexAttribute(const void* data, const Char* name, Shader::ConstantAttributeType type) const
    {
        return RenderDevice::SetConstantVertexAttribute(GetProgramHandle(), name, data, type);
    }


    bool Shader::SetConstantVertexAttribute(const void* data, ShaderParamNames::AttributeSemantic attributeSemantic, ConstantAttributeType type) const
    {
        Int attributeLocation = GetAttributeLocation(attributeSemantic);
        if (attributeLocation != -1) {
            return RenderDevice::SetConstantVertexAttribute(attributeLocation, data, type);
        }
        return false;
    }

    bool Shader::SetConstantVertexAttribute(const void* data, Int attributeLocation, ConstantAttributeType type) const
    {
        return RenderDevice::SetConstantVertexAttribute(attributeLocation, data, type);
    }

    bool Shader::SetShaderResourceHandle(ResourceDataHandle& handle, const ResourceDataHandle& source) const
    {
        if (IsUploaded()) {
            return false; // modification after upload not allowed
        }
        ShaderResource::DisposeData(handle);
        handle = source;

        return true;
    }

    bool Shader::UploadInternal(LoadingHint loadingHint)
    {
        bool result = false;
#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
        ShaderResource vertexShaderResource(m_vertexShaderResourceHandle);
        const void* vertexShader = vertexShaderResource.GetData();

        ShaderResource fragmentShaderResource(m_fragmentShaderResourceHandle);
        const void* fragmentShader = fragmentShaderResource.GetData();

        Shader::BuildType vertexShaderBuildType = Shader::ShaderSource;
        Shader::BuildType fragmentShaderBuildType = Shader::ShaderSource;

        bool hasNilGuid = m_guid.IsNil();
        bool storageEnabled = (0 != GlShaderStorageProvider::GetShaderCache()) && (RenderDevice::GetShaderBuildType(fragmentShader, fragmentShaderBuildType)) && (RenderDevice::GetShaderBuildType(vertexShader, vertexShaderBuildType));

        if (storageEnabled){
            //all text type shaders
            //deletion of GlShaderCacheInterface::CacheObject::m_data has to be handled by concrete implementation of GlShaderCacheInterface
            GlShaderStorageInterface::StorageObject storageObject(*FeatStd::Internal::PointerToPointer<const Char**>(&vertexShader), *FeatStd::Internal::PointerToPointer<const Char**>(&fragmentShader), GetName());

            if (!hasNilGuid){
                FeatStd::Internal::Guid::CopyBytes(m_guid, storageObject.m_identifier, storageObject.IdentifierSize);
            }
            if ((GetName() == 0) && (hasNilGuid)) {
                FEATSTD_LOG_WARN("Shader has neither Name nor Guid set.");
            }
            if (GlShaderStorageProvider::GetShaderCache()->Retrieve(storageObject)){
                result = RenderDevice::UploadStoredShaderBinaryProgram(*this, storageObject);
            }

        }
        if (!result){ // try compile from text
#endif
            result = Renderer::UploadShader(*this, loadingHint);
#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
            if (result && storageEnabled){ // compilation from text successful -> store it
                //remember to handle deletion of GlShaderCacheInterface::CacheObject::m_data
                GlShaderStorageInterface::StorageObject storageObject(*FeatStd::Internal::PointerToPointer<const Char**>(&vertexShader), *FeatStd::Internal::PointerToPointer<const Char**>(&fragmentShader), GetName(), GlShaderStorageInterface::StorageObject::DefaultCleanupProc);

                if (!hasNilGuid){
                    FeatStd::Internal::Guid::CopyBytes(m_guid, storageObject.m_identifier, storageObject.IdentifierSize);
                }
                if ((GetName() == 0) && (hasNilGuid)) {
                    FEATSTD_LOG_WARN("Shader has neither Name nor Guid set.");
                }

                if (RenderDevice::RetrieveShaderBinaryProgram(*this, storageObject)){
                    if (!GlShaderStorageProvider::GetShaderCache()->Persist(storageObject)){
                        FEATSTD_LOG_WARN("Shader could not be persisted.");
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Shader could not be retrieved from driver.");
                }
            }
        }
#endif
        if (!result) {
            return false;
        }

        UpdateAttributeCache();
        UpdateUniformCache();
        return true;
    }

    bool Shader::UnloadInternal(LoadingHint loadingHint)
    {
        m_attributeCache.Clear();
        ClearUniformAccessorIdCache();
        ClearCustomUniformCache();
        m_autoUniformCache.Clear();

        Handle& vertexShaderHandle = m_vertexShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
        Handle& fragmentShaderHandle = m_fragmentShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
        Handle& programHandle = m_programMemoryHandle[ContextResourcePool::GetActive().GetIndex()];

        bool ok = Renderer::DeleteShader(vertexShaderHandle, fragmentShaderHandle, programHandle, GetName(), loadingHint);
        if (ok) {
            vertexShaderHandle = 0;
            fragmentShaderHandle = 0;
            programHandle = 0;
        }
        return ok;
    }

    void Shader::DisposeInternal()
    {
        ShaderResource::DisposeData(m_fragmentShaderResourceHandle);
        ShaderResource::DisposeData(m_vertexShaderResourceHandle);
    }

    Handle Shader::GetProgramHandle() const
    {
        return m_programMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    Handle Shader::GetVertexShaderHandle() const
    {
        return m_vertexShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    Handle Shader::GetFragmentShaderHandle() const
    {
        return m_fragmentShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    void Shader::UpdateAttributeCache()
    {
        ShaderParamNames::InitAttributeNamesHashes();
        m_attributeCache.Clear();

        Int location = -1;
        Handle programHandle = GetProgramHandle();

        Int activeAttributeCount = 0;
        Int activeAttributeNameMaxLength = 0;
        if (RenderDevice::GetActiveAttributeCount(programHandle, activeAttributeCount) &&
            RenderDevice::GetActiveAttributeMaxLength(programHandle, activeAttributeNameMaxLength)) {
            if ((activeAttributeCount > 0) && (activeAttributeNameMaxLength > 0)) {
                Char* activeAttributeName = CANDERA_NEW_ARRAY(Char, activeAttributeNameMaxLength);
                for (Int i = 0; i < activeAttributeCount; ++i) {
                    Int activeAttributeNameLength = 0;
                    Int activeAttributeSize = 0;
                    UniformType activeUniformType;
                    if (RenderDevice::GetActiveAttribute(programHandle, i, activeAttributeNameMaxLength, activeAttributeName, activeAttributeNameLength,
                        activeAttributeSize, activeUniformType)) {
                        const UInt32 activeAttributeNameHash = FeatStd::Hash::CalcHash(activeAttributeName);
                        for (UInt index = 0; index < static_cast<UInt>(ShaderParamNames::AttributeSemanticCount); index++) {
                            ShaderParamNames::AttributeSemantic attributeSemantic = static_cast<ShaderParamNames::AttributeSemantic>(index);
                            if (activeAttributeNameHash == ShaderParamNames::GetAttributeNameHash(attributeSemantic)) {
                                if (0 == StringPlatform::CompareStrings(activeAttributeName, ShaderParamNames::GetAttributeName(attributeSemantic))) {
                                    location = RenderDevice::GetAttributeLocation(programHandle, attributeSemantic);
                                    if (location >= 0) {
                                        static_cast<void>(m_attributeCache.Prepend(AttributeCacheEntry(attributeSemantic, location)));
                                    }
                                    break;
                                }
                            }
                        }
                     }
                }

                CANDERA_DELETE_ARRAY(activeAttributeName);
            }
        }
    }


    Int Shader::GetAttributeLocation(ShaderParamNames::AttributeSemantic attributeSemantic) const
    {
        for (AttributeCache::Iterator it = m_attributeCache.Begin(); it != m_attributeCache.End(); it++) {
            if ((*it).m_attributeSemantic == attributeSemantic) {
                return (*it).m_attributeLocation;
            }
        }

        return -1;
    }


    void Shader::UpdateUniformCache()
    {
        ShaderParamNames::InitUniformNamesHashes();
        ClearUniformAccessorIdCache();
        ClearCustomUniformCache();
        m_autoUniformCache.Clear();

        AddUniformLocationsToCache();
        AddUniformBlockIndicesToCache();
    }

    void Shader::AddUniformLocationsToCache()
    {
        Int location = -1;
        Handle programHandle = GetProgramHandle();

        Int activeUniformCount = 0;
        Int activeUniformNameMaxLength = 0;
        if (GetActiveUniformCount(activeUniformCount) && GetActiveUniformMaxLength(activeUniformNameMaxLength)) {
            if ((activeUniformCount > 0) && (activeUniformNameMaxLength > 0)) {
                Char* activeUniformName = CANDERA_NEW_ARRAY(Char, activeUniformNameMaxLength);
                for (Int i = 0; i < activeUniformCount; ++i) {
                    Int activeUniformNameLength = 0;
                    Int activeUniformSize = 0;
                    UniformType activeUniformType;
                    if (GetActiveUniformInfo(i, activeUniformNameMaxLength, activeUniformName, activeUniformNameLength, activeUniformSize, activeUniformType)) {
                        if ((activeUniformSize > 1) && (activeUniformNameLength > 3)) {
                            // Uniform array can (but do not have to) be returned with a '[0]' postfix.
                            Char* activeUniformNameArrayPostfix = activeUniformName + activeUniformNameLength - 3;
                            if (0 == StringPlatform::CompareStrings(activeUniformNameArrayPostfix, "[0]")) {
                                // If this is the case, ignore the array postfix.
                                activeUniformNameLength -= 3;
                                activeUniformName[activeUniformNameLength] = '\0';
                            }
                        }

                        bool isAutoUniform = false;
                        const UInt32 activeUniformNameHash = FeatStd::Hash::CalcHash(activeUniformName);
                        for (UInt index = 0; index < static_cast<UInt>(ShaderParamNames::UniformSemanticCount); index++) {
                            ShaderParamNames::UniformSemantic uniformSemantic = static_cast<ShaderParamNames::UniformSemantic>(index);
                            if ((!ShaderParamNames::IsUniformBlock(uniformSemantic)) &&
                                (activeUniformNameHash == ShaderParamNames::GetUniformNameHash(uniformSemantic))) {
                                if (0 == StringPlatform::CompareStrings(activeUniformName, ShaderParamNames::GetUniformName(uniformSemantic))) {
                                    if (RenderDevice::GetUniformLocation(programHandle, uniformSemantic, location)) {
                                        static_cast<void>(m_autoUniformCache.Prepend(AutoUniformCacheEntry(uniformSemantic, location)));
                                    }
                                    isAutoUniform = true;
                                    break;
                                }
                            }
                        }

                        if (!isAutoUniform) {
                            Char* uniformNameCacheCopy = CANDERA_NEW_ARRAY(Char, static_cast<SizeType>(activeUniformNameLength + 1));
                            StringPlatform::Copy(uniformNameCacheCopy, activeUniformName);
                            static_cast<void>(RenderDevice::GetUniformLocation(programHandle, uniformNameCacheCopy, location));
                            // It's not an auto uniform, therefore we cache it in the custom uniform cache.
                            static_cast<void>(m_customUniformCache.Prepend(CustomUniformCacheEntry(uniformNameCacheCopy,
                                activeUniformNameHash, location)));
                        }
                    }
                }

                CANDERA_DELETE_ARRAY(activeUniformName);
            }
        }
    }

    void Shader::AddUniformBlockIndicesToCache()
    {
        Int activeUniformBlockCount = 0;
        Int activeUniformBlockNameMaxLength = 0;
        if (Renderer::GetActiveUniformBlockCount(*this, activeUniformBlockCount) &&
            Renderer::GetActiveUniformBlockMaxNameLength(*this, activeUniformBlockNameMaxLength)) {
            if ((activeUniformBlockCount > 0) && (activeUniformBlockNameMaxLength > 0)) {
                Char* activeUniformBlockName = CANDERA_NEW_ARRAY(Char, activeUniformBlockNameMaxLength);
                for (Int blockIndex = 0; blockIndex < activeUniformBlockCount; ++blockIndex) {
                    Int activeUniformBlockNameLength = 0;
                    if (Renderer::GetActiveUniformBlockName(*this, blockIndex, activeUniformBlockNameMaxLength,
                        activeUniformBlockNameLength, activeUniformBlockName)) {
                        bool isAutoUniformBlock = false;
                        const UInt32 activeUniformBlockNameHash = FeatStd::Hash::CalcHash(activeUniformBlockName);
                        for (UInt index = 0; index < static_cast<UInt>(ShaderParamNames::UniformSemanticCount); index++) {
                            ShaderParamNames::UniformSemantic uniformSemantic = static_cast<ShaderParamNames::UniformSemantic>(index);
                            if ((ShaderParamNames::IsUniformBlock(uniformSemantic)) &&
                                (activeUniformBlockNameHash == ShaderParamNames::GetUniformNameHash(uniformSemantic))) {
                                if (0 == StringPlatform::CompareStrings(activeUniformBlockName, ShaderParamNames::GetUniformName(uniformSemantic))) {
                                    if (ShaderParamNames::MaterialBlock == uniformSemantic) {
                                        Int blockSize = 0;
                                        if (!Renderer::GetActiveUniformBlockSize(*this, blockIndex, blockSize)) {
                                            break;
                                        }

                                        if (blockSize != sizeof(Material::UniformBlock)) {
                                            const Char* materialBlockName = ShaderParamNames::GetUniformName(ShaderParamNames::MaterialBlock);
                                            FEATSTD_UNUSED(materialBlockName); // if logging is disabled, variable/parameter is not used
                                            FEATSTD_LOG_ERROR("The size of MaterialBlock '%s' in shader '%s' does not match Candera's MaterialBlock.",
                                                materialBlockName == 0 ? "(null)" : materialBlockName, GetName() == 0 ? "(null)" : GetName());
                                            break;
                                        }
                                    }
                                    else {
                                        if (ShaderParamNames::LightsBlock == uniformSemantic) {
                                            Int blockSize = 0;
                                            if (!Renderer::GetActiveUniformBlockSize(*this, blockIndex, blockSize)) {
                                                break;
                                            }

                                            if (blockSize != (sizeof(Light::UniformBlock) * CANDERA_MAX_LIGHTS_COUNT)) {
                                                const Char* lightsBlockName = ShaderParamNames::GetUniformName(ShaderParamNames::LightsBlock);
                                                FEATSTD_UNUSED(lightsBlockName); // if logging is disabled, variable/parameter is not used
                                                FEATSTD_LOG_ERROR("The size of LightsBlock '%s' in shader '%s' does not match Candera's LightsBlock.",
                                                    lightsBlockName == 0 ? "(null)" : lightsBlockName, GetName() == 0 ? "(null)" : GetName());
                                                break;
                                            }
                                        }
                                    }

                                    static_cast<void>(m_autoUniformCache.Prepend(AutoUniformCacheEntry(uniformSemantic, blockIndex)));
                                    isAutoUniformBlock = true;
                                    break;
                                }
                            }
                        }

                        if (!isAutoUniformBlock) {
                            Char* uniformNameCacheCopy = CANDERA_NEW_ARRAY(Char, static_cast<SizeType>(activeUniformBlockNameLength + 1));
                            StringPlatform::Copy(uniformNameCacheCopy, activeUniformBlockName);
                            static_cast<void>(m_customUniformCache.Prepend(CustomUniformCacheEntry(uniformNameCacheCopy,
                                activeUniformBlockNameHash, blockIndex)));
                        }
                    }
                 }

                CANDERA_DELETE_ARRAY(activeUniformBlockName);
            }
        }
    }

    bool Shader::GetUniformLocation(const Char* uniformName, Int& location) const
    {
        const UInt32 hash = FeatStd::Hash::CalcHash(uniformName);
        for (CustomUniformCache::Iterator it = m_customUniformCache.Begin(); it != m_customUniformCache.End(); it++) {
            const CustomUniformCacheEntry& cacheEntry = *it;
            if (hash == cacheEntry.m_uniformNameHash) {
                if (0 == StringPlatform::CompareStrings(uniformName, cacheEntry.m_uniformName)) {
                    location = cacheEntry.m_uniformLocation;
                    return true;
                }
            }
        }

        // It's not a custom uniform, so it's either:
        // a) not a valid uniform
        // b) an auto uniform given by name instead of UniformSemantic
        // c) an element of a uniform array (e.g. "u_Textures[2]")
        // Neither of these cases is cached by name, so we let the RenderDevice perform the lookup.
        return RenderDevice::GetUniformLocation(GetProgramHandle(), uniformName, location);
    }


    bool Shader::GetUniformLocation(ShaderParamNames::UniformSemantic uniformSemantic, Int& location) const
    {
        location = -1;

        for (AutoUniformCache::Iterator it = m_autoUniformCache.Begin(); it != m_autoUniformCache.End(); it++) {
            if ((*it).m_uniformSemantic == uniformSemantic) {
                location = (*it).m_uniformLocation;
                break;
            }
        }

        return (location >= 0);
    }


    void Shader::SetShaderObjectMemoryHandle(ObjectType type, Handle handle)
    {
        if (type == VertexShader) {
            m_vertexShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
        }
        else if (type == FragmentShader) {
            m_fragmentShaderMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
        }
        else {
            //Intentionally blank.
        }
    }

    void Shader::SetShaderProgramMemoryHandle(Handle handle)
    {
        m_programMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
    }


    bool Shader::GetBuildType(ObjectType objectType, BuildType& shaderBuildType) const
    {
        bool result = false;
        if (objectType == VertexShader) {
            result = RenderDevice::GetShaderBuildType(m_vertexShaderResourceHandle, shaderBuildType);
        }
        else if (objectType == FragmentShader) {
            result = RenderDevice::GetShaderBuildType(m_fragmentShaderResourceHandle, shaderBuildType);
        }
        else {
            //Intentionally blank.
        }

        return result;
    }
    bool Shader::SetVertexShader(const void* vertexShader, DisposerFn disposerFn)
    {
        return SetShaderResourceHandle(m_vertexShaderResourceHandle, ShaderResource::CreateHandle(vertexShader, disposerFn, 0));
    }

    const void* Shader::GetVertexShader() const
    {
        return ShaderResource(GetVertexShaderResourceHandle()).GetData();
    }

    bool Shader::SetFragmentShader(const void* fragmentShader, DisposerFn disposerFn)
    {
        return SetShaderResourceHandle(m_fragmentShaderResourceHandle, ShaderResource::CreateHandle(fragmentShader, disposerFn, 0));
    }

    const void* Shader::GetFragmentShader() const
    {
        return ShaderResource(GetFragmentShaderResourceHandle()).GetData();
    }


    typedef Candera::Internal::Vector<const Shader*> InstancesContainer;
    static InstancesContainer& GetInstancesContainer()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(InstancesContainer, s_InstancesContainer);
        return s_InstancesContainer;
    }

#ifdef FEATSTD_THREADSAFETY_ENABLED
    static InstancesContainer& s_forceInitInstancesContainer = GetInstancesContainer();

    static FeatStd::Internal::CriticalSection& GetCriticalSection()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, cs);
        return cs;
    }

    static FeatStd::Internal::CriticalSection& s_forceInitCriticalSection = GetCriticalSection();
#endif

    Shader::UniformCacheHandle Shader::GetUniformCacheHandle(const void* const accessorId) const
    {
        for (SizeType i = 0; i < m_uniformAccessorIdCache.Size(); ++i) {
            if (accessorId == m_uniformAccessorIdCache[i].m_accessorId) {
                return UniformCacheHandle(FeatStd::Internal::NumericConversion<Int>(i));
            }
        }

        return UniformCacheHandle();
    }

    bool Shader::RegisterUniformCacheAccessor(const void* const accessorId, const ShaderParamNames::UniformSemantic uniformSemantics[],
        UInt uniformSemanticsCount, UniformCacheHandle& uniformCacheHandle) const
    {
        if (uniformSemanticsCount == 0) {
            return false;
        }

#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1938, "accesses global data [MISRA C++ Rule 12-8-1] because of thread-safe access for this type of object")
#endif

        Int uniformAccessorCacheEntryIndex = -1;
        if (VerifyCacheHandleAccessor(accessorId, uniformAccessorCacheEntryIndex)) {
            // handle has already been registered
            uniformCacheHandle = UniformCacheHandle(uniformAccessorCacheEntryIndex);
            return true;
        }

        Int* indexArray = CANDERA_NEW_ARRAY(Int, uniformSemanticsCount);
        if (0 == indexArray) {
            return false;
        }

        for (UInt i = 0; i < uniformSemanticsCount; ++i) {
            static_cast<void>(GetUniformLocation(uniformSemantics[i], indexArray[i]));
        }

        if (-1 == uniformAccessorCacheEntryIndex) {
            uniformCacheHandle = UniformCacheHandle(static_cast<Int>(m_uniformAccessorIdCache.Size()));
            return m_uniformAccessorIdCache.Add(UniformAccessorIdCacheEntry(accessorId, indexArray, uniformSemanticsCount));
        }
        else {
            uniformCacheHandle = UniformCacheHandle(uniformAccessorCacheEntryIndex);
            m_uniformAccessorIdCache[uniformAccessorCacheEntryIndex] = UniformAccessorIdCacheEntry(accessorId, indexArray, uniformSemanticsCount);
        }

        return true;
    }

    bool Shader::RegisterUniformCacheAccessor(const void* const accessorId, const Candera::Internal::UniformNode* uniformNode,
        UInt uniformNodeCount, UniformCacheHandle& uniformCacheHandle) const
    {
        if (uniformNodeCount == 0) {
            return false;
        }

#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1938, "accesses global data [MISRA C++ Rule 12-8-1] because of thread-safe access for this type of object")
#endif

        Int uniformAccessorCacheEntryIndex = -1;
        if (VerifyCacheHandleAccessor(accessorId, uniformAccessorCacheEntryIndex)) {
            // handle has already been registered
            uniformCacheHandle = UniformCacheHandle(uniformAccessorCacheEntryIndex);
            return true;
        }

        Int* indexArray = CANDERA_NEW_ARRAY(Int, uniformNodeCount);
        if (0 == indexArray) {
            return false;
        }

        UInt indexArrayIndex = 0;
        while (0 != uniformNode) {
            static_cast<void>(GetUniformLocation(uniformNode->GetName(), indexArray[indexArrayIndex++]));
            uniformNode = uniformNode->ListNode.GetNext();
        }

        if (-1 == uniformAccessorCacheEntryIndex) {
            uniformCacheHandle = UniformCacheHandle(static_cast<Int>(m_uniformAccessorIdCache.Size()));
            return m_uniformAccessorIdCache.Add(UniformAccessorIdCacheEntry(accessorId, indexArray, uniformNodeCount));
        }
        else {
            uniformCacheHandle = UniformCacheHandle(uniformAccessorCacheEntryIndex);
            m_uniformAccessorIdCache[uniformAccessorCacheEntryIndex] = UniformAccessorIdCacheEntry(accessorId, indexArray, uniformNodeCount);
        }

        return true;
    }

    void Shader::DeregisterUniformCacheAccessor(const void* const accessorId)
    {
        if (0 == accessorId) {
            return;
        }

        InstancesContainer& instancesContainer = GetInstancesContainer();
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        for (Candera::Internal::Vector<const Shader*>::ConstIterator it = instancesContainer.ConstBegin(); it != instancesContainer.ConstEnd(); ++it) {
            const Shader* shader = *it;
            FEATSTD_DEBUG_ASSERT(0 != shader);
            UniformCacheHandle uniformCacheHandle = shader->GetUniformCacheHandle(accessorId);
            static_cast<void>(shader->DeregisterUniformCacheAccessor(uniformCacheHandle));
        }
    }

    bool Shader::DeregisterUniformCacheAccessor(UniformCacheHandle& uniformCacheHandle) const
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1938, "accesses global data [MISRA C++ Rule 12-8-1] because of thread-safe access for this type of object")
#endif

        if (uniformCacheHandle.IsValid()) {
            Int uniformAccessorCacheEntryIndex = uniformCacheHandle.m_uniformCacheHandle;
            if (static_cast<SizeType>(uniformAccessorCacheEntryIndex) < m_uniformAccessorIdCache.Size()) {
                CANDERA_DELETE_ARRAY(m_uniformAccessorIdCache[uniformAccessorCacheEntryIndex].m_uniformLocations);
                m_uniformAccessorIdCache[uniformAccessorCacheEntryIndex] = UniformAccessorIdCacheEntry();
                uniformCacheHandle = UniformCacheHandle();
                return true;
            }
        }

        return false;
    }

    UInt Shader::GetUniformCacheHandleLocationsCount(UniformCacheHandle uniformCacheHandle) const
    {
        if (uniformCacheHandle.IsValid()) {
            return m_uniformAccessorIdCache[uniformCacheHandle.m_uniformCacheHandle].m_uniformLocationsCount;
        }

        return 0;
    }

    bool Shader::VerifyCacheHandleAccessor(const void* const accessorId, Int& uniformAccessorCacheEntryIndex) const
    {
        uniformAccessorCacheEntryIndex = -1;
        for (Int i = 0; i < FeatStd::Internal::NumericConversion<Int>(m_uniformAccessorIdCache.Size()); ++i) {
            if (accessorId == m_uniformAccessorIdCache[i].m_accessorId) {
                uniformAccessorCacheEntryIndex = i;
                return true;
            }
            else {
                if ((-1 == uniformAccessorCacheEntryIndex) && (0 == m_uniformAccessorIdCache[i].m_accessorId)) {
                    uniformAccessorCacheEntryIndex = i;
                }
            }
        }

        return false;
    }

    void Shader::ClearUniformAccessorIdCache()
    {
        for (SizeType i = 0; i < m_uniformAccessorIdCache.Size(); ++i) {
            CANDERA_DELETE_ARRAY(m_uniformAccessorIdCache[i].m_uniformLocations);
        }

        m_uniformAccessorIdCache.Clear();
    }

    void Shader::RegisterInstance() const
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        static_cast<void>(GetInstancesContainer().Add(this));
    }

    void Shader::DeregisterInstance() const
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        Int index = 0;
        InstancesContainer& instancesContainer = GetInstancesContainer();
        for (Candera::Internal::Vector<const Shader*>::ConstIterator it = instancesContainer.ConstBegin(); it != instancesContainer.ConstEnd(); ++it) {
             if (this == *it) {
                 static_cast<void>(instancesContainer.Remove(index));
                 break;
             }

             index++;
        }
    }

    void Shader::ClearCustomUniformCache()
    {
        for (CustomUniformCache::Iterator it = m_customUniformCache.Begin(); it != m_customUniformCache.End(); it++) {
            CANDERA_DELETE_ARRAY((*it).m_uniformName);
        }

        m_customUniformCache.Clear();
    }

    bool Shader::SetTransformVaryings(const Char** transformVaryings, SizeType tansformVaryingsCount)
    {
        if (IsUploaded()) {
            return false;
        }

        m_transformVaryings = transformVaryings;
        m_transformVaryingsCount = tansformVaryingsCount;
        return true;
    }

} // namespace Candera

