//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Scene.h"

#include <Candera/Engine3D/Canvas/Canvas.h>
#include <Candera/Engine3D/Core/SceneListener.h>

namespace Candera {
Scene::Scene() :
    Base(),
    m_lights(),
    m_lightIterator(0),
    m_renderOrder(0),
    m_sceneListeners()
#ifdef CANDERA_LAYOUT_ENABLED
    , m_isLayoutValid(false)
#endif
{
}

Scene::Scene(const Scene& src) :
    Base(src),
    m_lights(),
    m_lightIterator(0),
    m_renderOrder(0),
    m_sceneListeners()
#ifdef CANDERA_LAYOUT_ENABLED
    , m_isLayoutValid(false)
#endif
{
}

Scene::~Scene()
{
    if (m_renderOrder != 0) {
        m_renderOrder->Dispose();
    }
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::Scene::m_renderOrder, CANDERA_LINT_REASON_ASSOCIATION)
    m_lightIterator = 0;
}

Scene* Scene::Create()
{
    return FEATSTD_NEW(Scene);
}

void Scene::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

Scene* Scene::Clone() const
{
    return FEATSTD_NEW(Scene)(*this);
}

void Scene::SetRenderOrder(AbstractRenderOrder* renderOrder)
{
    m_renderOrder = renderOrder;
}

AbstractRenderOrder* Scene::GetRenderOrder() {
    return m_renderOrder;
}

const AbstractRenderOrder* Scene::GetRenderOrder() const
{
    return m_renderOrder;
}

bool Scene::AddSceneListener(SceneListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = m_sceneListeners.Append(listener);
    }
    return isSuccessful;
}

bool Scene::RemoveSceneListener(SceneListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = m_sceneListeners.Remove(listener);
    }
    return isSuccessful;
}

void Scene::NotifyListenersOnSceneActivated()
{
    for (SceneListenerContainer::Iterator it = m_sceneListeners.Begin(); it != m_sceneListeners.End(); ++it) {
        SceneListener* sceneListener = *it;
        sceneListener->OnSceneActivated(this);
    }
}

bool Scene::AddLight(Light* light)
{
    if (light != 0) {
        m_lights.Prepend(light);
        return true;
    }
    return false;
}

bool Scene::RemoveLight(Light* light)
{
    if (light != 0) {
        static_cast<void>(m_lights.Remove(light));
        return true;
    }
    return false;
}

SizeType Scene::GetNumberOfLights() const
{
    return m_lights.GetSize();
}

const Light* Scene::GetFirstLight()
{
    m_lightIterator= (m_lights.GetSize()>0) ? &*m_lights.Begin() : 0;
    return m_lightIterator;
}

const Light* Scene::GetNextLight()
{
    if (m_lightIterator != 0) {
        m_lightIterator = m_lightIterator->LightListNode.GetNext();
    }
    return (m_lightIterator != 0) ? m_lightIterator : 0;
}

void Scene::Activate()
{
    RenderDevice::SetActiveLights(&m_lights);
    // Notify SceneListeners that scene has been activated. Camera and lights have been set active at that moment.
    NotifyListenersOnSceneActivated();

#ifdef CANDERA_3D_CANVAS_ENABLED
    for (SizeType i = 0; i < m_canvases.Size(); ++i) {
        Canvas* canvas = m_canvases[i];
        FEATSTD_DEBUG_ASSERT(0 != canvas);
        if (!canvas->UsesUpdateSystem()) {
            canvas->PrepareRenderableText();
        }
    }
#endif
}


/******************************************************************************
 *  ValidateLayout
 ******************************************************************************/
#if defined(CANDERA_LAYOUT_ENABLED)
void Scene::ValidateLayout()
{
    if (!GetLayoutValid()) {
        SetLayoutValid(true);
        Layout();
    }
}
#endif


void Scene::OnCanvasAdded(Canvas* canvas)
{
    FEATSTD_DEBUG_ASSERT(!m_canvases.Contains(canvas));
    static_cast<void>(m_canvases.Add(canvas));
}

void Scene::OnCanvasRemoved(Canvas const* canvas)
{
    SizeType idx = 0;
    for (; idx < m_canvases.Size(); ++idx) {
        if (m_canvases[idx] == canvas) {
            break;
        }
    }

    FEATSTD_DEBUG_ASSERT(idx < m_canvases.Size());
    static_cast<void>(m_canvases.Remove(idx));
}

FEATSTD_RTTI_DEFINITION(Scene, Group)
} // namespace Candera
