//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RendererStatisticsOverlay.h"

#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Group.h>
#include <Candera/Engine3D/Core/OrthographicProjection.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/TextMesh.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>

namespace Candera {

static const UInt8 s_font[] =
{
    0, 16, 40, 40, 16, 96, 32, 16, 16, 16, 16, 0, 0, 0, 0, 0, 0, 16, 40, 40, 60, 100, 80, 16, 32, 8, 84,
    16, 0, 0, 0, 4, 0, 16, 40, 124, 80, 8, 80, 16, 64, 4, 56, 16, 0, 0, 0, 8, 0, 16, 0, 40, 56, 16, 32, 0,
    64, 4, 16, 124, 0, 124, 0, 16, 0, 16, 0, 124, 20, 32, 84, 0, 64, 4, 56, 16, 16, 0, 0, 32, 0, 0, 0,
    40, 120, 76, 72, 0, 32, 8, 84, 16, 16, 0, 0, 64, 0, 16, 0, 40, 16, 12, 52, 0, 16, 16, 16, 0, 32, 0,
    16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 16, 56, 124, 8, 124, 28, 124, 56, 56, 0, 0,
    8, 0, 32, 56, 68, 48, 68, 4, 24, 64, 32, 4, 68, 68, 0, 0, 16, 0, 16, 68, 76, 16, 4, 8, 40, 120, 64,
    8, 68, 68, 16, 16, 32, 124, 8, 8, 84, 16, 24, 24, 72, 4, 120, 16, 56, 60, 0, 0, 64, 0, 4, 16, 100,
    16, 32, 4, 124, 4, 68, 32, 68, 4, 16, 16, 32, 124, 8, 16, 68, 16, 64, 68, 8, 68, 68, 32, 68, 8, 0,
    16, 16, 0, 16, 0, 56, 56, 124, 56, 8, 56, 56, 32, 56, 112, 0, 32, 8, 0, 32, 16, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 16, 120, 56, 120, 124, 124, 60, 68, 56, 4, 68, 64, 68, 68, 56, 68,
    40, 68, 68, 68, 64, 64, 64, 68, 16, 4, 72, 64, 108, 68, 68, 84, 68, 68, 64, 68, 64, 64, 64, 68,
    16, 4, 80, 64, 84, 100, 68, 92, 68, 120, 64, 68, 120, 120, 64, 124, 16, 4, 96, 64, 84, 84, 68,
    88, 124, 68, 64, 68, 64, 64, 76, 68, 16, 4, 80, 64, 68, 76, 68, 64, 68, 68, 68, 68, 64, 64, 68,
    68, 16, 68, 72, 64, 68, 68, 68, 60, 68, 120, 56, 120, 124, 64, 60, 68, 56, 56, 68, 124, 68, 68,
    56, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 56, 120, 56, 124, 68, 68, 68, 68, 68, 124,
    124, 0, 124, 0, 0, 68, 68, 68, 68, 16, 68, 68, 68, 68, 68, 4, 96, 64, 12, 0, 0, 68, 68, 68, 64, 16,
    68, 68, 68, 40, 40, 8, 96, 32, 12, 16, 0, 120, 68, 120, 56, 16, 68, 68, 84, 16, 16, 16, 96, 16,
    12, 40, 0, 64, 84, 80, 4, 16, 68, 68, 84, 40, 16, 32, 96, 8, 12, 68, 0, 64, 72, 72, 68, 16, 68, 40,
    108, 68, 16, 64, 96, 4, 12, 0, 0, 64, 52, 68, 56, 16, 56, 16, 68, 68, 16, 124, 124, 0, 124, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 254, 32, 0, 64, 0, 4, 0, 24, 0, 64, 16, 8, 64, 48, 0, 0, 0,
    16, 0, 64, 0, 4, 0, 36, 0, 64, 0, 0, 64, 16, 0, 0, 0, 8, 56, 120, 60, 60, 56, 32, 56, 120, 48, 24,
    68, 16, 108, 120, 56, 0, 4, 68, 64, 68, 68, 120, 68, 68, 16, 8, 72, 16, 84, 68, 68, 0, 60, 68, 64,
    68, 124, 32, 68, 68, 16, 8, 112, 16, 84, 68, 68, 0, 68, 68, 64, 68, 64, 32, 60, 68, 16, 8, 72, 16,
    84, 68, 68, 0, 60, 120, 60, 60, 60, 32, 4, 68, 56, 72, 68, 56, 68, 68, 56, 0, 0, 0, 0, 0, 0, 0, 56,
    0, 0, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 28, 16, 112, 52, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0,
    0, 0, 48, 16, 24, 88, 84, 120, 60, 92, 60, 120, 68, 68, 68, 68, 68, 124, 48, 16, 24, 0, 40, 68,
    68, 96, 64, 32, 68, 68, 68, 40, 68, 8, 96, 16, 12, 0, 84, 68, 68, 64, 56, 32, 68, 68, 84, 16, 68,
    16, 48, 16, 24, 0, 40, 120, 60, 64, 4, 36, 76, 40, 84, 40, 60, 32, 48, 16, 24, 0, 84, 64, 4, 64,
    120, 24, 52, 16, 108, 68, 4, 124, 28, 16, 112, 0, 0, 64, 4, 0, 0, 0, 0, 0, 0, 0, 56, 0, 0, 16, 0, 0, 0
};

static const SizeType s_texWidth = 128;
static const SizeType s_texHeight = 64;
static const SizeType s_maxCharacters = 1000;
static const Float s_fontHeight = 9.0F;

static Bitmap::SharedPointer CreateFontBitmap()
{
    const SizeType texWidth = s_texWidth;
    const SizeType texHeight = s_texHeight;
    const SizeType bitmapDataSize = texWidth * texHeight * 4;
    UInt8* const bitmapData = CANDERA_NEW_ARRAY(UInt8, bitmapDataSize);
    if (0 == bitmapData) {
        return Bitmap::SharedPointer(0);
    }

    UInt8* dst = bitmapData;
    const UInt8* src = s_font;

    for (SizeType j = 0; j < 16 * 48; j++) {
        for (SizeType i = 0; i < 8; i++) {
            if ((*src & static_cast<UInt8>((0x80U >> i))) != 0) {
                *dst++ = 0xFF;
                *dst++ = 0xFF;
                *dst++ = 0xFF;
                *dst++ = 0xFF;
            }
            else {
                *dst++ = 0;
                *dst++ = 0;
                *dst++ = 0;
                *dst++ = 0;
            }
        }

        src++;
    }

    Bitmap::SharedPointer bitmap = Bitmap::Create(texWidth, texHeight, Bitmap::RgbaUnsignedBytePixelFormat, Candera::Bitmap::PackAlignment1,
        bitmapData, Candera::Bitmap::Disposer::Dispose, bitmapDataSize, true, true);
    return bitmap;
}

static Internal::TextMesh* CreateTextMesh(SizeType maxLength)
{
    const SizeType texWidth = s_texWidth;
    const SizeType texHeight = s_texHeight;
    const SizeType totalCharCount = 128;
    Internal::Vector<Vector4> glyphData;
    if (!glyphData.Reserve(totalCharCount)) {
        return 0;
    }

    Internal::Vector<Vector2> uvs;
    if (!uvs.Reserve(totalCharCount * 4)) {
        return 0;
    }

    // Assign dedicated quad to the first 32 non-printable characters.
    for (SizeType i = 0; i < 32; i++)
    {
        Vector2 pos(120.0F, 40.0F);
        Float u0 = pos.GetX() / static_cast<Float>(texWidth);
        Float v0 = pos.GetY() / static_cast<Float>(texHeight);
        Float u1 = (pos.GetX() + 8.0F) / static_cast<Float>(texWidth);
        Float v1 = (pos.GetY() + 8.0F) / static_cast<Float>(texHeight);
        static_cast<void>(uvs.Add(Vector2(u0, v0)));
        static_cast<void>(uvs.Add(Vector2(u1, v0)));
        static_cast<void>(uvs.Add(Vector2(u1, v1)));
        static_cast<void>(uvs.Add(Vector2(u0, v1)));
        static_cast<void>(glyphData.Add(Vector4(8.0F, 8.0F, 8.0F, 0.0F)));
    }

    // Rest of the characters
    Vector2 pos(0.0F, 0.0F);
    for (SizeType y = 0; y < 6; y++)
    {
        pos.SetX(0.0F);
        for (SizeType x = 0; x < 16; x++)
        {
            Float u0 = pos.GetX() / static_cast<Float>(texWidth);
            Float v0 = pos.GetY() / static_cast<Float>(texHeight);
            Float u1 = (pos.GetX() + 8.0F) / static_cast<Float>(texWidth);
            Float v1 = (pos.GetY() + 8.0F) / static_cast<Float>(texHeight);
            static_cast<void>(uvs.Add(Vector2(u0, v0)));
            static_cast<void>(uvs.Add(Vector2(u1, v0)));
            static_cast<void>(uvs.Add(Vector2(u1, v1)));
            static_cast<void>(uvs.Add(Vector2(u0, v1)));
            static_cast<void>(glyphData.Add(Vector4(8.0F, 8.0F, 8.0F, 0.0F)));
            pos.SetX(pos.GetX() + 8.0F);
        }

        pos.SetY(pos.GetY() + 8.0F);
    }

    return Internal::TextMesh::Create(glyphData, uvs, s_fontHeight, maxLength);
}

static Appearance::SharedPointer CreateAppearance(MemoryManagement::SharedPointer<Shader> shader)
{
    MemoryManagement::SharedPointer<GenericShaderParamSetter> shaderParameterSetter = GenericShaderParamSetter::Create();
    shaderParameterSetter->SetLightActivationEnabled(false);
    shaderParameterSetter->SetMaterialActivationEnabled(false);
    Appearance::SharedPointer appearance = Appearance::Create();
    appearance->SetShaderParamSetter(shaderParameterSetter);
    appearance->SetRenderMode(RenderMode::Create());
    appearance->GetRenderMode()->SetBlendingEnabled(true);
    appearance->SetShader(shader);
    return appearance;
}

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

RendererStatisticsOverlay* RendererStatisticsOverlay::Create(const SharedPointer<Shader>& shader)
{
    return CANDERA_NEW(RendererStatisticsOverlay)(shader);
}

RendererStatisticsOverlay::RendererStatisticsOverlay(const SharedPointer<Shader>& shader)
    :
    m_appearance(CreateAppearance(shader)),
    m_textMesh(0),
    m_dummyNode(Group::Create()),
    m_camera(Camera::Create()),
    m_textColor(Color(1.0F, 1.0F, 1.0F, 0.6F)),
    m_valueColor(Color(1.0F, 1.0F, 1.0F, 0.85F))
{
    FEATSTD_DEBUG_ASSERT(m_appearance != 0);
    BitmapTextureImage::SharedPointer bitmapTextureImage = BitmapTextureImage::Create();
    static_cast<void>(bitmapTextureImage->SetBitmap(CreateFontBitmap()));

    Texture::SharedPointer texture = Texture::Create();
    texture->SetTextureImage(bitmapTextureImage);
    texture->SetMinificationFilter(Texture::MinMagNearest);
    texture->SetMagnificationFilter(Texture::MinMagNearest);
    static_cast<void>(m_appearance->SetTexture(texture));
    m_appearance->GetRenderMode()->SetDepthTestEnabled(false);
    m_appearance->GetRenderMode()->SetDepthWriteEnabled(false);

    SharedPointer<OrthographicProjection> orthoProjection = OrthographicProjection::Create();
    m_camera->SetProjection(orthoProjection);
    Vector3 cameraPosition(0.0F, 0.0F, 1.0F);

    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1058, "False positive. LINT thinks this is Rectangle::SetPosition, but it is Transformable::SetPosition.")
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "False positive. LINT thinks this is Rectangle::SetPosition, but it is Transformable::SetPosition.")
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1025, "False positive. LINT thinks this is Rectangle::SetPosition, but it is Transformable::SetPosition.")
    m_camera->SetPosition(cameraPosition);

    m_textMesh = CreateTextMesh(s_maxCharacters);
}

RendererStatisticsOverlay::~RendererStatisticsOverlay()
{
    if (0 != m_camera) {
        m_camera->Dispose();
        m_camera = 0;
    }

    if (0 != m_dummyNode) {
        m_dummyNode->Dispose();
        m_dummyNode = 0;
    }

    CANDERA_DELETE(m_textMesh);
    m_textMesh = 0;
}

void RendererStatisticsOverlay::UpdateText()
{
    const Renderer::Statistics& stats = Renderer::GetStatistics();

    const bool isInstancingSupported = RenderDevice::IsInstancingSupported();
    const bool isUniformBufferSupported = Renderer::IsUniformBufferSupported();
    const bool isPixelBufferSupported = Renderer::IsPixelBufferSupported();

    m_textMesh->Clear();

    using namespace FeatStd::Internal;
    Char tempTxt[s_maxCharacters];

    // Drawcalls
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("Drawcalls: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", (stats.DrawCalls + stats.InstancedDrawCalls)));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    // Instanced, Instances
    if (isInstancingSupported) {
        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(" (Instanced: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.InstancedDrawCalls));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(", Instances: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.Instances));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(")"));
    }

    // Clears
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("\nClears: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.ColorBufferClears));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("/"));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.DepthBufferClears));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("/"));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.StencilBufferClears));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(" (Color/Depth/Stencil)"));

    // Vertices, Indices
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("\nVertices: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.Vertices));
    static_cast<void>(m_textMesh->AppendText(tempTxt));


    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(", Indices: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.Indices));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    // Uniform calls, Uniforms set
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("\nUniform calls: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.UniformCalls));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(" (Set: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.Uniforms));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(")"));

    // Dirty Area
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("\nDirty Area (avg): "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%.2f", (stats.AverageDirtyAreaFactor * 100.0F)));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("% (Culled: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.CulledNodesUsingDirtyAreaScissor));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(")"));

    // Uniform buffers
    if (isUniformBufferSupported) {
        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText("\nUniform Buffers: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.UniformBufferObjects));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(" (Updates: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.UniformBufferUpdates));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(")"));
    }

    // Pixel buffers
    if (isPixelBufferSupported) {
        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText("\nPixel Buffers: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.PixelBufferObjects));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(" (Mapped: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.PixelBufferMappings));
        static_cast<void>(m_textMesh->AppendText(tempTxt));

        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText(")"));
    }

    // Textures
    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText("\nTextures: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.UploadedTextures - 1));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(" (Pooled: "));

    m_textMesh->SetPenColor(m_valueColor);
    static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.UploadedTexturesInReusePool));
    static_cast<void>(m_textMesh->AppendText(tempTxt));

    m_textMesh->SetPenColor(m_textColor);
    static_cast<void>(m_textMesh->AppendText(")"));

    // Sampler Objects
    if (stats.SamplerObjects > 0) {
        m_textMesh->SetPenColor(m_textColor);
        static_cast<void>(m_textMesh->AppendText("\nSampler Objects: "));

        m_textMesh->SetPenColor(m_valueColor);
        static_cast<void>(String::StringPrintf(tempTxt, s_maxCharacters, "%d", stats.SamplerObjects));
        static_cast<void>(m_textMesh->AppendText(tempTxt));
    }

    m_textMesh->Update();
}

void RendererStatisticsOverlay::Render(RenderTarget3D* renderTarget)
{
    FEATSTD_DEBUG_ASSERT(0 != m_dummyNode);
    FEATSTD_DEBUG_ASSERT(0 != m_textMesh);
    FEATSTD_DEBUG_ASSERT(m_appearance != 0);
    FEATSTD_DEBUG_ASSERT(m_appearance->GetShader() != 0);
    FEATSTD_DEBUG_ASSERT(m_appearance->GetShaderParamSetter() != 0);

    if (0 != RenderDevice::GetActiveCamera()) {
        FEATSTD_LOG_ERROR("There is already an active camera. Camera activation cannot be nested.");
        return;
    }

    if (0 == renderTarget) {
        return;
    }

    m_camera->SetRenderTarget(renderTarget);
    const Float width = static_cast<Float>(renderTarget->GetWidth());
    const Float height = static_cast<Float>(renderTarget->GetHeight());
    m_dummyNode->SetPosition(-width / 2.0F, (height / 2.0F) - s_fontHeight, 0.0F);
    m_textMesh->SetLineWidth(static_cast<Float>(width));
    SharedPointer<Projection> projection = m_camera->GetProjection();
    OrthographicProjection* orthoProjection = Dynamic_Cast<OrthographicProjection*>(projection.GetPointerToSharedInstance());
    if (0 == orthoProjection) {
        return;
    }

    orthoProjection->SetNearZ(0.001F);
    orthoProjection->SetFarZ(2.0F);
    orthoProjection->SetWidth(width);
    orthoProjection->SetHeight(height);

    Texture::SharedPointer texture = m_appearance->GetTexture();
    if (texture == 0) {
        return;
    }

    SharedPointer<TextureImage> textureImage = texture->GetTextureImage();
    if ((textureImage != 0) && (!textureImage->IsUploaded())) {
        if (!textureImage->Upload(DeviceObject::NoHint)) {
            FEATSTD_LOG_ERROR("Upload of texture by renderer statistics overlay failed.");
            return;
        }
    }

    const SharedPointer<Shader>& shader = m_appearance->GetShader();
    if (!shader->IsUploaded()) {
        if (!shader->Upload(DeviceObject::NoHint)) {
            FEATSTD_LOG_ERROR("Upload of user provided shader by renderer statistics overlay failed.");
            return;
        }
    }

    UpdateText();

    if (!Renderer::ActivateCamera(m_camera)) {
        return;
    }

    if (!m_appearance->Activate()) {
        FEATSTD_LOG_ERROR("Activation of renderer statistics overlay appearance failed.");
        return;
    }

    const VertexBuffer::SharedPointer& vertexBuffer = m_textMesh->GetVertexBuffer();
    if (!vertexBuffer->Activate()) {
        FEATSTD_LOG_ERROR("Activation of renderer statistics overlay vertexBuffer failed.");
        return;
    }

    if (!m_appearance->GetShader()->BindAttributes(vertexBuffer)) {
        FEATSTD_LOG_ERROR("BindAttributes of renderer statistics overlay failed.");
        return;
    }

    if (!m_appearance->GetShaderParamSetter()->Activate(*m_dummyNode, m_appearance)) {
        FEATSTD_LOG_ERROR("Activation of renderer statistics overlay ShaderParamSetter failed");
        return;
    }

    vertexBuffer->Render();

    static_cast<void>(Renderer::DeactivateCamera(m_camera));
}

}
