//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_RendererListener_H
#define CANDERA_RendererListener_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
    class Node;
    class Camera;

    /**
    * @brief A RendererListener defines hooks that are called before or after a Node is rendered.
    */
    class RendererListener
    {
    public:
        /**
        * Constructs a RendererListener object.
        */
        RendererListener() {}

        /**
        * Destructs a RendererListener object.
        */
        virtual ~RendererListener() {}

        /**
        * Is invoked before a node is rendered.
        * @param node Node that renderer listeners will be notified that it will be rendered next.
        */
        virtual void OnNodePreRender(Node* node) =0; 

        /**
        * Is invoked after a node is rendered.
        * @param node Node that renderer listeners will be notified that it was be rendered last.
        */
        virtual void OnNodePostRender(Node* node) =0;

        /**
        *  Is invoked after a camera has been rendered.
        *  @param camera The camera that was rendered.
        */
        virtual void OnCameraPostRender(Camera* /*camera*/) {};

        /**
        *  Is invoked before a camera is activated.
        *  @param camera The camera that will be activated.
        */
        virtual void OnCameraPreActivate(const Camera* /*camera*/) {};

        /**
        *  Is invoked after a camera is activated.
        *  @param camera The camera that was activated.
        */
        virtual void OnCameraPostActivate(const Camera* /*camera*/) {};

        /**
        *  Is invoked before clearing the framebuffer.
        *  @param camera The camera that will be activated.
        */
        virtual void OnCameraPreClear(const Camera* /*camera*/) {};

        /**
        *  Is invoked after a clearing the framebuffer.
        *  @param camera The camera that was activated.
        */
        virtual void OnCameraPostClear(const Camera* /*camera*/) {};


    protected:
        RendererListener(const RendererListener&);
        RendererListener& operator=(const RendererListener&);
    };
 
/** @} */ // end of Core3D
 
} // namespace Candera

#endif  // CANDERA_RendererListener_H

