//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RENDERABLE_H)
#define CANDERA_RENDERABLE_H

#include <Candera/Engine3D/Core/Node.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   The class Renderable can serve as a base for nodes that can be recognized and rendered by the Renderer.
 *          In addition to the Node class providing an Appearance including Shader and RenderMode, this involves a
 *          VertexBuffer and ShaderParameterSetter.
 *          If a class wants to potentially benefit from automatic draw-call batching (which also depends on
 *          hardware support for instancing), it needs to be derived from this base class.
 */
class Renderable : public Node
{
    FEATSTD_TYPEDEF_BASE(Node);

    public:
        /**
         *  Destructor
         */
        virtual ~Renderable() override;

        /**
         * Overrides Node::IsRenderPrerequisiteFulfilled.
         * @return true if precondition to be rendered is fulfilled, false otherwise.
         * The render precondition is not fulfilled if at least one of following statements is true:
         * - VertexBuffer is either null or no VertexGeometry is associated with VertexBuffer.
         * - Node::IsRenderPrerequisiteFulfilled returns false
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override;

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit, protected Constructor and Copy-Constructor.
        Renderable();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Renderable);

        // to bypass expensive SharedPointer getters
        friend class BatchOrderCriterion;
        friend class Renderer;

        MemoryManagement::SharedPointer<VertexBuffer> m_vertexBuffer;

        bool m_isBatchable; ///< this flag indicates to the Renderer if this Renderable should be considered for batching

};

/** @} */ // end of Core3D

} // namespace Candera

#endif  // CANDERA_RENDERABLE_H
