//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderMode.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    FEATSTD_RTTI_DEFINITION(RenderMode, Base)

RenderMode::RenderMode() :
    Base(),
    m_isColorWriteRedEnabled(true),
    m_isColorWriteGreenEnabled(true),
    m_isColorWriteBlueEnabled(true),
    m_isColorWriteAlphaEnabled(true),
    m_isDepthWriteEnabled(true),
    m_isDepthTestEnabled(true),
    m_isStencilTestEnabled(false),
    m_isBlendingEnabled(false),
    m_isRasterizerDiscardEnabled(false),
    m_stencilWriteMaskBackPlane(~(static_cast<UInt32>(0U))),
    m_stencilWriteMaskFrontPlane(~(static_cast<UInt32>(0U))),
    m_inheritanceMask(0),
    m_cullMode(BackFaceCulling),
    m_windMode(CounterClockWise),
    m_blendMode(),                  // Default values for RGB: SourceAlpha, InverseSourceAlpha, Add; Default values for Alpha: One, Zero, Add.
    m_blendColor(),                 // Default value is black.
    m_depthCompFunc(CompareLess),
    m_stencilFunctionBackPlane(),
    m_stencilFunctionFrontPlane(),
    m_stencilOperationBackPlane(),
    m_stencilOperationFrontPlane(),
    m_depthBias()                  // Default values are scaleFactor: 0.0F, units: 0.0F.
{
}

RenderMode::RenderMode(const RenderMode& renderMode):
    Base(renderMode),
    m_isColorWriteRedEnabled(renderMode.m_isColorWriteRedEnabled),
    m_isColorWriteGreenEnabled(renderMode.m_isColorWriteGreenEnabled),
    m_isColorWriteBlueEnabled(renderMode.m_isColorWriteBlueEnabled),
    m_isColorWriteAlphaEnabled(renderMode.m_isColorWriteAlphaEnabled),
    m_isDepthWriteEnabled(renderMode.m_isDepthWriteEnabled),
    m_isDepthTestEnabled(renderMode.m_isDepthTestEnabled),
    m_isStencilTestEnabled(renderMode.m_isStencilTestEnabled),
    m_isBlendingEnabled(renderMode.m_isBlendingEnabled),
    m_isRasterizerDiscardEnabled(renderMode.m_isRasterizerDiscardEnabled),
    m_stencilWriteMaskBackPlane(renderMode.m_stencilWriteMaskBackPlane),
    m_stencilWriteMaskFrontPlane(renderMode.m_stencilWriteMaskFrontPlane),
    m_inheritanceMask(renderMode.m_inheritanceMask),
    m_cullMode(renderMode.m_cullMode),
    m_windMode(renderMode.m_windMode),
    m_blendMode(renderMode.m_blendMode),
    m_blendColor(renderMode.m_blendColor),
    m_depthCompFunc(renderMode.m_depthCompFunc),
    m_stencilFunctionBackPlane(renderMode.m_stencilFunctionBackPlane),
    m_stencilFunctionFrontPlane(renderMode.m_stencilFunctionFrontPlane),
    m_stencilOperationBackPlane(renderMode.m_stencilOperationBackPlane),
    m_stencilOperationFrontPlane(renderMode.m_stencilOperationFrontPlane),
    m_depthBias(renderMode.m_depthBias)
{
}

void RenderMode::SetInheritanceMask(UInt32 inheritanceMask)
{
    m_inheritanceMask = inheritanceMask;
}

void RenderMode::SetInheritanceBitsToOne(UInt32 inheritanceBitSelection)
{
    m_inheritanceMask |= inheritanceBitSelection;
}

void RenderMode::SetInheritanceBitsToZero(UInt32 inheritanceBitSelection)
{
    m_inheritanceMask &= (~inheritanceBitSelection);
}

UInt32 RenderMode::GetInheritanceMask() const
{
    return m_inheritanceMask;
}

bool RenderMode::IsInherited(InheritanceBit bit) const
{
    return (m_inheritanceMask & static_cast<UInt32>(bit)) != 0;
}

RenderMode& RenderMode::operator=(const RenderMode& renderMode)
{
    if (this != &renderMode) {
        Base::operator =(renderMode);
        m_cullMode = renderMode.m_cullMode;
        m_windMode = renderMode.m_windMode;
        m_isColorWriteRedEnabled = renderMode.m_isColorWriteRedEnabled;
        m_isColorWriteGreenEnabled = renderMode.m_isColorWriteGreenEnabled;
        m_isColorWriteBlueEnabled = renderMode.m_isColorWriteBlueEnabled;
        m_isColorWriteAlphaEnabled = renderMode.m_isColorWriteAlphaEnabled;
        m_isDepthWriteEnabled = renderMode.m_isDepthWriteEnabled;
        m_isDepthTestEnabled = renderMode.m_isDepthTestEnabled;
        m_isStencilTestEnabled = renderMode.m_isStencilTestEnabled;
        m_depthCompFunc = renderMode.m_depthCompFunc;
        m_isBlendingEnabled = renderMode.m_isBlendingEnabled;
        m_isRasterizerDiscardEnabled = renderMode.m_isRasterizerDiscardEnabled;
        m_blendMode = renderMode.m_blendMode;
        m_blendColor = renderMode.m_blendColor;
        m_stencilFunctionBackPlane = renderMode.m_stencilFunctionBackPlane;
        m_stencilFunctionFrontPlane = renderMode.m_stencilFunctionFrontPlane;
        m_stencilOperationBackPlane = renderMode.m_stencilOperationBackPlane;
        m_stencilOperationFrontPlane = renderMode.m_stencilOperationFrontPlane;
        m_stencilWriteMaskBackPlane = renderMode.m_stencilWriteMaskBackPlane;
        m_stencilWriteMaskFrontPlane = renderMode.m_stencilWriteMaskFrontPlane;
        m_depthBias = renderMode.m_depthBias;
        m_inheritanceMask = renderMode.m_inheritanceMask;
        // m_retainCount must not be copied!
    }
    return *this;
}

RenderMode::SharedPointer RenderMode::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    RenderMode* ptr = FEATSTD_NEW(RenderMode)();
    if (ptr == 0) {
            FEATSTD_LOG_ERROR("Render mode create failed, out of memory.");
    }
    RenderMode::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

RenderMode::~RenderMode()
{
    //nothing to do
}

RenderMode::SharedPointer RenderMode::Clone() const
{
    return RenderMode::SharedPointer(FEATSTD_NEW(RenderMode)(*this));
}

void RenderMode::SetColorWriteEnabled(bool enableRed, bool enableGreen, bool enableBlue, bool enableAlpha)
{
    SetColorWriteRedEnabled(enableRed);
    SetColorWriteGreenEnabled(enableGreen);
    SetColorWriteBlueEnabled(enableBlue);
    SetColorWriteAlphaEnabled(enableAlpha);
}

void RenderMode::SetBlendMode(BlendFactor sourceFactor, BlendFactor destFactor, BlendOperation operation)
{
    SetBlendModeSeparate(sourceFactor, destFactor, operation, sourceFactor, destFactor, operation);
}

void RenderMode::SetBlendModeSeparate(BlendFactor sourceRGBFactor, BlendFactor destRGBFactor, BlendOperation operationRGB,
                                      BlendFactor sourceAlphaFactor, BlendFactor destAlphaFactor, BlendOperation operationAlpha)
{
    m_blendMode.sourceRGBFactor = sourceRGBFactor;
    m_blendMode.destRGBFactor = destRGBFactor;
    m_blendMode.operationRGB = operationRGB;
    m_blendMode.sourceAlphaFactor = sourceAlphaFactor;
    m_blendMode.destAlphaFactor = destAlphaFactor;
    m_blendMode.operationAlpha = operationAlpha;
}

void RenderMode::SetStencilOperation(const StencilOperationData& operation, const StencilPlane face)
{
    if (AffectsFrontFace(face)) {
        m_stencilOperationFrontPlane = operation;
    }
    if (AffectsBackFace(face)) {
        m_stencilOperationBackPlane = operation;
    }
}

const RenderMode::StencilOperationData& RenderMode::GetStencilOperation(const StencilPlane face) const
{
    return (AffectsFrontFace(face)) ? m_stencilOperationFrontPlane : m_stencilOperationBackPlane;
}

void RenderMode::SetStencilFunction(const StencilFunctionData& function, const StencilPlane face)
{
    if (AffectsFrontFace(face)) {
        m_stencilFunctionFrontPlane = function;
    }
    if (AffectsBackFace(face)) {
        m_stencilFunctionBackPlane = function;
    }
}

const RenderMode::StencilFunctionData& RenderMode::GetStencilFunction(const StencilPlane face) const
{
    return (AffectsFrontFace(face)) ? m_stencilFunctionFrontPlane : m_stencilFunctionBackPlane;
}

void RenderMode::SetStencilWriteMask(const UInt32 writeMask, const StencilPlane face)
{
    if (AffectsFrontFace(face)) {
        m_stencilWriteMaskFrontPlane = writeMask;
    }
    if (AffectsBackFace(face)) {
        m_stencilWriteMaskBackPlane = writeMask;
    }
}

UInt32 RenderMode::GetStencilWriteMask(const StencilPlane face) const
{
    return (AffectsFrontFace(face)) ? m_stencilWriteMaskFrontPlane : m_stencilWriteMaskBackPlane;
}

void RenderMode::SetDepthBias(Float scaleFactor, Float units)
{
    m_depthBias.scaleFactor = scaleFactor;
    m_depthBias.units = units;
}

bool RenderMode::Activate(const RenderMode::SharedPointer& baseMode) const
{
    if (baseMode.GetPointerToSharedInstance() == 0) {
        FEATSTD_LOG_ERROR("Render mode activate failed, baseMode == 0.");
        return false;
    }

    bool success = true;

    const Culling cullMode = IsInherited(CullingModeBit) ? baseMode->m_cullMode : m_cullMode;
    if (!RenderDevice::ActivateRenderState(RenderDevice::CullFaceMode, static_cast<UInt32>(cullMode))) {
        success = false;
    }

    const Winding windingMode = IsInherited(WindingModeBit) ? baseMode->m_windMode : m_windMode;
    if (!RenderDevice::ActivateRenderState(RenderDevice::WindingOrder, static_cast<UInt32>(windingMode))) {
        success = false;
    }

    if (IsInherited(ColorWriteEnabledBit)) {
        if (!RenderDevice::ActivateColorWriteEnabled(
            baseMode->m_isColorWriteRedEnabled,
            baseMode->m_isColorWriteGreenEnabled,
            baseMode->m_isColorWriteBlueEnabled,
            baseMode->m_isColorWriteAlphaEnabled)) {
            success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateColorWriteEnabled(
            this->m_isColorWriteRedEnabled,
            this->m_isColorWriteGreenEnabled,
            this->m_isColorWriteBlueEnabled,
            this->m_isColorWriteAlphaEnabled)) {
            success = false;
        }
    }

    const bool isDepthWriteEnabled = IsInherited(DepthWriteEnabledBit) ? baseMode->m_isDepthWriteEnabled : m_isDepthWriteEnabled;
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthWriteEnable, static_cast<UInt32>(isDepthWriteEnabled))) {
        success = false;
    }

    const bool isDepthTestEnabled = IsInherited(DepthTestEnabledBit) ? baseMode->m_isDepthTestEnabled : m_isDepthTestEnabled;
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthTestEnable, static_cast<UInt32>(isDepthTestEnabled))) {
        success = false;
    }

    const ComparisonFunction depthCompFunc = IsInherited(DepthComparisonFunctionBit) ? baseMode->m_depthCompFunc : m_depthCompFunc;
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthComparisonFunction, static_cast<UInt32>(depthCompFunc))) {
        success = false;
    }

    const bool isBlendingEnabled = IsInherited(BlendingEnabledBit) ? baseMode->m_isBlendingEnabled : m_isBlendingEnabled;
    if (!RenderDevice::ActivateRenderState(RenderDevice::BlendingEnable, static_cast<UInt32>(isBlendingEnabled))) {
        success = false;
    }

    const bool isStencilTestEnabled = IsInherited(StencilTestEnabledBit) ? baseMode->m_isStencilTestEnabled : m_isStencilTestEnabled;
    if (!RenderDevice::ActivateRenderState(RenderDevice::StencilTestEnable, static_cast<UInt32>(isStencilTestEnabled))) {
        success = false;
    }

    const BlendMode* const blendMode = IsInherited(BlendModeBit) ? &baseMode->m_blendMode : &m_blendMode;
    if (!RenderDevice::ActivateBlendModeSeparate(blendMode->sourceRGBFactor, blendMode->destRGBFactor, blendMode->operationRGB,
                                                 blendMode->sourceAlphaFactor, blendMode->destAlphaFactor, blendMode->operationAlpha)) {
        success = false;
    }

    const Color* const blendColor = IsInherited(BlendColorBit) ? &baseMode->m_blendColor : &m_blendColor;
    if (!RenderDevice::ActivateBlendColor(*blendColor)) {
        success = false;
    }

    const StencilFunctionData& stencilFunctionBackPlane = IsInherited(StencilFunctionBackPlaneBit) ? baseMode->m_stencilFunctionBackPlane : m_stencilFunctionBackPlane;
    const StencilFunctionData& stencilFunctionFrontPlane = IsInherited(StencilFunctionFrontPlaneBit) ? baseMode->m_stencilFunctionFrontPlane : m_stencilFunctionFrontPlane;

    if ((stencilFunctionBackPlane.compFunc == stencilFunctionFrontPlane.compFunc) && 
        (stencilFunctionBackPlane.mask == stencilFunctionFrontPlane.mask) && 
        (stencilFunctionBackPlane.refValue == stencilFunctionFrontPlane.refValue)) {
        if (!RenderDevice::ActivateStencilFunction(stencilFunctionBackPlane.compFunc, stencilFunctionBackPlane.refValue, stencilFunctionBackPlane.mask, FrontAndBackFace)) {
            success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateStencilFunction(stencilFunctionBackPlane.compFunc, stencilFunctionBackPlane.refValue, stencilFunctionBackPlane.mask, BackFace)) {
            success = false;
        }

        if (!RenderDevice::ActivateStencilFunction(stencilFunctionFrontPlane.compFunc, stencilFunctionFrontPlane.refValue, stencilFunctionFrontPlane.mask, FrontFace)) {
            success = false;
        }
    }

    const StencilOperationData& stencilOperationBackPlane = IsInherited(StencilOperationBackPlaneBit) ? baseMode->m_stencilOperationBackPlane : m_stencilOperationBackPlane;
    const StencilOperationData& stencilOperationFrontPlane = IsInherited(StencilOperationFrontPlaneBit) ? baseMode->m_stencilOperationFrontPlane : m_stencilOperationFrontPlane;
    
    if ((stencilOperationBackPlane.depthFail == stencilOperationFrontPlane.depthFail) &&
        (stencilOperationBackPlane.depthPass == stencilOperationFrontPlane.depthPass) &&
        (stencilOperationBackPlane.stencilFail == stencilOperationFrontPlane.stencilFail)) {
        if (!RenderDevice::ActivateStencilOperation(stencilOperationFrontPlane.stencilFail, stencilOperationFrontPlane.depthFail, stencilOperationFrontPlane.depthPass, FrontAndBackFace)) {
            success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateStencilOperation(stencilOperationBackPlane.stencilFail, stencilOperationBackPlane.depthFail, stencilOperationBackPlane.depthPass, BackFace)) {
            success = false;
        }

        if (!RenderDevice::ActivateStencilOperation(stencilOperationFrontPlane.stencilFail, stencilOperationFrontPlane.depthFail, stencilOperationFrontPlane.depthPass, FrontFace)) {
            success = false;
        }
    }

    const UInt32 stencilWriteMaskBackPlane = IsInherited(StencilWriteMaskBackPlaneBit) ? baseMode->m_stencilWriteMaskBackPlane : m_stencilWriteMaskBackPlane;
    const UInt32 stencilWriteMaskFrontPlane = IsInherited(StencilWriteMaskFrontPlaneBit) ? baseMode->m_stencilWriteMaskFrontPlane : m_stencilWriteMaskFrontPlane;

    if (stencilWriteMaskBackPlane == stencilWriteMaskFrontPlane) {
        if (!RenderDevice::ActivateStencilWriteMask(stencilWriteMaskBackPlane, FrontAndBackFace)) {
            success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateStencilWriteMask(stencilWriteMaskBackPlane, BackFace)) {
            success = false;
        }

        if (!RenderDevice::ActivateStencilWriteMask(stencilWriteMaskFrontPlane, FrontFace)) {
            success = false;
        }
    }

    const DepthBias depthBias = IsInherited(DepthBiasBit) ? baseMode->GetDepthBias() : m_depthBias;
    if (!RenderDevice::ActivateDepthBias(depthBias.scaleFactor, depthBias.units))  {
        success = false;
    }

    const bool isRasterizerDiscardEnabled = IsInherited(RasterizerDiscardBit) ? baseMode->m_isRasterizerDiscardEnabled : m_isRasterizerDiscardEnabled;
    if (!RenderDevice::ActivateRenderState(RenderDevice::RasterizerDiscardEnable, static_cast<UInt32>(isRasterizerDiscardEnabled))) {
        success = false;
    }

    return success;
}

bool RenderMode::Activate() const
{
    bool success = true;

    if (!RenderDevice::ActivateRenderState(RenderDevice::CullFaceMode, static_cast<UInt32>(m_cullMode))) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::WindingOrder, static_cast<UInt32>(m_windMode))) {
        success = false;
    }
    if (!RenderDevice::ActivateColorWriteEnabled(m_isColorWriteRedEnabled, m_isColorWriteGreenEnabled, m_isColorWriteBlueEnabled, m_isColorWriteAlphaEnabled)) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthWriteEnable, m_isDepthWriteEnabled)) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthTestEnable, m_isDepthTestEnabled)) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::DepthComparisonFunction, static_cast<UInt32>(m_depthCompFunc))) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::BlendingEnable, m_isBlendingEnabled)) {
        success = false;
    }
    if (!RenderDevice::ActivateRenderState(RenderDevice::StencilTestEnable, m_isStencilTestEnabled)) {
        success = false;
    }
    if (!RenderDevice::ActivateBlendModeSeparate(m_blendMode.sourceRGBFactor, m_blendMode.destRGBFactor, m_blendMode.operationRGB,
                                                 m_blendMode.sourceAlphaFactor, m_blendMode.destAlphaFactor, m_blendMode.operationAlpha)) {
        success = false;
    }
    if (!RenderDevice::ActivateBlendColor(m_blendColor)) {
        success = false;
    }

    if ((m_stencilFunctionFrontPlane.compFunc == m_stencilFunctionBackPlane.compFunc) && 
      (m_stencilFunctionFrontPlane.mask == m_stencilFunctionBackPlane.mask) && 
      (m_stencilFunctionFrontPlane.refValue == m_stencilFunctionBackPlane.refValue)) {
        if (!RenderDevice::ActivateStencilFunction(m_stencilFunctionBackPlane.compFunc, m_stencilFunctionBackPlane.refValue, m_stencilFunctionBackPlane.mask, FrontAndBackFace)) {
          success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateStencilFunction(m_stencilFunctionBackPlane.compFunc, m_stencilFunctionBackPlane.refValue, m_stencilFunctionBackPlane.mask, BackFace)) {
            success = false;
        }
        if (!RenderDevice::ActivateStencilFunction(m_stencilFunctionFrontPlane.compFunc, m_stencilFunctionFrontPlane.refValue, m_stencilFunctionFrontPlane.mask, FrontFace)) {
            success = false;
        }
    }

    if ((m_stencilOperationFrontPlane.depthFail == m_stencilOperationBackPlane.depthFail) &&
      (m_stencilOperationFrontPlane.depthPass == m_stencilOperationBackPlane.depthPass) &&
      (m_stencilOperationFrontPlane.stencilFail == m_stencilOperationBackPlane.stencilFail)) {
        if (!RenderDevice::ActivateStencilOperation(m_stencilOperationBackPlane.stencilFail, m_stencilOperationBackPlane.depthFail, m_stencilOperationBackPlane.depthPass, FrontAndBackFace)) {
          success = false;
        }
    }
    else {
        if (!RenderDevice::ActivateStencilOperation(m_stencilOperationBackPlane.stencilFail, m_stencilOperationBackPlane.depthFail, m_stencilOperationBackPlane.depthPass, BackFace)) {
            success = false;
        }
        if (!RenderDevice::ActivateStencilOperation(m_stencilOperationFrontPlane.stencilFail, m_stencilOperationFrontPlane.depthFail, m_stencilOperationFrontPlane.depthPass, FrontFace)) {
            success = false;
        }
    }

    if (m_stencilWriteMaskBackPlane == m_stencilWriteMaskFrontPlane) {
        if (!RenderDevice::ActivateStencilWriteMask(m_stencilWriteMaskBackPlane, FrontAndBackFace)) {
          success = false;
      }
    }
    else {
        if (!RenderDevice::ActivateStencilWriteMask(m_stencilWriteMaskBackPlane, BackFace)) {
            success = false;
        }
        if (!RenderDevice::ActivateStencilWriteMask(m_stencilWriteMaskFrontPlane, FrontFace)) {
            success = false;
        }
    }
    if (!RenderDevice::ActivateDepthBias(m_depthBias.scaleFactor, m_depthBias.units)) {
        success = false;
    }

    if (!RenderDevice::ActivateRenderState(RenderDevice::RasterizerDiscardEnable, static_cast<UInt32>(m_isRasterizerDiscardEnabled))) {
        success = false;
    }

    return success;
}

} // namespace Candera
