//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_RenderBuffer_h)
#define Candera_RenderBuffer_h

#include <Candera/Engine3D/Core/DeviceObject.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
    /** @addtogroup Core3D
     *  @{
     */

    /**
     * @brief   class RenderBuffer represents a two-dimensional image buffer allocated by the application. The
     *          RenderBuffer can be attached to FrameBufferRenderTarget as color-, depth-, or stencil-buffer attachment.
     *          Note: Use a RenderBuffer if the image format is not supported by texture (e.g. stencil index values).
     *          RenderBuffer objects must not be used as textures directly. Thus, RenderBuffer objects may have performance
     *          advantages as the render device implementation can store render buffers in a more efficient format.
     */
    class RenderBuffer : public DeviceObject {
            FEATSTD_TYPEDEF_BASE(DeviceObject);

        public:
            enum {
                InvalidHandle = 0
            };

            /**
             *  Format of RenderBuffer that can be used as color, depth, or stencil buffer.
             */
            enum Format {
                InvalidFormat = 0,              ///< Invalid format/No format specified
                Rgb565Format = 1,               ///< R 5 bits; G 6 bits; B 5 bits.
                Rgba4Format = 2,                ///< RGBA 4 bits each.
                Rgb5A1Format = 3,               ///< RGB 5 bits each; A 1 bit.
                DepthComponent16Format = 4,     ///< 16 depth bits
                StencilIndex8Format = 5,        ///< 8 stencil bits
                NvDepth16NonlinearFormat = 6,   ///< Non linear depth buffer, 16 bits
                OesDepth24Format = 7,           ///< 24 depth bits
                OesDepth32Format = 8            ///< 32 depth bits
            };

            /**
             *  Creates an instance of this RenderBuffer class.
             *  Use Dispose() to delete the instance and possible children, if any.
             *  @return A shared pointer to the created RenderBuffer object.
             */
            static MemoryManagement::SharedPointer<RenderBuffer> Create();

            /**
             *  Destructs this RenderBuffer object.
             */
            virtual ~RenderBuffer() override;

            /**
             *  Activates this RenderBuffer in render device.
             *  @return True if the render buffer could be activated, false otherwise.
             */
            bool Activate() const;

            /**
             *  Sets Width of this RenderBuffer in pixels. Must be set before uploaded to VRAM (see RenderBuffer::Upload).
             *  @param width Width of this RenderBuffer in pixels.
             *  @return true if successfully set, false otherwise, this is if RenderBuffer object is already uploaded.
             */
            bool SetWidth(Int32 width);

            /**
             *  Retrieve the Width of this RenderBuffer in pixels.
             *  @return The Width of this RenderBuffer in pixels.
             */
            Int32 GetWidth() const {
                return m_width;
            }

            /**
             *  Sets Height of this RenderBuffer in pixels. Must be set before uploaded to VRAM (see RenderBuffer::Upload).
             *  @param height Height of this RenderBuffer in pixels.
             *  @return true if successfully set, false otherwise, this is if RenderBuffer object is already uploaded.
             */
            bool SetHeight(Int32 height);

            /**
             *  Retrieve the Height of this RenderBuffer in pixels.
             *  @return The Height of this RenderBuffer in pixels.
             */
            Int32 GetHeight() const {
                return m_height;
            }

            /**
             *  Sets Format of this RenderBuffer for color, depth, or stencil buffer.
             *  Must be set before uploaded to VRAM (see RenderBuffer::Upload).
             *  @param format Color,depth or stencil buffer format of this RenderBuffer to be set.
             *  @return true if successfully set, false otherwise, this is if RenderBuffer object is already uploaded.
             */
            bool SetFormat(Format format);

            /**
             *  Retrieve the Format of this RenderBuffer for color, depth, or stencil buffer.
             *  @return The Fromat of this RenderBuffer.
             */
            Format GetFormat() const {
                return m_format;
            }

            /**
             *  Retrieves RenderBuffer's handle to video memory.
             *  @return Handle of render buffer to video memory.
             */
            Handle GetVideoMemoryHandle() const {
                return m_videoMemoryHandle;
            }

        protected:
            /**
             *  overridden from DeviceObject
             *  @return True if upload succeeded, False otherwise.
             */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
             *  overridden from DeviceObject
             *  @return True if unload succeeded, False otherwise.
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override {}

        private:
            friend class RenderDevice;

            Format m_format;    // Format of RenderBuffer for color, depth, or stencil buffer.
            Int32 m_width;      // Width of RenderBuffer in pixels.
            Int32 m_height;     // Height of RenderBuffer in pixels.
            Handle m_videoMemoryHandle;   // Handle of RenderBuffer to video memory.

            /**
             *  Sets RenderBuffer's handle to video memory.
             *  @param handle handle of render buffer to video memory.
             */
            void SetVideoMemoryHandle(Handle handle) {
                m_videoMemoryHandle = handle;
            }

            // Constructor. Call Create() to retrieve an instance of this class.
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::RenderBuffer::RenderBuffer, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            RenderBuffer();

            // Make this class manageable by MemoryManagement::SharedPointer
            CANDERA_SHARED_POINTER_DECLARATION();
    };

    /** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_Renderbuffer_H

