//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "QueryProperty.h"

#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/OcclusionCullingCameraRenderStrategy.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/VertexGeometryBuilder.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#if defined(FEATSTD_THREADSAFETY_ENABLED)
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera {
namespace Internal {

static const Float c_cubeData[][3] = {
    { -1.0F, 1.0F, 1.0F },  // 0
    { 1.0F, 1.0F, 1.0F },   // 1
    { -1.0F, -1.0F, 1.0F }, // 2
    { 1.0F, -1.0F, 1.0F },  // 3
    { -1.0F, 1.0F, -1.0F }, // 4
    { 1.0F, 1.0F, -1.0F },  // 5
    { -1.0F, -1.0F, -1.0F },// 6
    { 1.0F, -1.0F, -1.0F }  // 7
};

static const UInt16 c_cubeIndexData[] = { 3, 2, 1, 0, 4, 2, 6, 3, 7, 1, 5, 4, 7, 6 };

static const Char c_VertexShader[] =
{
    "uniform mat4 u_MVPMatrix;\n"
    "attribute vec4 a_Position;\n"
    "void main(void)\n"
    "{\n"
    "    gl_Position = u_MVPMatrix * a_Position;\n"
    "}\n"
};

static const Char c_FragmentShader[] =
{
    "void main(void)\n"
    "{\n"
    "    gl_FragColor = vec4(1.0, 1.0, 1.0, 1.0);\n"
    "}\n"
};

#ifdef FEATSTD_THREADSAFETY_ENABLED
static FeatStd::Internal::CriticalSection& GetCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, cs);
    return cs;
}

static FeatStd::Internal::CriticalSection& s_forceInitCriticalSection = GetCriticalSection();
#endif


QueryProperty::SingleInstance QueryProperty::s_Instance;

QueryProperty::QueryProperty() :
    m_isDefault(false)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1938, "accesses global data [MISRA C++ Rule 12-8-1] because of reference counting for this type of object")
    QueryProperty::s_Instance.m_referenceCounter++;

    // Using LinearIncreasePolicy<1>, so we don't want to increase by c_MemoryInitialCapacity on first Add.
    static_cast<void>(m_queryInfos.Reserve(1));
}

QueryProperty::QueryProperty(const QueryProperty& queryProperty) :
    Base(queryProperty),
    m_isDefault(queryProperty.m_isDefault)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1938, "accesses global data [MISRA C++ Rule 12-8-1] because of reference counting for this type of object")
    QueryProperty::s_Instance.m_referenceCounter++;

    m_queryInfos = queryProperty.m_queryInfos;
}

QueryProperty::~QueryProperty()
{
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        QueryProperty::s_Instance.m_referenceCounter--;

        if (m_isDefault) {
            return;
        }

        // every instance is destructed except for the default property one
        if (1 == QueryProperty::s_Instance.m_referenceCounter) {
            if (0 != QueryProperty::s_Instance.m_boundingBox) {
                static_cast<void>(QueryProperty::s_Instance.m_boundingBox->Unload());
                QueryProperty::s_Instance.m_boundingBox->DisposeSelf();
                QueryProperty::s_Instance.m_boundingBox = 0;
            }

            FEATSTD_DEBUG_ASSERT(0 != QueryProperty::s_Instance.m_defaultQueryProperty);
            FEATSTD_DELETE(QueryProperty::s_Instance.m_defaultQueryProperty);
            QueryProperty::s_Instance.m_defaultQueryProperty = 0;
        }
    }

    CleanupQueryInfos();
}

void QueryProperty::CleanupQueryInfos()
{
    m_queryInfos.Clear();
}

QueryProperty& QueryProperty::operator= (const QueryProperty& other)
{
    m_queryInfos = other.m_queryInfos;
    m_isDefault = other.m_isDefault;

    return *this;
}

bool QueryProperty::Set(Node& node, const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy, Query::Type type)
{
    if (0 == occlusionCullingCameraRenderStrategy) {
        return false;
    }

    bool foundStrategy = false;
    QueryProperty* setQueryProperty = Get(node);

    if (0 == setQueryProperty) {
        QueryProperty queryProperty;
        bool wasSuccessful = node.SetValue(CdaDynamicPropertyInstance(QueryProperty), queryProperty);
        if (wasSuccessful) {
            // a copy of queryProperty was automatically created and set by DynamicPropertyHost,
            // so that copy has to be retrieved for actual use.
            setQueryProperty = Get(node);
            if (setQueryProperty == 0) {
                return false;
            }
            FEATSTD_DEBUG_ASSERT(setQueryProperty->m_queryInfos.Size() == 0);
        }
        
        FEATSTD_DEBUG_ASSERT(node.RemoveNodeListener(setQueryProperty) == false);
        if (!node.AddNodeListener(setQueryProperty)) {
            return false;
        }
    }
    else {
        for (UInt i = 0; i < static_cast<UInt>(setQueryProperty->m_queryInfos.Size()); ++i) {
            if (occlusionCullingCameraRenderStrategy == setQueryProperty->m_queryInfos[i].m_occlusionCullingCameraRenderStrategy) {
                foundStrategy = true;
                break;
            }
        }
    }

    if (!foundStrategy) {
        QueryInfo queryInfo;
        queryInfo.m_occlusionCullingCameraRenderStrategy = occlusionCullingCameraRenderStrategy;
        queryInfo.m_query = Query::Create(type);
        static_cast<void>(queryInfo.m_query->Upload());
        static_cast<void>(setQueryProperty->m_queryInfos.Add(queryInfo));
    }

    return true;
}

bool QueryProperty::Remove(Node& node, const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy)
{
    QueryProperty* queryProperty = Get(node);
    if (0 == queryProperty) {
        return false;
    }

    return queryProperty->Remove(occlusionCullingCameraRenderStrategy);
}

bool QueryProperty::Remove(const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy)
{
    for (UInt i = 0; i < static_cast<UInt>(m_queryInfos.Size()); ++i) {
        if (occlusionCullingCameraRenderStrategy == m_queryInfos[i].m_occlusionCullingCameraRenderStrategy) {
            static_cast<void>(m_queryInfos.Remove(i));
            return true;
        }
    }

    return false;
}

QueryProperty* QueryProperty::Get(const Node& node)
{
    QueryProperty& queryProperty = const_cast<QueryProperty&>(node.GetValue(CdaDynamicPropertyInstance(QueryProperty)));
    return queryProperty.m_isDefault ? 0 : &queryProperty;
}

const QueryProperty& QueryProperty::GetDefault()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    if (0 == QueryProperty::s_Instance.m_defaultQueryProperty) {
        QueryProperty::s_Instance.m_defaultQueryProperty = FEATSTD_NEW(QueryProperty);
        QueryProperty::s_Instance.m_defaultQueryProperty->m_isDefault = true;
        FEATSTD_DEBUG_ASSERT(0 == QueryProperty::s_Instance.m_boundingBox);
        QueryProperty::s_Instance.m_boundingBox = CreateBoundingBoxMesh();
    }

    return *(QueryProperty::s_Instance.m_defaultQueryProperty);
}

QueryProperty::QueryInfo* QueryProperty::GetQueryInfo(const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy)
{
    for (UInt i = 0; i < static_cast<UInt>(m_queryInfos.Size()); ++i) {
        if (occlusionCullingCameraRenderStrategy == m_queryInfos[i].m_occlusionCullingCameraRenderStrategy) {
            return &(m_queryInfos[i]);
        }
    }

    return 0;
}

void QueryProperty::RenderBoundingBox(const Node* node) const
{
    Mesh* boundingBox = QueryProperty::s_Instance.m_boundingBox;
    FEATSTD_DEBUG_ASSERT(0 != boundingBox);
    if (0 != boundingBox) {
        // if the assert hits, you need to call ComputeBoundingBox() on your node first
        FEATSTD_DEBUG_ASSERT(node->IsBoundingBoxValid());
        Vector3 minBounds;
        Vector3 maxBounds;
        node->GetBoundingBox(minBounds, maxBounds);

        Vector3 scale = maxBounds - minBounds;
        scale *= 0.5F;
        boundingBox->SetScale(scale);
        Vector3 boundCenter = maxBounds - scale;
        boundingBox->SetPosition(boundCenter);

        boundingBox->SetGenericTransform(node->GetWorldTransform());
        boundingBox->Render();
    }
}


/**
*  ClearQueryPropertyTraverser clears the QueryProperties at the Node given and all its descendants.
*/
class ClearQueryPropertyTraverser : public TreeTraverser {
    typedef TreeTraverser Base;

public:
    ClearQueryPropertyTraverser() {}
    ~ClearQueryPropertyTraverser() {}

protected:
    virtual TraverserAction ProcessNode(Node& node) override
    {
        QueryProperty* queryProperty = QueryProperty::Get(node);
        if (0 != queryProperty) {
            queryProperty->CleanupQueryInfos();
        }

        return ProceedTraversing;
    }

private:

    FEATSTD_MAKE_CLASS_UNCOPYABLE(ClearQueryPropertyTraverser);
};

void QueryProperty::OnNodeRemoved(Node* parent, Node* node)
{
    FEATSTD_UNUSED(parent);
    ClearQueryPropertyTraverser clearQueryPropertyTraverser;
    clearQueryPropertyTraverser.Traverse(*node);
}

Mesh* QueryProperty::CreateBoundingBoxMesh()
{
    VertexGeometryBuilder builder;
    builder.Clear();
    UInt16 vertexSize = 3 * sizeof(Float);
    UInt32 vertexCount = static_cast<UInt32>(sizeof(c_cubeData)) / static_cast<UInt32>(vertexSize);
    bool success = builder.SetVertexElementRange(VertexGeometry::Position, 0, 0, VertexGeometry::Float32_3,
        vertexCount, vertexSize, c_cubeData);
    success = success && builder.SetIndexElementRange(0, sizeof(c_cubeIndexData) / sizeof(UInt16), c_cubeIndexData);

    VertexBuffer::SharedPointer vb = VertexBuffer::Create();
    success = success && (vb != 0);
    if (success) {
        VertexGeometry* cubeGeometry = builder.GetVertexGeometry();
        success = vb->SetVertexGeometry(cubeGeometry, VertexBuffer::VertexGeometryDisposer::Dispose);
        vb->SetPrimitiveType(VertexBuffer::TriangleStrip);
    }

    Appearance::SharedPointer appearance = Appearance::Create();
    RenderMode::SharedPointer renderMode = RenderMode::Create();
    MemoryManagement::SharedPointer<Shader> shader = Shader::Create();
    success = success && (appearance != 0) && (renderMode != 0) && (shader != 0);
    if (success) {
        static_cast<void>(shader->SetVertexShader(c_VertexShader, 0));
        static_cast<void>(shader->SetFragmentShader(c_FragmentShader, 0));
        appearance->SetShader(shader);
        renderMode->SetColorWriteEnabled(false, false, false, false);
        renderMode->SetCulling(RenderMode::BackFaceCulling);
        renderMode->SetDepthTestEnabled(true);
        renderMode->SetDepthWriteEnabled(false);
        appearance->SetRenderMode(renderMode);
    }

    Mesh* boundingBox = 0;
    if (success) {
        boundingBox = Mesh::Create();
        if (boundingBox != 0) {
            boundingBox->SetVertexBuffer(vb);
            boundingBox->SetAppearance(appearance);
            if (!boundingBox->Upload())
            {
                boundingBox->DisposeSelf();
                boundingBox = 0;
            }
        }
    }

    return boundingBox;
}

}

}
