//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CanderaQuery_h)
#define CanderaQuery_h

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>



namespace Candera {

class RenderDevice;


/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief The class Query enables an interrogation of the Render Device for certain states, for instance whether an object is occluded or not.
 * Queries are related to one specific rendering context and must not be uploaded to multiple contexts.
 * Find below an example, how to test if any pixels have been rendered by using a query:
 *
 * \code
 * SharedPointer<Query> query;
 * query = Query::Create(Query::QueryAnySamplesPassed);
 * query->Upload();
 * query->Begin(); 
 *
 * // Do some rendering here.
 *
 * query->End(); 
 *
 * // Process other operations to provide GPU with sufficient time to complete query or skip query evaluation if result is not available yet.
 *
 * if (query->IsResultAvailable() && (query->GetResult() > 0)) {
 *     // Samples of low resolution model passed the depth test, so render high resolution model, ideally in next render pass.
 * }
 * \endcode
 */


class Query {

public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Query);
 
        enum Type
        {
            QueryAnySamplesPassed,                   ///< Query determines if any sample passes the depth test.
            QueryAnySamplesPassedConservative,       ///< Similar to QueryAnySamplesPassed type, but less precise for queries performance's sake. May return true even if no sample passed.
            QueryTransformFeedbackPrimitivesWritten  ///< Query determines how many vertices have been written into bound transform feedback buffer(s).
        };
 
        /**
         *  Creates an instance of this class.
         *  @return   A Query::SharedPointer, which manages the lifetime of the instance.
         */
        static Query::SharedPointer Create(Type queryType);

        /**
         *  Destructor
         */
        ~Query();
 
        /**
        *  Upload Query object and acquire a video memory handle.
        *  Note: When the query is uploaded it is linked to the context currently active. It is prohibited to invoke functions of Query object in other contexts.
        *        Multiple uploads of a query object are not possible.
        *  @return true if creation of video memory handle succeeded. Returns false if query object is already uploaded in another context.
        */
        bool Upload();

        /**
        *  Unload Query object and free video memory handle.
        *  Note: Unload the query in the context it is linked to.
        *  @return true if deletion of video memory handle succeeded. Returns false if query object is unloaded from another context.
        */
        bool Unload();

        /**
          *  Starts the query with given query type. The Functions Begin() and End() delimit the boundaries of a query.
          *  Note: A Query object cannot be activated unless uploaded to video memory (See function Upload).
          *  @return Whether query could be activated in the render device (true) or not (false).
          */
        bool Begin() const;

        /**
          *  Ends the query with given query type. The Functions Begin() and End() delimit the boundaries of a query.
          *  @return Whether query could be deactivated in the render device (true) or not (false).
          */
        bool End() const;

        /**
        *  GetType returns the type of this query object.
        *  @return type of this query object.
        */
        Type GetType() const { return m_queryType; }

        /**
        * Check if query is complete and result can be retrieved immediately without any delay (see function GetResult).
        * Note: Frequent invocation of this function with same query object is guaranteed to return true eventually, but may result in a considerable performance loss.
        *       It is recommended to wait between one and three frames to check state, concrete number is platform dependent.
        * @return bool True if query object result is available, false otherwise.
        */
        bool IsResultAvailable() const;

        /**
        *  GetResult returns the result of the query.
        *  If Type is QueryAnySamplesPassed or QueryAnySamplesPassedConservative the return-value is a boolean, telling whether any samples have passed the depth test.
        *  Note: When the depth test is disabled the depth test is considered as passed.
        *  If Type is QueryTransformFeedbackPrimitivesWritten the value is an integer value telling how many vertices are written into the bound transform feedback buffer(s).
        *  Note: Invoke function IsResultAvailable() to determine whether result is returned immediately or a delay will occur. This function implicitly flushes the command pipeline.
        *        A query result cannot be retrieved when query is not uploaded, has not been activated before, or is in active state (see function Begin, End, and IsActive).
        *  @return Query result holds the result according to query type. If query result is invalid, the value is 0xFFffFFff.
        */
        UInt32 GetResult() const;

        /**
        *  Retrieve whether this Query object is active or not in current context (See functions Begin and End).
        *  Note: A query result cannot be retrieved while the query object is active.
        *  @return whether this query object is currently active or not.
        */
        bool IsActive() const;

        /**
        *  Retrieve query-handle of graphics device.
        *  @return the query-handle of graphics device.
        */
        Handle GetVideoMemoryHandle() const { return m_videoMemoryHandle; }

    private:
        friend class RenderDevice;

        Type m_queryType;
        Handle m_videoMemoryHandle;
        Int m_Context;

        Query(Type queryType);

        void SetVideoMemoryHandle(Handle handle) { m_videoMemoryHandle = handle;  }

        void SetContext(Int context) { m_Context = context; }
        Int GetContext() const { return m_Context; }

        FEATSTD_MAKE_CLASS_UNCOPYABLE(Query);
        CANDERA_SHARED_POINTER_DECLARATION();
};


/** @} */ // end of Core3D
} // namespace Candera

#endif
