//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Projection_H)
#define CANDERA_Projection_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

//forward declaration 
class ProjectionListener;

/**
 * @brief   A Projection provides a projection matrix (for use with Cameras).
 */
class Projection
{
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Projection);

        /**
         *  Constructor
         */
        Projection();

        /**
         *  Destructor
         */
        virtual ~Projection();// base class, polymorphic use likely

        /**
         *  Create a clone of this projection object.
         *  @return A shared pointer to a clone.
         */
        virtual SharedPointer Clone() const = 0;

        /**
         *  Retrieve the current projection matrix.
         *  @return The current projection matrix.
         */
        const Matrix4& GetProjectionMatrix() const;

        /**
         *  Adds a ProjectionListener object. Event/timing of the calls: when a parameter of the projection is set/modified.
         *  @param listener The lifetime of the listener is managed by the calling code. Projection does not take ownership.
         *  @return True if successful, false otherwise.
         */
        bool AddProjectionListener(ProjectionListener* listener);

        /**
         *  Removes a ProjectionListener object.
         *  @param listener Specifies the ProjectionListener object to remove from this Camera.
         *  @return True if successful, false otherwise.
         */
        bool RemoveProjectionListener(ProjectionListener* listener);

        FEATSTD_RTTI_DECLARATION();
    protected:
        Projection(const Projection& other);

        /**
         *  Notifies the listeners that a parameter of the projection has been modified;
         *  Should be called in all methods of inheriting classes that modify a projection parameter;
        */
        void NotifyListenersOnProjectionChanged();

        // used by derived classes for maintaining the result matrix, see member documentation
        void SetProjectionMatrix(const Matrix4& projectionMatrix) const;

        // used by derived classes for maintaining the result matrix, see member documentation
        virtual void UpdateProjectionMatrix() const = 0;

    private:
        Projection& operator=(const Projection& other);

        // mutable: derived classes are expected to implement caching / lazy computation of this
        // matrix, which will be functionally dependent on some parameters; this matrix here is not
        // a directly settable parameter, but a common storage point of an end result.
        mutable Matrix4 m_projection;

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

        typedef Internal::SingleLinkedList<ProjectionListener*> ProjectionListenerContainer;
        ProjectionListenerContainer m_projectionListeners;
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_Projection_H

