//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "PointSprite.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Line.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>

#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

static inline MemoryManagement::SharedPointer<GenericShaderParamSetter> CreateGenericShaderParamSetterInstance()
{
    MemoryManagement::SharedPointer<GenericShaderParamSetter> gsps = GenericShaderParamSetter::Create();
    gsps->SetLightActivationEnabled(false);
    gsps->SetPointSpriteActivationEnabled(true);
    return gsps;
}

static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance() {
    FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, s_gsps, CreateGenericShaderParamSetterInstance());
    return s_gsps;
}

static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

PointSprite::PointSprite() :
    Base(),
    m_pointSize(100.0F),
    m_pointSizeScaleA(1.0F),
    m_pointSizeScaleB(0.0F),
    m_pointSizeScaleC(0.0F)
{
}

PointSprite::PointSprite(const Candera::PointSprite &other) :
    Base(other),
    m_pointSize(other.m_pointSize),
    m_pointSizeScaleA(other.m_pointSizeScaleA),
    m_pointSizeScaleB(other.m_pointSizeScaleB),
    m_pointSizeScaleC(other.m_pointSizeScaleC)
{
}

PointSprite* PointSprite::Create()
{

    PointSprite* pointSprite = FEATSTD_NEW(PointSprite);
    if ((pointSprite!=0) && (!pointSprite->Initialize())) {
        FEATSTD_DELETE(pointSprite);
        pointSprite = 0;
    }
    return pointSprite;
}

void PointSprite::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

PointSprite::~PointSprite()
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::PointSprite::m_vertexGeometry, CANDERA_LINT_REASON_ASSOCIATION)
}

PointSprite* PointSprite::Clone() const
{
    PointSprite* pointSprite = FEATSTD_NEW(PointSprite)(*this);
    if ((pointSprite!=0) && (!pointSprite->Initialize())) {
        FEATSTD_DELETE(pointSprite);
        pointSprite = 0;
    }
    return pointSprite;
}

FEATSTD_LINT_NONCONST_METHOD(Candera::PointSprite::SetColor, "method not made const due to deprecation and therefore not actually setting any value")
void PointSprite::SetColor(const Color& color)
{
    //deprecated function that does not do anything anymore
    FEATSTD_UNUSED(color);
}

bool PointSprite::Upload()
{
    // Upload mandatory vertex buffer.
    bool success = m_vertexBuffer->Upload();
    if (! success) {
        FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
    }

    // Upload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> appearance = GetAppearance();
    if (appearance != 0) {
        if (!appearance->Upload()) {
            FEATSTD_LOG_ERROR("Upload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

bool PointSprite::Unload()
{
    // Unload mandatory vertex buffer.
    bool success = m_vertexBuffer->Unload();
    if (! success) {
        FEATSTD_LOG_ERROR("Unload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
    }

    // Unload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> appearance = GetAppearance();
    if (appearance != 0) {
        if (!appearance->Unload()) {
            FEATSTD_LOG_ERROR("Unload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

static NodeRenderSequence& GetNodeRenderSequence()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
    return renderSequence;
}

static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

void PointSprite::Render()
{
    NodeRenderSequence& renderSequence = GetNodeRenderSequence();

    if (!IsRenderPrerequisiteFulfilled()) {
        FEATSTD_LOG_ERROR("Render precondition is not fulfilled for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        return;
    }

    renderSequence.Render(*this, m_vertexBuffer);
}

bool PointSprite::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
{
    Line line;
    if (!Math3D::CalculatePickLine(camera, static_cast<Float>(x), static_cast<Float>(y), line)) {
        return false;
    }
    if ((camera.GetRenderTarget()->GetWidth() == 0) || (camera.GetRenderTarget()->GetHeight() == 0)) {
        FEATSTD_LOG_ERROR("Render target has not been initialized correctly!");
        return false;
    }
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Rectangle::GetPosition, False positive because here GetPosition is not called from Candera::Rectangle.)
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Rectangle = const Candera::Camera, False positive because here no function matches with Candera::Rectangle.)
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Vector3 = Candera::Vector2, False positive because both relevant the argument and the parameter are Candera::Vector3.)
    const Float distanceToCamera = GetPosition().GetDistanceTo(camera.GetPosition());
    const Float sizeFactorAbc = Math::SquareRoot(1.0F/(GetPointSizeScaleA() + (GetPointSizeScaleB() * distanceToCamera) +
                                                        (GetPointSizeScaleC() * distanceToCamera * distanceToCamera)));
    const Float finalSize = GetPointSize() * sizeFactorAbc; //Size of point sprite in pixels.

    Matrix4 mvp = GetWorldTransform() * camera.GetViewProjectionMatrix(); 
    Vector4 posMVPTransformed(0.0F, 0.0F, 0.0F, 1.0F);
    //Get position in normalized device coordinates.
    posMVPTransformed.TransformCoordinate(mvp);

    bool picked = false;

    if (posMVPTransformed.GetW() != 0.0F) {

        //Bring pixel size of PointSprite into normalized device coordinate size.
        Float XSizeHalf = finalSize / static_cast<Float>(camera.GetRenderTarget()->GetWidth());
        Float YSizeHalf = finalSize / static_cast<Float>(camera.GetRenderTarget()->GetHeight());

        //Get PointSprite quad in normalized device coordinates.
        Vector3 posNormalizedDeviceCoordinates = Vector3(posMVPTransformed)/posMVPTransformed.GetW(); //Perspective Division 
        Float left = posNormalizedDeviceCoordinates.GetX() - XSizeHalf; 
        Float right = posNormalizedDeviceCoordinates.GetX() + XSizeHalf;  
        Float top = posNormalizedDeviceCoordinates.GetY() + YSizeHalf; 
        Float bottom = posNormalizedDeviceCoordinates.GetY() - YSizeHalf;    

        //Bring picking test pixel coordinates into normalized device coordinates.
        Float xNormalized = -1.0F + ((static_cast<Float>(x) / static_cast<Float>(camera.GetRenderTarget()->GetWidth())) * 2.0F );
        Float yNormalized = 1.0F - ((static_cast<Float>(y) / static_cast<Float>(camera.GetRenderTarget()->GetHeight())) * 2.0F );

        //Finally simply check if test coordinates are inside the PointSprite quad.
        picked = (xNormalized >= left) && (xNormalized <= right)
            && (yNormalized <= top) && (yNormalized >= bottom); 

        if (picked) {
            //If picked get distance to near plane.
            distance = camera.GetViewingFrustum().GetDistanceToNearPlane(GetWorldPosition());
        }
    }

    return picked;
}

FEATSTD_RTTI_DEFINITION(PointSprite, Renderable)

bool PointSprite::Initialize()
{
    FEATSTD_SYNCED_STATIC_OBJECT(VertexBuffer::SharedPointer, s_vertexBuffer, CreateVertexBuffer());

    m_vertexBuffer = s_vertexBuffer;
    return (0 != m_vertexBuffer.GetPointerToSharedInstance());
}

VertexBuffer::SharedPointer PointSprite::CreateVertexBuffer() const
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_INSTANCESOBTAINABLE);
    static const VertexGeometry::VertexElementFormat vertexDataFormat[] = {
        { 0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 }
    };
    static const Float vertexData[3] = { 0.F, 0.F, 0.F };
    static VertexGeometry vertexGeometry(
        vertexData, 0, vertexDataFormat, 0, 0, 0,
        1, sizeof(vertexData), 1, 0, VertexGeometry::VideoMemory,
        VertexGeometry::ArrayBuffer, VertexGeometry::StaticWrite);

    VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
    if (vertexBuffer != 0)
    {
        if (!vertexBuffer->SetVertexGeometry(&vertexGeometry, 0))
        {
            return VertexBuffer::SharedPointer(0);
        }

        vertexBuffer->SetPrimitiveType(VertexBuffer::Points);
    }

    return vertexBuffer;
}
} // namespace Candera
