//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "PlanarShadow.h"
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/Billboard.h>
#include <Candera/Engine3D/Core/LineList.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {
using namespace Diagnostics;

class IdGenerator{
public:
    static Int32 GetNext()
    {
        return ++Get();
    }
    static void Set(Int32 id) {
        if (id > Get()) {
            Get() = id;
        }
    }
private:
    static Int32& Get() {
        static Int32 s_id = 0;
        return s_id;
    }
};

PlanarShadow::PlanarShadow():
    Base(),
    m_isAutoVertexBufferEnabled(true),
    m_Id(IdGenerator::GetNext()),
    m_associatedLight(0),
    m_plane(Plane(Vector3(0.0F, 1.0F, 0.0F),0.0F)),
    m_alignmentNode(0)
{
}

PlanarShadow::PlanarShadow(const PlanarShadow& rhs) :
    Base(rhs),
    m_isAutoVertexBufferEnabled(rhs.m_isAutoVertexBufferEnabled),
    m_Id(IdGenerator::GetNext()),
    m_associatedLight(0),
    m_plane(rhs.GetPlane()),
    m_alignmentNode(rhs.GetAlignmentNode())
{
}

PlanarShadow::~PlanarShadow()
{
    m_associatedLight = 0;
    m_alignmentNode = 0;
}

PlanarShadow* PlanarShadow::Create()
{
    PlanarShadow* shadow = FEATSTD_NEW(PlanarShadow);

    if (shadow != 0) {
        MemoryManagement::SharedPointer<Appearance> shadowAppearance = Appearance::Create();
        if (shadowAppearance != 0) {
            shadowAppearance->SetMaterial(Material::Create());
            shadowAppearance->GetMaterial()->SetDiffuse(Color(0.0F, 0.0F, 0.0F, 0.5F));
            shadowAppearance->SetRenderMode(RenderMode::Create());
            shadowAppearance->GetRenderMode()->SetBlendingEnabled(true);
            shadowAppearance->GetRenderMode()->SetDepthWriteEnabled(false);
            shadowAppearance->GetRenderMode()->SetDepthTestEnabled(true);
            shadowAppearance->GetRenderMode()->SetStencilTestEnabled(true);
            shadowAppearance->GetRenderMode()->SetStencilWriteMask(~(static_cast<UInt32>(0U)));
            shadowAppearance->GetRenderMode()->SetStencilFunction(RenderMode::StencilFunctionData(RenderMode::CompareNotEqual, shadow->GetStencilBufferId (), 0xFFFFFFFFU));
            shadowAppearance->GetRenderMode()->SetStencilOperation(RenderMode::StencilOperationData(RenderMode::Keep, RenderMode::Keep, RenderMode::Replace));
            shadow->SetAppearance(shadowAppearance);
        }
    }

    return shadow;
}

const Matrix4& PlanarShadow::GetWorldTransform() const
{
    m_shadowTransformation = Matrix4(Base::GetWorldTransform());

    if (m_associatedLight != 0) {

        Plane plane = m_plane; //we need the plane locally so we can transform it
        if(m_alignmentNode != 0) {
            Vector3 planeDirection = plane.GetDirection();
            Float offset = plane.GetDistance();

            //align the receiving plane with the alignment node
            planeDirection.TransformCoordinate(m_alignmentNode->GetWorldRotation());
            Vector3 position = m_alignmentNode->GetWorldPosition();

            //we now have a point on the plane and the correct plane direction, 
            //we need to use this plane to calculate the correct distance to the world origin 
            Float distance = (-position).GetDotProduct(planeDirection) + offset;
            plane.SetDirection(planeDirection);
            plane.SetDistance(distance);
        }

        Matrix4 shadow;
        if ((m_associatedLight->GetType() == Light::Point) ||
            (m_associatedLight->GetType() == Light::Spot) ||
            (m_associatedLight->GetType() == Light::Directional)) {
            static_cast<void>(Math3D::SetShadow(shadow, *m_associatedLight, plane));
        }
        m_shadowTransformation = m_shadowTransformation * shadow;
    }

    return m_shadowTransformation;
}

void PlanarShadow::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

PlanarShadow* PlanarShadow::Clone() const
{
    return FEATSTD_NEW(PlanarShadow)(*this);
}

void PlanarShadow::OnAncestorAdded(Scene* /*scene*/)
{
    if (m_isAutoVertexBufferEnabled) {
        SetVertexBuffer(GetParentVertexBuffer());
    }
}

void PlanarShadow::OnAncestorRemoved(Scene* /*scene*/)
{
    if (m_isAutoVertexBufferEnabled && (GetParent() == 0)) {
        if ( GetVertexBuffer() != 0 ) {
            SetVertexBuffer(MemoryManagement::SharedPointer<VertexBuffer>(0));
        }
    }
}

MemoryManagement::SharedPointer<VertexBuffer> PlanarShadow::GetParentVertexBuffer() const
{
    MemoryManagement::SharedPointer<VertexBuffer> parentVb(0);

    Node* parent = GetParent();

    if (parent != 0) {
        if (parent->IsTypeOf(Billboard::GetTypeId())) {
            Billboard* bb = Dynamic_Cast<Billboard*>(parent);
            FEATSTD_DEBUG_ASSERT(bb != 0);
            parentVb = bb->GetVertexBuffer();
        }
        else if(parent->IsTypeOf(LineList::GetTypeId())) {
            LineList* ll = Dynamic_Cast<LineList*>(parent);
            FEATSTD_DEBUG_ASSERT(ll != 0);
            parentVb = ll->GetVertexBuffer();
        }
        else if(parent->IsTypeOf(Mesh::GetTypeId())) {
            Mesh* mesh = Dynamic_Cast<Mesh*>(parent);
            FEATSTD_DEBUG_ASSERT(mesh != 0);
            parentVb = mesh->GetVertexBuffer();
        }
        else {
            //do nothing
        }
    }

    return parentVb;
}

void PlanarShadow::SetStencilBufferId(Int32 id)
{
    m_Id = id;
    IdGenerator::Set(id);
}

bool PlanarShadow::IsRenderPrerequisiteFulfilled() const
{
    return ((m_associatedLight !=0) && (m_associatedLight->GetType() != Light::Ambient) && Base::IsRenderPrerequisiteFulfilled());
}

FEATSTD_RTTI_DEFINITION(PlanarShadow, Mesh)
} // namespace Candera
