//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "PickFrustum.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/Frustum.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Line.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

PickFrustum::PickFrustum(Camera* camera, const Rectangle& rect):
    m_camera(camera),
    m_rectangle(rect)
{
    PickFrustum::Recalculate();
}

PickFrustum::~PickFrustum()
{
    m_camera = 0;
}

void PickFrustum::SetCamera(Camera* camera)
{
    m_camera = camera;
    Recalculate();
}

void PickFrustum::SetRectangle(const Rectangle& rect)
{
    m_rectangle = rect;
    Recalculate();
}

void PickFrustum::Recalculate()
{
    if (m_camera == 0) {
        FEATSTD_LOG_ERROR("Pick frustum recalculate failed, m_camera == 0.");
        return;
    }

    Line lines[4];
    bool success = Math3D::CalculatePickLine(*m_camera, m_rectangle.GetLeft(), m_rectangle.GetTop(), lines[0]);
    success = success && Math3D::CalculatePickLine(*m_camera, m_rectangle.GetLeft() + m_rectangle.GetWidth(), m_rectangle.GetTop(), lines[1]);
    success = success && Math3D::CalculatePickLine(*m_camera, m_rectangle.GetLeft() + m_rectangle.GetWidth(), m_rectangle.GetTop() + m_rectangle.GetHeight(), lines[2]);
    success = success && Math3D::CalculatePickLine(*m_camera, m_rectangle.GetLeft(), m_rectangle.GetTop() + m_rectangle.GetHeight(), lines[3]);

    if (!success) {
        FEATSTD_LOG_ERROR("Pick frustum recalculate failed, pick line calculation failed.");
        return;
    }

    Vector3 a;
    Vector3 b;
    Vector3 normal;

    //Near Plane
    a = lines[1].GetStart() - lines[0].GetStart();
    b = lines[3].GetStart() - lines[0].GetStart();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[NearPlane].SetDirection(normal);
    m_planes[NearPlane].SetDistance(-lines[0].GetStart().GetDotProduct(m_planes[NearPlane].GetDirection()));
    //Far Plane
    a = lines[0].GetEnd() - lines[1].GetEnd();
    b = lines[3].GetEnd() - lines[0].GetEnd();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[FarPlane].SetDirection(normal);
    m_planes[FarPlane].SetDistance(-lines[0].GetEnd().GetDotProduct(m_planes[FarPlane].GetDirection()));
    //Left Plane
    a = lines[0].GetStart() - lines[0].GetEnd();
    b = lines[3].GetStart() - lines[0].GetStart();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[LeftPlane].SetDirection(normal);
    m_planes[LeftPlane].SetDistance(-lines[0].GetStart().GetDotProduct(m_planes[LeftPlane].GetDirection()));
    //Right Plane
    a = lines[1].GetEnd() - lines[1].GetStart();
    b = lines[2].GetStart() - lines[1].GetStart();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[RightPlane].SetDirection(normal);
    m_planes[RightPlane].SetDistance(-lines[1].GetStart().GetDotProduct(m_planes[RightPlane].GetDirection()));
    //Top Plane
    a = lines[0].GetEnd() - lines[1].GetStart();
    b = lines[0].GetEnd() - lines[0].GetStart();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[TopPlane].SetDirection(normal);
    m_planes[TopPlane].SetDistance(-lines[1].GetStart().GetDotProduct(m_planes[TopPlane].GetDirection()));
    //Bottom Plane
    a = lines[2].GetStart() - lines[3].GetEnd();
    b = lines[3].GetEnd() - lines[3].GetStart();
    normal = a.GetCrossProduct(b);
    static_cast<void>(normal.Normalize());
    m_planes[BottomPlane].SetDirection(normal);
    m_planes[BottomPlane].SetDistance(-lines[2].GetStart().GetDotProduct(m_planes[BottomPlane].GetDirection()));
}

} // namespace Candera

