//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(PerspectiveProjection_H)
#define PerspectiveProjection_H

#include <Candera/Engine3D/Core/Projection.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
/**
 *  @brief   Perspective projection.
 *
 *  This class represents a type of transformation that projects a 3D frustum to a 2D surface.
 *  The 3D frustum can be constructed by sectioning a rectangular pyramid with two parallel planes,
 *  perpendicular to the Z axis of the camera space. The pyramid is constructed such that 
 *  it has two of the bounding planes intersecting over the X axis and the other two over the Y axis,
 *  and the Z axis goes exactly through the middle of the pyramid.
 *  This type of pyramid can be uniquely identified by the angle between the two planes intersecting over
 *  the X axis (the two planes are called "top plane" and "bottom plane", and the angle  "field of view")
 *  and the ratio between the width and height of any rectangle generated by the sectioning planes. (this
 *  is called "aspect ratio").
 *  The frustum obtained from this pyramid can be uniquely identified by the distance from origin to
 *  the two sectioning planes. As such the planes are depicted as "nearest view plane" and "farthest view plane".
 *
 *  The projection matrix generated by this object looks like this:
 *  p0 = tangent("field of view" / 2)
 *  p1 = p0 * "aspect ratio"
 *  p2 = ("nearest view plane" + "farthest view plane") / ("nearest view plane" - "farthest view plane")
 *  p3 = (2 * "nearest view plane" * "farthest view plane") / ("nearest view plane" - "farthest view plane")
 *       Projection matrix:
 *       --------------------- 
 *       | p1 |  0 |  0 |  0 | 
 *       |  0 | p0 |  0 |  0 | 
 *       |  0 |  0 | p2 | -1 | 
 *       |  0 |  0 | p3 |  0 | 
 *       ---------------------
 */
 
class PerspectiveProjection : public Projection
{
    FEATSTD_TYPEDEF_BASE(Projection);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(PerspectiveProjection);

        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~PerspectiveProjection() override;

        // Overrides Projection::Clone.
        virtual Projection::SharedPointer Clone() const override;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Set the distance to the nearest of the parallel planes of the frustum.
         *  @param nearZ Distance to near plane of the frustum to be set.
         */
        void SetNearZ(Float nearZ);

        /**
         *  Retrieve the distance to the nearest of the parallel planes of the frustum.
         *  @return The distance to the nearest of the parallel planes of the frustum.
         */
        Float GetNearZ() const;

        /**
         *  Set the distance to the farthest of the parallel planes of the frustum.
         *  @param farZ Distance to far plane of the frustum to be set.
         */
        void SetFarZ(Float farZ);

        /**
         *  Retrieve the distance to the farthest of the parallel planes of the frustum.
         *  @return The distance to the farthest of the parallel planes of the frustum.
         */
        Float GetFarZ() const;

        /**
         *  Set the vertical field of view (the angle in degrees between the top plane and the bottom plane).
         *  @param fovYDegrees Vertical field of view (angle in degrees between top and bottom plane) to be set.
         */
        void SetFovYDegrees(Float fovYDegrees);

        /**
         *  Retrieve the vertical field of view (the angle in degrees between the top plane and the bottom plane).
         *  @return The vertical field of view.
         */
        Float GetFovYDegrees() const;

        /**
         *  Set the aspect ratio (the ratio between the horizontal extent and the vertical extent of the
         *  view frustum measured on any of the parallel planes). 
         *  @param aspectRatio Aspect ratio to be set.
         */
        void SetAspectRatio(Float aspectRatio);

        /**
         *  Retrieve the aspect ratio (the ratio between the horizontal extent and the vertical extent of the
         *  view frustum measured on any of the parallel planes). 
         *  @return The aspect ratio.
         */
        Float GetAspectRatio() const;

    protected:
        mutable bool m_isMatrixUpToDate;// $CACHE

        // override (Projection)
        virtual void UpdateProjectionMatrix() const override;

        // protected because only ::Create() should be used to create an instance of this class, but derived ones should be able to use the constructors.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::PerspectiveProjection::PerspectiveProjection, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            PerspectiveProjection();
        PerspectiveProjection(const PerspectiveProjection& other);

    private:
        PerspectiveProjection& operator=(const PerspectiveProjection& other);

        Float m_nearZ;
        Float m_farZ;
        Float m_fovYDegrees;
        Float m_aspectRatio;
};
 
/** @} */ // end of Core3D
 
} // namespace Candera

#endif

