//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "OrthographicProjection.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

OrthographicProjection::OrthographicProjection():
    Projection(),
    m_isMatrixUpToDate(false),
    m_nearZ(1.0F),
    m_farZ(1000.0F),
    m_width(100.0F),
    m_height(100.0F)
{
}

OrthographicProjection::SharedPointer OrthographicProjection::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    OrthographicProjection* ptr = FEATSTD_NEW(OrthographicProjection)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Orthographic projection create failed, out of memory.");
    }
    OrthographicProjection::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

OrthographicProjection::~OrthographicProjection()
{
}

Projection::SharedPointer OrthographicProjection::Clone() const
{
    return SharedPointer(FEATSTD_NEW(OrthographicProjection)(*this));
}

OrthographicProjection::OrthographicProjection(const OrthographicProjection& other):
    Projection(other),
    m_isMatrixUpToDate(other.m_isMatrixUpToDate),
    m_nearZ(other.m_nearZ),
    m_farZ(other.m_farZ),
    m_width(other.m_width),
    m_height(other.m_height)
{
}

FEATSTD_RTTI_DEFINITION(OrthographicProjection, Projection)

void OrthographicProjection::SetNearZ(Float nearZ)
{
    m_nearZ = nearZ;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float OrthographicProjection::GetNearZ() const
{
    return m_nearZ;
}

void OrthographicProjection::SetFarZ(Float farZ)
{
    m_farZ = farZ;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float OrthographicProjection::GetFarZ() const
{
    return m_farZ;
}

void OrthographicProjection::SetWidth(Float width)
{
    m_width = width;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float OrthographicProjection::GetWidth() const
{
    return m_width;
}

void OrthographicProjection::SetHeight(Float height)
{
    m_height = height;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float OrthographicProjection::GetHeight() const
{
    return m_height;
}

void OrthographicProjection::UpdateProjectionMatrix() const
{
    if (m_isMatrixUpToDate) {
        return;
    }
    Matrix4 orthoMatrix;
    orthoMatrix = Math3D::SetOrthographicProjection(orthoMatrix, m_nearZ, m_farZ, -m_width / 2.0F, +m_width / 2.0F, -m_height / 2.0F, +m_height / 2.0F);
    SetProjectionMatrix(orthoMatrix);
    m_isMatrixUpToDate = true;
}
}

