//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_NodeRenderSequence_H)
#define CANDERA_NodeRenderSequence_H

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

class Appearance;
class Node;
class Shader;
class AbstractShaderParamSetter;

/**
 *  @brief The class NodeRenderSequence implements an activation sequence common to several concrete Node classes like
 *         Mesh, Billboard, PointSprite, and LineList and also takes care of rendering. In case
 *         of MultiPassAppearances
 */
class NodeRenderSequence
{
    public:
        /**
         *  Constructs a NodeRenderSequence object.
         *  @param defaultShaderParams ShaderParamSetter associated to this NodeRenderSequence.
         */
        NodeRenderSequence(MemoryManagement::SharedPointer<AbstractShaderParamSetter> defaultShaderParams):  m_defaultShaderParams(defaultShaderParams) {}

        /**
         *  Destructs a NodeRenderSequence object.
         */
        ~NodeRenderSequence() {}

        /**
         *  Renders the given node considering Appearances and MultiPassAppearances. In
         *  case of MultiPassAppearances Activate is called for every render pass before invoking the
         *  VertexBuffers render method.
         *  @param node defines the Node object to render.
         *  @param vertexBuffer Vertex Buffer to render.
         */
        void Render(Node& node, const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer);

        /**
         *  Activates the given node using the provided Appearance.
         *  @param node defines the Node object to activate.
         *  @param vertexBuffer Vertex Buffer to bind to the given shader.
         *  @param appearance Appearance to activate for rendering.
         *  @return True if appearance is not null and appearance (along with it's shader) could be activated,
         *          vertex buffer could be bound to the shader and the shader parameter setter could be activated.
         *          False if any of those failed.
         */
        bool Activate(Node& node, const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer,const MemoryManagement::SharedPointer<Appearance>& appearance);

    private:
        MemoryManagement::SharedPointer<AbstractShaderParamSetter> m_defaultShaderParams; // ShaderParamSetter object used, if node given has no ShaderParamSetter object set explicitly.

        /**
         *  Binds the attributes of the shader given.
         *  @param shader Shader to which the attribute shall be bound.
         *  @param vertexBuffer Vertex buffer which shall be bound to the shader.
         *  @return False if binding of vertex buffer to the shader failed, true otherwise.
         */
        bool BindAttributes(const MemoryManagement::SharedPointer<Shader>& shader, const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer);
};

/** @} */ // end of Core3D

} // namespace Candera

#endif
