//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_NodeListener_H)
#define CANDERA_NodeListener_H

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

class Node;

/**
 * @brief   A NodeListener defines hooks that are called when certain Node functions are triggered, e.g. when a Node's transformation changes.
 *          In order to register a NodeListener to a Node simply derive from NodeListener and override pure virtual functions with custom code.
 */
class NodeListener
{
    public:
        /**
         * Constructs a NodeListener object.
         */
        NodeListener() {}

        /**
         * Destructs a NodeListener object.
         */
        virtual ~NodeListener() {}

        /**
         * Is invoked when the transformation of a node is assumed to change.
         * @param node The node to listen to.
         */
        virtual void OnTransformChange(Node* node) = 0;

        /**
         * Invoked whenever the node is removed from its parent.
         */
        virtual void OnNodeRemoved(Node* parent, Node* node) = 0;

        /**
         * Invoked whenever a parent is set for the node.
         */
        virtual void OnNodeAdded(Node* parent, Node* node) { FEATSTD_UNUSED2(parent, node); }

        /**
         * Invoked whenever a child node is added to the node.
         */
        virtual void OnChildAdded(Node* child, Node* node) { FEATSTD_UNUSED2(child, node); }

        /**
         * Invoked whenever a child of the node is removed.
         */
        virtual void OnChildRemoved(Node* child, Node* node) { FEATSTD_UNUSED2(child, node); }

    protected:
        NodeListener(const NodeListener&) {};
        NodeListener& operator=(const NodeListener&);
};

/** @} */ // end of Core3D

} // namespace Candera

#endif// CANDERA_NodeListener_H

