//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MorphingMesh_H)
#define CANDERA_MorphingMesh_H

#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 *  @brief MorphingMesh is a Mesh that stores weight values for morphing, and activates them as uniforms at render time.
 *  MorphingMesh is intended to be used with a VertexBuffer/Shader combination that is actually useful for morphing.
 *  
 *  One way of obtaining a suitable VertexBuffer is using the class function CreateSplicedVertexBuffer, that creates a combined vertex buffer
 *  that can be passed to function SetVertexBuffer. 
 * 
 *  Example: morphMesh->SetVertexBuffer(MorphingMesh::CreateSplicedVertexBuffer(vertexBufferArray, arraySize));
 *  
 *  A Shader must make use of the morphWeight uniforms (as named in ShaderParamNames) to benefit from MorphingMesh, see shader RefTransLight1Morph.vert.
 */
class MorphingMesh:  public Mesh
{
    FEATSTD_TYPEDEF_BASE(Mesh);

    public:
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::MorphingMesh::Create, "Create")
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to created MorphingMesh object.
         */
        static MorphingMesh* Create();

        /**
         *  Destructor
         */
        virtual ~MorphingMesh() override {}

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance or VertexBuffer are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual MorphingMesh* Clone() const override;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Sets a morph weight.
         *  @param index of the conceptual Sub-VertexBuffer that the VertexBuffer of this object consists of.
         *  @param weight of the Sub-VertexBuffer, passed as a uniform to the Shader.
         */
        void SetMorphWeight(Int32 index, Float weight);

        /**
         *  Retrieves the morph weight of this mesh.
         *  @param index The index of the conceptual Sub-VertexBuffer that the VertexBuffer of this object consists of.
         *  @return The morph weight of this mesh.
         */
        Float GetMorphWeight(Int32 index) const;

        /**
         *  Creates a new VertexBuffer from an array of input VertexBuffers.
         *  Think of a VertexBuffer as a table where each column corresponds to an attribute, and each row to a vertex. The new VertexBuffer
         *  corresponds to the table that results when all columns are put together side-by-side.
         *
         *  Preconditions:
         *  - The number of vertices matches in all VertexBuffers.
         *  - The buffer types (indexed vs array) match.
         *  - The total number that an attribute (like a_Position) occurs (considering all VertexBuffers) is 8 or less.
         *  - If the result should be useful, the vertices in each row must indeed correspond to each other semantically.
         *
         *  Properties of the result VertexBuffer:
         *  - The VertexBuffers are collated in the order they appear in the input array.
         *  - The offsets given in the ElementFormat entries have been adapted to match the new structure.
         *  - All usages of attributes (like Position/a_Position) are numbered (starting at 0) in order of occurrence, and named/assigned an
         *  AttributeSemantic according to the resulting index and ShaderParamNames.
         *  - All buffers (vertex array, vertex element format array, index buffer array (if needed) are allocated using FEATSTD_NEW, matching Disposers
         *   are set on the VertexGeometry.
         *  - The properties "memory pool" and "buffer usage" are copied from the input VertexBuffer at index 0 (which is often a "base" VertexBuffer).
         *
         *  @param vb an array of VertexBuffers. (Frequent use case: one "base" VertexBuffer and up to 7 targets.)
         *  @param vbCount the number of VertexBuffers in the array. Must be >= 1 and <= 8.
         *  @return a new VertexBuffer resulting from collating the inputs on success, a 0 VertexBuffer on failure.
         */
        static Candera::MemoryManagement::SharedPointer<Candera::VertexBuffer> CreateSplicedVertexBuffer(const Candera::MemoryManagement::SharedPointer<Candera::VertexBuffer>* vb, Int32 vbCount);

    protected:
        // Deliberately protected constructor and copy constructor, use Create() to create an instance of this object.
        MorphingMesh();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(MorphingMesh);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Renders this MorphingMesh. This method is overridden from Node.
         */
        virtual void Render() override;

        /**
         *  Computes axis-aligned bounding box (AABB) in object coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

    private:
        friend class GenericShaderParamSetter;
        enum { MaxMorphWeightCount = 8 }; ///< 8 corresponds to the maximum of 8 attributes like Position, Position1, .. Position7
        Float m_morphWeight[MaxMorphWeightCount];

        CdaDynamicProperties(Candera::MorphingMesh, Candera::Mesh);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_MorphingMesh_H
