//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MorphingMesh.h"
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);


static inline MemoryManagement::SharedPointer<GenericShaderParamSetter> CreateGenericShaderParamSetterInstance()
{
    MemoryManagement::SharedPointer<GenericShaderParamSetter> gsps = GenericShaderParamSetter::Create();
    gsps->SetMorphingMeshActivationEnabled(true);
    return gsps;
}
    
static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, s_gsps, CreateGenericShaderParamSetterInstance());
    return s_gsps;
}

static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

MorphingMesh::MorphingMesh() :
    Base()
{
    m_isBatchable = false; // Uniform array is used, therefore batching is not possible.

    m_morphWeight[0] = 1.0F;
    for (Int32 i = 1; i < MaxMorphWeightCount; ++i) {
        m_morphWeight[i] = 0.0F;
    }
}

MorphingMesh::MorphingMesh(const MorphingMesh& other) :
    Base(other)
{
    MemoryPlatform::Copy(&m_morphWeight[0], &other.m_morphWeight[0], sizeof(m_morphWeight));
}

MorphingMesh* MorphingMesh::Create()
{
    return FEATSTD_NEW(MorphingMesh);
}

void MorphingMesh::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

MorphingMesh* MorphingMesh::Clone() const
{
    return FEATSTD_NEW(MorphingMesh)(*this);
}

static NodeRenderSequence& GetNodeRenderSequence()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
    return renderSequence;
}

static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

void MorphingMesh::Render()
{
    NodeRenderSequence& renderSequence = GetNodeRenderSequence();

    if (!IsRenderPrerequisiteFulfilled()) {
        FEATSTD_LOG_ERROR("Render precondition is not fulfilled for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        return;
    }

    renderSequence.Render(*this, m_vertexBuffer);
}

FEATSTD_RTTI_DEFINITION(MorphingMesh, Mesh)

void MorphingMesh::SetMorphWeight(Int32 index, Float weight)
{
    if ((index < 0) || (index >= MaxMorphWeightCount)) {
        FEATSTD_LOG_ERROR("Morphing mesh set morph weight failed, index out of range.");
        return;
    }
    m_morphWeight[index] = weight;
}

Float MorphingMesh::GetMorphWeight(Int32 index) const
{
    if ((index < 0) || (index >= MaxMorphWeightCount)) {
        FEATSTD_LOG_ERROR("Morphing mesh get morph weight failed, index out of range.");
        return 0.0F;
    }

    return m_morphWeight[index];
}

/**
 * Helper functions for CreateSplicedVertexBuffer.
 */

/**
 * Finds next multiple; useful eg for alignment purposes.
 * @param n >= 0
 * @param modulus >= 0
 * @return N such that N >= n and N is a multiple of modulus
 */
static Int32 NextMultiple(Int32 n, Int32 modulus)
{
    const Int32 remainder = n % modulus;
    if (remainder == 0) {
        return n;
    }
    else {
        return (n - remainder) + modulus;
    }
}

static bool AreVertexBuffersCompatible(const SharedPointer<VertexBuffer>* vb, Int32 vbCount, UInt32 vertexCount, VertexGeometry::BufferType bufferType)
{
    for (Int32 i = 0; i < vbCount; ++i) {
        const VertexGeometry*const vg = vb[i]->GetVertexGeometry();
        if (vg->GetVertexCount() != vertexCount) {
            return false;
        }
        if (vg->GetBufferType() != bufferType) {
            return false;
        }
    }
    return true;
}

static UInt16 GetTotalElementFormatCount(const SharedPointer<VertexBuffer>* vb, Int32 vbCount)
{
    UInt16 sumElementCount = 0;
    for (Int32 i = 0; i < vbCount; ++i) {
        sumElementCount += vb[i]->GetVertexGeometry()->GetVertexElementCount();
    }
    return sumElementCount;
}

static UInt16 GetTotalStride(const SharedPointer<VertexBuffer>* vb, Int32 vbCount)
{
    UInt16 totalStride = 0;
    for (Int32 i = 0; i < vbCount; ++i) {
        const SharedPointer<VertexBuffer> vBuffer = vb[i];
        const VertexGeometry* vg = vBuffer->GetVertexGeometry();
        totalStride += vg->GetVertexStride();
    }
    return totalStride;
}

static VertexGeometry::VertexElementFormat* AllocateElementFormats(Int32 totalElementFormatCount)
{
    return FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, static_cast<UInt32>(totalElementFormatCount));
}

static void FreeElementFormats(VertexGeometry::VertexElementFormat* elementFormats)
{
    FEATSTD_DELETE_ARRAY_T(elementFormats, VertexGeometry::VertexElementFormat);
}

static Char* AllocateVertexArray(Int32 vertexCount, Int32 totalStride)
{
    const Int32 splicedVertexArraySizeInBytes = totalStride * vertexCount;
    const Int32 splicedVertexArraySizeMultipleOf4InBytes = NextMultiple(splicedVertexArraySizeInBytes, 4);
    Int32* const splicedVertexArray = FEATSTD_NEW_ARRAY(Int32, static_cast<SizeType>(splicedVertexArraySizeMultipleOf4InBytes / 4));
    return reinterpret_cast<Char*>(splicedVertexArray);
}

static void FreeVertexArray(Char* vertexArray)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, "these casts are necessary to handle alignment and size in bytes")
        Int32* vertexArrayAsInt32Array = FeatStd::aligned_cast<Int32*>(vertexArray);
    FEATSTD_DELETE_ARRAY_T(vertexArrayAsInt32Array, Int32);
}

//commeted out due to cleaning W4 warnings, uncomment it if you'd like to use it
//static void FreeIndexArray(UInt16* indexArray)
//{
//    FEATSTD_DELETE_ARRAY_T(indexArray, UInt16);
//}

SharedPointer<VertexBuffer> MorphingMesh::CreateSplicedVertexBuffer(const SharedPointer<VertexBuffer>* vb, Int32 vbCount)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(641, "The AttributeSemantic values are designed and set to be compatible with integer arithmetic.")
        const Int32 usageCount = ShaderParamNames::AttributeSemanticCount / 8;// max index per attribute is 8
    UInt8 usageCounter[usageCount];
    for (Int32 i = 0; i < usageCount; ++i) {
        usageCounter[i] = 0;
    }

    SharedPointer<VertexBuffer> nullVertexBuffer(0);
    if ((vb == 0) || (vbCount <= 0)) {
        FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer failed, no input vertex buffers.");
        return nullVertexBuffer;
    }

    SharedPointer<VertexBuffer> result = VertexBuffer::Create();

    const UInt32 vertexCount = vb[0]->GetVertexGeometry()->GetVertexCount();
    const VertexGeometry::BufferType bufferType = vb[0]->GetVertexGeometry()->GetBufferType();

    if (!AreVertexBuffersCompatible(vb, vbCount, vertexCount, bufferType)) {
        FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer failed, input vertex buffers incompatible.");
        return nullVertexBuffer;
    }

    const UInt16 totalElementFormatCount = GetTotalElementFormatCount(vb, vbCount);
    const UInt16 totalStride = GetTotalStride(vb, vbCount);

    Char* const splicedVertexArrayAsByteArray = AllocateVertexArray(static_cast<Int32>(vertexCount), static_cast<Int32>(totalStride));
    if (splicedVertexArrayAsByteArray == 0) {
        FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer failed, AllocateVertexArray failed.");
        return nullVertexBuffer;
    }
    VertexGeometry::VertexElementFormat* const splicedElementFormats = AllocateElementFormats(totalElementFormatCount);
    if (splicedElementFormats == 0) {
        FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer failed, AllocateElementFormats failed.");
        FreeVertexArray(splicedVertexArrayAsByteArray);
        return nullVertexBuffer;
    }

    ResourceDataHandle indexDataHandle = ResourceDataHandle::InvalidHandle();
    if (bufferType != VertexGeometry::ArrayBuffer) {
        UInt32 indexDataSize = vb[0]->GetVertexGeometry()->GetIndexArrayResourceHandle().m_size;
        void* indexBuffer = FEATSTD_ALLOC(indexDataSize);
        if (indexBuffer == 0) {
            FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer failed, AllocateIndexBuffer failed.");
            FreeVertexArray(splicedVertexArrayAsByteArray);
            FreeElementFormats(splicedElementFormats);
            return nullVertexBuffer;
        }

        if (indexDataSize != VertexGeometry::IndexArrayResource::CopyData(indexBuffer, vb[0]->GetVertexGeometry()->GetIndexArrayResourceHandle(), 0, indexDataSize)) {
            FEATSTD_LOG_ERROR("Morphing mesh create spliced vertex buffer error. IndexBuffer not completely initialized.");
        }

        indexDataHandle = VertexGeometry::IndexArrayResource::CreateHandle(indexBuffer, MemoryManagement::FreeMemoryDisposer<const void*>::Dispose, indexDataSize);
    }

    VertexGeometry::VertexElementFormat* currentElementFormat = splicedElementFormats;
    UInt16 cumulatingStride = 0;
    for (Int32 i = 0; i < vbCount; ++i) {
        const SharedPointer<VertexBuffer> vBuffer = vb[i];
        const VertexGeometry* vg = vBuffer->GetVertexGeometry();
        const Int32 formatCount = vg->GetVertexElementCount();
        VertexGeometry::FormatArrayResource formatArrayResource(vg->GetFormatArrayResourceHandle());
        const VertexGeometry::VertexElementFormat* formats = formatArrayResource.GetData();
        for (Int32 k = 0; k < formatCount; ++k) {
            *currentElementFormat = formats[k];
            currentElementFormat->Offset += cumulatingStride;
            currentElementFormat->UsageIndex = usageCounter[currentElementFormat->Usage];
            ++usageCounter[currentElementFormat->Usage];
            ++currentElementFormat;
        }
        cumulatingStride += vg->GetVertexStride();
    }

    const UInt16 splicedStride = totalStride;

    cumulatingStride = 0;
    for (Int32 i = 0; i < vbCount; ++i) {
        SharedPointer<VertexBuffer> vBuffer = vb[i];
        const VertexGeometry* vg = vBuffer->GetVertexGeometry();
        VertexGeometry::VertexArrayResource vertexArrayResource(vg->GetVertexArrayResourceHandle());
        const Char* singleVertexArray = FeatStd::Internal::PointerToPointer<const Char*>(vertexArrayResource.GetData());
        const UInt16 singleStride = vg->GetVertexStride();

        for (UInt32 k = 0; k < vertexCount; ++k) {
            Char* splicedRecordBegin = splicedVertexArrayAsByteArray + (k * splicedStride) + cumulatingStride;
            const Char* singleRecordBegin = singleVertexArray + (k * singleStride);
            MemoryPlatform::Copy(splicedRecordBegin, singleRecordBegin, singleStride);
        }
        cumulatingStride += singleStride;
    }

    const VertexGeometry::MemoryPool memoryPool = vb[0]->GetVertexGeometry()->GetMemoryPool();
    const VertexGeometry::BufferUsage bufferUsage = VertexGeometry::StaticWrite;

    typedef AdaptedArrayDisposer<const void*, const Int32*> Int32ArrayDisposer;

    VertexGeometry* const vg = FEATSTD_NEW(VertexGeometry)(
        VertexGeometry::VertexArrayResource::CreateHandle(splicedVertexArrayAsByteArray, Int32ArrayDisposer::Dispose, vertexCount * splicedStride),
        indexDataHandle, 
        VertexGeometry::FormatArrayResource::CreateHandle(splicedElementFormats, VertexGeometry::VertexElementFormatDisposer::Dispose, static_cast<UInt32>(sizeof(VertexGeometry::VertexElementFormat)) * totalElementFormatCount),
        splicedStride, memoryPool, bufferType, bufferUsage);

    // Since it was just crated, result may not be uploaded.
    static_cast<void>(result->SetVertexGeometry(vg, &VertexBuffer::VertexGeometryDisposer::Dispose));
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(429, vg, "Ownership of vg is transferred to the result VertexBuffer.")
        return result;
}

bool MorphingMesh::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
{
    minBounds = Vector3(0.0F, 0.0F, 0.0F);
    maxBounds = Vector3(0.0F, 0.0F, 0.0F);

    bool result = true;
    for (UInt8 i = 0; i < MaxMorphWeightCount; i++) {
        if (m_morphWeight[i] != 0.0F) {
            Vector3 minBoundsTmp;
            Vector3 maxBoundsTmp;

            result = result && Math3D::ComputeBoundingBox(GetVertexBuffer(), minBoundsTmp, maxBoundsTmp, i);

            if (result) {
                minBounds += minBoundsTmp * m_morphWeight[i];
                maxBounds += maxBoundsTmp * m_morphWeight[i];
            }
        }
    }
    return result;
}
} // namespace Candera
