//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Mesh.h"
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/System/Diagnostics/Log.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance() {
    FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, gsps, GenericShaderParamSetter::Create());
    return gsps;
}

static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

Mesh::Mesh() :
    Base()
{
}

Mesh::Mesh(const Mesh& rhs) :
    Base(rhs)
{
}

Mesh::~Mesh()
{
}

Mesh* Mesh::Create()
{
    return FEATSTD_NEW(Mesh);
}

void Mesh::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

Mesh* Mesh::Clone() const
{
    return FEATSTD_NEW(Mesh)(*this);
}

bool Mesh::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
{
    return Math3D::ComputeBoundingBox(m_vertexBuffer, minBounds, maxBounds);
}

static NodeRenderSequence& GetNodeRenderSequence()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
    return renderSequence;
}

static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

void Mesh::Render()
{
    NodeRenderSequence& renderSequence = GetNodeRenderSequence();

    if ( !IsRenderPrerequisiteFulfilled() ) {
        FEATSTD_LOG_ERROR("MeshRender precondition is not fulfilled for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        return;
    }

    renderSequence.Render(*this, m_vertexBuffer);
}

bool Mesh::IsLineIntersectingGeometry(const Line& line, Float& distance) const
{
    if (!IsIntersectionTestEnabled()) {
        return false;
    }

    if ((m_vertexBuffer == 0) || (m_vertexBuffer->GetVertexGeometry() == 0) ||
        (m_vertexBuffer->GetVertexGeometry()->GetVertexCount() == 0)) {
        FEATSTD_LOG_ERROR("Is line intersecting geometry failed, invalid Vertex Buffer.");
        return false;
    }

    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(613, "Null value is asserted.");
    const VertexGeometry& vertexGeometry = *m_vertexBuffer->GetVertexGeometry();

    ElementFormatAccessor elementFormatAccessor(vertexGeometry);
    const VertexGeometry::VertexElementFormat* positionElement = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Position, 0);
    if (positionElement == 0) {
        return false;
    }

    bool intersection = false;
    Float min_distance = Math::MaxFloat();
    
    const Matrix4& worldTransform = GetWorldTransform();
    VertexAccessor vertexAccessor(vertexGeometry);
    for (VertexBuffer::PrimitiveIterator iterator = m_vertexBuffer->GetPrimitiveIterator(); iterator.IsValid(); ++iterator) {
        // get one triangle
        const VertexBuffer::Primitive& triangle = *iterator;
        const Float* position1 = FeatStd::Internal::PointerAdd<const Float*>(vertexAccessor.GetVertex(triangle.m_index[0]), positionElement->Offset);
        const Float* position2 = FeatStd::Internal::PointerAdd<const Float*>(vertexAccessor.GetVertex(triangle.m_index[1]), positionElement->Offset);
        const Float* position3 = FeatStd::Internal::PointerAdd<const Float*>(vertexAccessor.GetVertex(triangle.m_index[2]), positionElement->Offset);

        Vector3 triangleVertex1(position1[0], position1[1], position1[2]);
        Vector3 triangleVertex2(position2[0], position2[1], position2[2]);
        Vector3 triangleVertex3(position3[0], position3[1], position3[2]);

        if (positionElement->Type == VertexGeometry::Float32_4) {
            triangleVertex1 /= position1[3];
            triangleVertex2 /= position2[3];
            triangleVertex3 /= position3[3];
        }

        triangleVertex1.TransformCoordinate(worldTransform);
        triangleVertex2.TransformCoordinate(worldTransform);
        triangleVertex3.TransformCoordinate(worldTransform);

        Vector3 hitPosition;
        // check intersection
        if (Math3D::TriangleLineIntersection(triangleVertex1,
            triangleVertex2,
            triangleVertex3,
            line,
            hitPosition)) {
                // store smallest distance
                Float temp_distance = line.GetStart().GetDistanceTo(hitPosition);
                if (temp_distance < min_distance) {
                    min_distance = temp_distance;
                    distance = min_distance;
                }
                intersection = true;
                // do not break here as another triangle could have also an intersection but another distance
        }
    }
    return intersection;
}

bool Mesh::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
{
    return Math3D::IsPickIntersectingGeometry(*this, camera, x, y, distance);
}

bool Mesh::Upload()
{
    bool success = true;

    // Upload vertex buffer if not null.
    if (m_vertexBuffer != 0) {
        if( ! m_vertexBuffer->Upload()) {
            FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    // Upload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
    if (!app.PointsToNull()) {
        if (!app->Upload()) {
            FEATSTD_LOG_ERROR("Upload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

bool Mesh::Unload()
{
    bool success = true;

    // Unload vertex buffer if not null.
    if (m_vertexBuffer != 0) {
        if ( ! m_vertexBuffer->Unload()) {
            FEATSTD_LOG_ERROR("Unload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    // Unload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
    if (!app.PointsToNull()) {
        if (!app->Unload()) {
            FEATSTD_LOG_ERROR("Unload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

FEATSTD_RTTI_DEFINITION(Mesh, Renderable)
} // namespace Candera
