//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Material.h"
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/ShaderParamNames.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
using namespace Diagnostics;

FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

FEATSTD_RTTI_DEFINITION(Material, Base)

//private default material is black
static const Material::UniformBlock defaultMaterial = { {0.0F, 0.0F, 0.0F, 1.0F}, // Ambient
                                              {0.0F, 0.0F, 0.0F, 1.0F}, // Diffuse
                                              {0.0F, 0.0F, 0.0F, 1.0F}, // Emissive
                                              {0.0F, 0.0F, 0.0F, 1.0F}, // Specular
                                               0.0F,                    // Power
                                              {0.0F, 0.0F, 0.0F}};      // Padding

Material::Material():
    Base()
{
    m_uniformBuffer.m_data = defaultMaterial;
}

Material::Material(const Material& material):
    Base(material),
    m_uniformBuffer(material.m_uniformBuffer)
{
    //nothing to do
}

Material::SharedPointer Material::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    Material* ptr = FEATSTD_NEW(Material)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Material create failed, out of memory.");
    }
    Material::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

Material::~Material()
{
    //nothing to do
}

Material::SharedPointer Material::Clone() const
{
    return Material::SharedPointer(FEATSTD_NEW(Material)(*this));
}

void Material::SetDiffuse(const Color& color)
{
    const Color::Data& colorData = color.GetData();
    if (colorData != m_uniformBuffer.m_data.diffuse) {
        m_uniformBuffer.m_data.diffuse = colorData;
        m_uniformBuffer.SetDirty();
    }
}

void Material::SetAmbient(const Color& color)
{
    const Color::Data& colorData = color.GetData();
    if (colorData != m_uniformBuffer.m_data.ambient) {
        m_uniformBuffer.m_data.ambient = colorData;
        m_uniformBuffer.SetDirty();
    }
}

void Material::SetEmissive(const Color& color)
{
    const Color::Data& colorData = color.GetData();
    if (colorData != m_uniformBuffer.m_data.emissive) {
        m_uniformBuffer.m_data.emissive = colorData;
        m_uniformBuffer.SetDirty();
    }
}

void Material::SetSpecular(const Color& color)
{
    const Color::Data& colorData = color.GetData();
    if (colorData != m_uniformBuffer.m_data.specular) {
        m_uniformBuffer.m_data.specular = colorData;
        m_uniformBuffer.SetDirty();
    }
}

void Material::SetSpecularPower(Float power)
{
    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (power != m_uniformBuffer.m_data.power) {
        m_uniformBuffer.m_data.power = power;
        m_uniformBuffer.SetDirty();
    }
}

void Material::SetAlphaValue(Float alphaValue)
{
    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (alphaValue != m_uniformBuffer.m_data.diffuse.alpha) {
        m_uniformBuffer.m_data.diffuse.alpha = alphaValue;
        m_uniformBuffer.SetDirty();
    }
}

enum UniformCacheIndices {
    Ambient = 0,
    Diffuse,
    Emissive,
    Specular,
    SpecularPower,
    MaterialBlock,
    UniformCacheIndicesCount
};

static const ShaderParamNames::UniformSemantic UniformCacheParams[UniformCacheIndicesCount] = {
    ShaderParamNames::MaterialAmbient,
    ShaderParamNames::MaterialDiffuse,
    ShaderParamNames::MaterialEmissive,
    ShaderParamNames::MaterialSpecular,
    ShaderParamNames::MaterialSpecularPower,
    ShaderParamNames::MaterialBlock
};

bool Material::Activate(const Shader& shader, Float alphaFactor, Int instanceIndex) const
{
    bool tmpResult = true;
    bool finalResult = true;
    Int uniformLocation;
    const UniformBlock& materialData = GetMaterialData();

    const void* accessorId = UniformCacheParams;
    Shader::UniformCacheHandle uniformCacheHandle = shader.GetUniformCacheHandle(accessorId);
    if (!uniformCacheHandle.IsValid()) {
        if (!shader.RegisterUniformCacheAccessor(accessorId, UniformCacheParams, UniformCacheIndicesCount, uniformCacheHandle)) {
            FEATSTD_LOG_ERROR("Material::Activate failed, RegisterUniformCacheHandle failed.");
            return false;
        }
    }

    // Set ambient.
    if (shader.GetUniformLocation(uniformCacheHandle, Ambient, uniformLocation)) {
        tmpResult = Renderer::SetUniform<Shader::FloatVec4>(uniformLocation, FeatStd::Internal::PointerToPointer<const void*>(&materialData.ambient), instanceIndex);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Material activate failed, setuniform ambient failed.");
            finalResult = false;
        }
    }
    // Set diffuse color. Multiply Material's alpha value with effective alpha value of Node.
    if (shader.GetUniformLocation(uniformCacheHandle, Diffuse, uniformLocation)) {
        Color::Data diffuse = materialData.diffuse;
        const Camera* activeCamera = RenderDevice::GetActiveCamera();
        if (0 != activeCamera) {
            if (activeCamera->IsCameraEffectiveAlphaEnabled()) {
                alphaFactor *= activeCamera->GetEffectiveAlphaValue();
            }
        }
        diffuse.alpha *= alphaFactor;
        tmpResult = Renderer::SetUniform<Shader::FloatVec4>(uniformLocation, FeatStd::Internal::PointerToPointer<const void*>(&diffuse), instanceIndex);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Material activate failed, setuniform diffuse failed.");
            finalResult = false;
        }
    }
    // Set emissive color.
    if (shader.GetUniformLocation(uniformCacheHandle, Emissive, uniformLocation)) {
        tmpResult = Renderer::SetUniform<Shader::FloatVec4>(uniformLocation, FeatStd::Internal::PointerToPointer<const void*>(&materialData.emissive), instanceIndex);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Material activate failed, setuniform emissive failed.");
            finalResult = false;
        }
    }
    // Set specular color.
    if (shader.GetUniformLocation(uniformCacheHandle, Specular, uniformLocation)) {
        tmpResult = Renderer::SetUniform<Shader::FloatVec4>(uniformLocation, FeatStd::Internal::PointerToPointer<const void*>(&materialData.specular), instanceIndex);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Material activate failed, setuniform specular failed.");
            finalResult = false;
        }
    }
    // Set specular power.
    if (shader.GetUniformLocation(uniformCacheHandle, SpecularPower, uniformLocation)) {
        tmpResult = Renderer::SetUniform<Shader::Float>(uniformLocation, FeatStd::Internal::PointerToPointer<const void*>(&materialData.power), instanceIndex);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Material activate failed, setuniform power failed.");
            finalResult = false;
        }
    }
    // Set material block
    if (shader.GetUniformLocation(uniformCacheHandle, MaterialBlock, uniformLocation)) {
        bool isUploaded = m_uniformBuffer.IsUploaded();
        if (!isUploaded) {
            isUploaded = m_uniformBuffer.Upload(DeviceObject::NoHint);
            if (!isUploaded) {
                FEATSTD_LOG_ERROR("MaterialBlock upload failed.");
                finalResult = false;
            }
        }

        if (isUploaded) {
            if (m_uniformBuffer.IsDirty()) {
                if (!Renderer::UpdateUniformBuffer(m_uniformBuffer)) {
                    FEATSTD_LOG_ERROR("MaterialBlock update failed.");
                    finalResult = false;
                }
            }

            if (!Renderer::BindUniformBufferToShader(m_uniformBuffer, shader, uniformLocation)) {
                FEATSTD_LOG_ERROR("MaterialBlock shader binding failed.");
                finalResult = false;
            }
        }
    }

    return finalResult;
}

Material& Material::operator=(const Material& material)
{
    if (this == &material){
        return *this;
    }
    MemoryPlatform::Copy(this, &material,sizeof(Material));
    return *this;
}


SizeType Material::MaterialUniformBuffer::GetSize() const
{
    // Assert that UniformBlock size fits the shader uniform block size
    FEATSTD_COMPILETIME_ASSERT(sizeof(Color::Data) == sizeof(Vector4));
    FEATSTD_COMPILETIME_ASSERT((
        sizeof(Color::Data) /* ambient */ +
        sizeof(Color::Data) /* diffuse */ +
        sizeof(Color::Data) /* emissive */ +
        sizeof(Color::Data) /* specular */ +
        sizeof(Float)       /* power */ +
        sizeof(Float) * 3   /* padding */)
        == sizeof(m_data));

    // Assert that the layout of UniformBlock matches with the layout of the uniform
    // block (ShaderParamNames::MaterialBlock) in the shader.
    FEATSTD_DEBUG_ASSERT(
        (FEATSTD_OFFSET_OF(Material::UniformBlock, ambient) == 0) &&
        (FEATSTD_OFFSET_OF(Material::UniformBlock, diffuse) == 16) &&
        (FEATSTD_OFFSET_OF(Material::UniformBlock, emissive) == 32) &&
        (FEATSTD_OFFSET_OF(Material::UniformBlock, specular) == 48) &&
        (FEATSTD_OFFSET_OF(Material::UniformBlock, power) == 64) &&
        (FEATSTD_OFFSET_OF(Material::UniformBlock, padding) == 68));

    return sizeof(m_data);
}


} // namespace Candera
