//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_LINELIST_H)
#define CANDERA_LINELIST_H

#include <Candera/Engine3D/Core/Renderable.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   The LineList represents a list of lines (segments). It can be used to store a wireframe, a list of connected
            vertices, or just a list of isolated segments.
            A generic LineList instance can be created by the static method LineList::Create. A wireframe LineList can be
            created out of a Mesh by the static helper function Math3D::CreateLineListFromMesh.
 */
class LineList:  public Renderable
{
    FEATSTD_TYPEDEF_BASE(Renderable);

    public:

        /**
         *  The Type of the Line.
         */
        enum LineType
        {
            Lines = 0,      ///< Lines: Vertices as a list of isolated line segments.
            LineStrip = 1,  ///< LineStrip: Vertices as a single polyline.
            LineLoop = 2    ///< LineLoop: Vertices are a single polyline, where last line segment is from Vertex n-1 to Vertex 0.
        };

        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created LineList object.
         */
        static LineList* Create();

        /**
         *  Destructor
         */
        virtual ~LineList() override;

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance or VertexBuffer are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual LineList* Clone() const override;

        /**
         *  Set the VertexBuffer for the LineList object.
         *  @param vertexBuffer Vertex buffer to be set.
         */
        void SetVertexBuffer(MemoryManagement::SharedPointer<VertexBuffer> vertexBuffer);

        /**
         *  Retrieves the VertexBuffer for the LineList object.
         *  @return The  VertexBuffer for the LineList object.
         */
        MemoryManagement::SharedPointer<VertexBuffer> GetVertexBuffer() const { return m_vertexBuffer; }

        /**
         *  Set the width for the LineList object.
         *  @param width for the LineList object.
         */
        void SetWidth(Float width) { m_width = width; }

        /**
         *  Retrieves the width for the LineList object.
         *  @return The width of the LineList object.
         */
        Float GetWidth() const { return m_width; }

        /**
         *  Set the LineType for the LineList object.
         *  @param  lineType    The LineType of this List object -> Lines, LineList or LineLoop.
         */
        void SetLineType(LineType lineType);

        /**
         *  Retrieves the LineType for the LineList object.
         *  @return The LineType of this List object -> Lines, LineList or LineLoop.
         */
        LineType GetLineType() const { return m_type; }

        /**
         *  Tests if geometry of this LineList intersects with the given Line.
         *  @param line               The intersection Line.
         *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this LineList, otherwise false.
         */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

        /**
         *  Upload asset data attached to this LineList to video memory.
         *  @return True if vertex buffer and appearance could be uploaded, false otherwise.
         */
        virtual bool Upload() override;

        /**
         *  Unload asset data attached to this Node from video memory.
         *  @return True if vertex buffer and appearance could be unloaded, false otherwise.
         */
        virtual bool Unload() override;

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        LineList();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(LineList);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Renders this LineList. This method is overridden from Node.
         */
        virtual void Render() override;

        /**
         *  Computes axis-aligned bounding box (AABB) in object coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
         *  Tests if geometry of this LineList intersects with a pick in absolute Viewport coordinate in pixels.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera).
         *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this LineList, otherwise false.
         */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;

    private:
        Float m_width;
        LineType m_type;

        void SetVertexBufferLineType();

        CdaDynamicProperties(Candera::LineList, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_LINELIST_H
