//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_GendericValueLodCriterion_h
#define Candera_GendericValueLodCriterion_h

#include <Candera/Engine3D/Core/LodCriterion.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
//Forward declaration
class LodNode;

/**
 *  @brief The class GenericValueLodCriterion allows to manually set the criterion value for LOD level selection.
 *         The generic value is compared with the LOD's boundaries (see LodNode::LodLevelConfig) according to the LodRenderStrategy attached to LodNode.
 */
class GenericValueLodCriterion: public LodCriterion
{
    FEATSTD_TYPEDEF_BASE(LodCriterion);

    public:

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Constructs a GenericValueLodCriterion object. Per default the criterion value is 0.0F.                        
         */ 
        GenericValueLodCriterion();

        /**
         *  Destructs a GenericValueLodCriterion object.                         
         */
        virtual ~GenericValueLodCriterion() override;

        /**
         *  Specifies a generic value that is used as the criterion value for LOD level selection by LodRenderStrategy.  
         *  @param criterionValue to be set.
         */
        void SetCriterionValue(Float criterionValue) { m_criterionValue = criterionValue; }

        /**
         *  Retrieves the criterion value for the LOD level.
         *  @return The criterion value.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1411, "the hidden virtual GetCriterionValue is unlikely to be mistaken with the getter function (receives a node parameter)")
        Float GetCriterionValue() const { return m_criterionValue; }     

        /**
         *  Overrides GetCriterionValue of class LodCriterion.
         *  Retrieves a criterion value for further LOD level calculation.    
         *  @param lodNode LodNode to which CriterionValue shall be evaluated to.
         *  @return The set criterion value. No computations are done here.
         */
        virtual Float GetCriterionValue(const LodNode& lodNode) const override;
    
    private:
        Float m_criterionValue; // The criterion value is set directly. 
};
 
/** @} */ // end of Core3D
 
}

#endif // Candera_GendericValueLodCriterion_h
