//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_DiscreteLodRenderStrategy_h
#define Candera_DiscreteLodRenderStrategy_h

#include <Candera/Engine3D/Core/LodRenderStrategy.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
/**
 *  @brief The class DiscreteLodRenderStrategy defines an interface to configure the LodNode's visual appearance according to the LodCriterion retrieved.
 *  A hysteresis is enabled, if LOD levels have non-adjacent boundaries set. Hence, small criterion value changes do not toggle LOD level!  
 *  Note: If a single LodNode is rendered by multiple cameras, hysteresis must not be enabled. Hysteresis further assumes that criterion values change 
 *  continuous and avoid leaps between two non-adjacent LOD level boundaries. DiscreteLodRenderStrategy can be shared among different LodNodes. 
 *
 *  Hysteresis example path with LOD level 0: lowerBound = 0, upperBound = 40;  LOD level 1: lowerBound = 60, uppderBound = 100.
 *  Step 1: criterion value: 30 -> LOD level activated: 0.      
 *  Step 2: criterion value: 50 -> LOD level activated: 0. (Hysteresis)
 *  Step 3: criterion value: 60 -> LOD level activated: 1. 
 *  Step 4: criterion value: 50 -> LOD level activated: 1. (Hysteresis)
 *  Step 5: criterion value: 40 -> LOD level activated: 0.   
 *  
 *  Note: If no LodLevel can be applied and no LOD level has been set active before, manually initialize LOD level, for
 *  instance following line would activate LOD level 0: lodNode->GetLodLevel(0).node->SetRenderingEnabled(true); 
 */
class DiscreteLodRenderStrategy: public LodRenderStrategy
{
    FEATSTD_TYPEDEF_BASE(LodRenderStrategy);

    public:

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Constructor
         */
        DiscreteLodRenderStrategy() {}

        /**
         *  Destructor
         */
        virtual ~DiscreteLodRenderStrategy() override {}

        /**
         *  Configures LodNode before rendering. E.g. sets level of detail level to render according to LodCrition value.   
         *  Activates the LOD level that coincides with LOD level's boundaries. If no LodCriterion is set or no LOD level matches the criterion value
         *  given by LodCriterion, the LodNode is not updated.         
         *  @param lodNode LodNode to be updated.
         */
        virtual void Update(LodNode& lodNode) override;

    private: 
        /**
         *  Activates distinctive LOD level and deactivates all others.
         *  @param lodNode LodNode containing LOD levels.
         *  @lodIndex LOD level to be activated. All other levels are deactivated.
         */   
        void ActivateLodLevel(LodNode& lodNode, UInt lodIndex) const;
};
 
/** @} */ // end of Core3D
 
}

#endif // Candera_DiscreteLodRenderStrategy_h
