//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEVICEOBJECT_H)
#define CANDERA_DEVICEOBJECT_H

#include <FeatStd/Container/SingleLinkedList.h>

#include <Candera/EngineBase/Common/CanderaObject.h>

namespace Candera {

    class DeviceObjectListener;
    class ContextResourcePool;

    /** @addtogroup Core3D
     *  @{
     */

    /**
     * @brief DeviceObject is the pure abstract base class for all objects which are able to
     *        upload or unload itself from Video Device Memory. (VRAM)
     *        DeviceObject encapsulates also the upload reference counting mechanism.
     */
    class DeviceObject : public CanderaObject {
            FEATSTD_TYPEDEF_BASE(CanderaObject);

        public:

            /**
            * Loading hint for uploading and unloading.
            */
            enum LoadingHint
            {
                WarnWhileRendering,   ///< Up-/Unloading while rendering throws a warning.
                NoHint,               ///< Up-/Unloading while rendering does not throw a warning.
                Force,                ///< Force an unload from the currently active context resource pool, regardless of the current state of the object, the reference count will be reset.
                ForceAll              ///< Force an unload from all context resource pools, regardless of the current state of the object, the reference count for each resource pool will be reset.
            };

            /**
             *   Constructor.
             */
            DeviceObject();

            /**
             *   Destructor.
             */
            virtual ~DeviceObject() override;

            /**
            *   Uploads the data to Video Memory. (VRAM) The upload functionality is reference counted.
            *   See details in the overridden implementation of UploadInternal.
            *   @param loadingHint How far uploading is restricted.
            *   @return true if object is successfully uploaded, false otherwise.
            */
            bool Upload(LoadingHint loadingHint = WarnWhileRendering);

            /**
            *   Unloads the data to Video Memory. (VRAM) The unload functionality is reference counted.
            *   See details in the overridden implementation of UnloadInternal.
            *   @param loadingHint How far unloading is restricted.
            *   @return true if unload count has been decreased or object is successfully unloaded, false if object cannot be unloaded.
            */
            bool Unload(LoadingHint loadingHint = WarnWhileRendering);

            /**
            *   Check whether the object was uploaded to the current active context resource pool.
            *   @return true if the object was successfully uploaded to VRAM, false otherwise.
            */
            bool IsUploaded() const;

            /**
             *  Sets the following behavior: After a successful upload, the asset data (often residing in system memory) is disposed.
             *  By default, this behavior is NOT enabled.
             *  Use this to free system memory if you are sure that only the representation uploaded to VRAM is needed any more.
             *  @param disposeAfterUpload Enables(true)/Disables(false) the behavior described above.
             */
            void SetDisposedAfterUpload(bool disposeAfterUpload) {
                m_isDisposedAfterUpload = disposeAfterUpload;
            }

            /**
             *  Retrieves whether the asset data is disposed after upload or not.
             *  @return Boolean whether the asset data is disposed after upload or not.
             */
            bool IsDisposedAfterUpload() const {
                return m_isDisposedAfterUpload;
            }

            /**
             * @brief Register an action listener to this device object.
             *
             * The listener will be notified when the DeviceObject is uploaded,
             *  unloaded or destroyed.
             *
             * @param deviceObjectListener Listener that will receive
             *  action notifications.
             * @return true if listener was successful registered and will
             *  receive notifications.
             * @return false if listener could not be registered.
             *
             * @see DeviceObjectListener
             */
            bool AddListener(DeviceObjectListener* deviceObjectListener) {
                return m_listenerList.Append(deviceObjectListener);
            }

            /**
             * @brief Unregister an action listener of this device object.
             *
             * The listener will not receive action notifications anymore.
             *
             * @param deviceObjectListener Listener that will not receive
             *  notifications anymore.
             * @return true if listener was successful unregistered.
             * @return false if listener could not be unregistered.
             *
             * @see DeviceObjectListener
             */
            bool RemoveListener(DeviceObjectListener* deviceObjectListener) {
                return m_listenerList.Remove(deviceObjectListener);
            }

            FEATSTD_RTTI_DECLARATION();
        protected:
            /**
             *  Implementation for child classes where the actual upload happens.
             *  @return true if the object was successfully uploaded to VRAM, false otherwise.
             */
            virtual bool UploadInternal(LoadingHint loadingHint) = 0; //TODOX @return

            /**
             *  Implementation for child classes where the actual unload happens.
             *  @return true if the object was successfully unloaded from VRAM, false otherwise.
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) = 0;

            /**
             *  Implementation for child classes where the data are disposed after upload if DisposeAfterUpload is set.
             */
            virtual void DisposeInternal() = 0;

        private:
            typedef FeatStd::Internal::SingleLinkedList<DeviceObjectListener*> ListenerList;

            // Reference counters for VRAM uploads exist for each ContextResourcePool.
            Int32 m_uploadRefCounts[CANDERA_MAX_CONTEXT_COUNT];
            bool m_isDisposedAfterUpload;
            ListenerList m_listenerList;

            void NotifyListenersOnPreUpload();
            void NotifyListenersOnPostUpload();
            void NotifyListenersOnPreUnload();
            void NotifyListenersOnPostUnload();
            void NotifyListenersOnDestruction();

            bool DoUpload(const ContextResourcePool& contextResourcePool);
            bool DoUnload(const ContextResourcePool& contextResourcePool);

    };

    /** @} */ // end of CommonBase
} // namespace Candera

#endif    // CANDERA_DEVICEOBJECT_H
