//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DeviceObject.h"

#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

#include <Candera/Engine3D/Core/DeviceObjectListener.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(DeviceObject, CanderaObject)

    DeviceObject::DeviceObject() :
        Base(),
        m_isDisposedAfterUpload(false)
    {
        MemoryPlatform::Set(&m_uploadRefCounts[0], 0U, sizeof(m_uploadRefCounts));
    }

    DeviceObject::~DeviceObject()
    {
        NotifyListenersOnDestruction();
    }

    bool DeviceObject::Upload(LoadingHint loadingHint)
    {
        Int32& refCount = m_uploadRefCounts[ContextResourcePool::GetActive().GetIndex()];

        if (refCount > 0) {
            ++refCount;
            return true; // object is already uploaded
        }

        NotifyListenersOnPreUpload();
        bool result = UploadInternal(loadingHint);
        refCount = result ? 1 : 0;
        NotifyListenersOnPostUpload();

        if (result && m_isDisposedAfterUpload) {
            DisposeInternal();
        }

        return result;
    }

    bool DeviceObject::Unload(LoadingHint loadingHint)
    {
        if (ForceAll == loadingHint) {
            // Unload device object in current context resource pool active.
            bool result = Unload(Force);

            // Save current active render target
            RenderTarget* activeRenderTarget = RenderTarget::GetActiveRenderTarget();

            // Unload device object in all other context resource pools.
            ContextResourcePool* sentinel = &ContextResourcePool::GetActive();
            ContextResourcePool* pool = &sentinel->GetNext();
            bool poolChanged = false;
            while (pool != sentinel) {
                Int32 refCount = m_uploadRefCounts[pool->GetIndex()];
                if ((refCount > 0) && pool->Activate()) {
                    poolChanged = true;
                    result = Unload(Force) && result;
                }
                pool = &pool->GetNext();
            }
            if (poolChanged) {
                if (0 != activeRenderTarget) {
                    result = result && activeRenderTarget->Activate();
                }
                else {
                    result = result && sentinel->Activate();
                }
            }

            return result;
        }

        const ContextResourcePool& contextResourcePool = ContextResourcePool::GetActive();
        Int32& refCount = m_uploadRefCounts[contextResourcePool.GetIndex()];

        if (Force == loadingHint) {

            if (refCount <= 0) {
                return true;
            }

            NotifyListenersOnPreUnload();
            bool result = UnloadInternal(NoHint);
            refCount = result ? 0 : refCount;
            NotifyListenersOnPostUnload();

            return result;
        }
        else {

            if (refCount > 1) {
                --refCount;
                return true; // no need to unload yet
            }
            else if (refCount <= 0) {
                return true; 
            }
            else {
                NotifyListenersOnPreUnload();
                bool result = UnloadInternal(loadingHint);
                refCount = result ? 0 : 1;
                NotifyListenersOnPostUnload(); 
                return result;
            }
        }
    }

    bool DeviceObject::IsUploaded() const
    {
        return m_uploadRefCounts[ContextResourcePool::GetActive().GetIndex()] > 0;
    }

    void DeviceObject::NotifyListenersOnPreUpload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPreUpload(*this, ContextResourcePool::GetActive());
        }
    }

    void DeviceObject::NotifyListenersOnPostUpload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPostUpload(*this, ContextResourcePool::GetActive());
        }
    }

    void DeviceObject::NotifyListenersOnPreUnload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPreUnload(*this, ContextResourcePool::GetActive());
        }
    }

    void DeviceObject::NotifyListenersOnPostUnload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPostUnload(*this, ContextResourcePool::GetActive());
        }
    }

    void DeviceObject::NotifyListenersOnDestruction()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnDestruction(*this);
        }
    }

} // namespace Candera
