//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CUBEMAP_TEXTURE_IMAGE_H)
#define CANDERA_CUBEMAP_TEXTURE_IMAGE_H

#include <Candera/Engine3D/Core/TextureImage.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera {
    /** @addtogroup Core3D
     *  @{
     */

    class Texture;

    /**
    * @brief  CubeMapTextureImage encapsulates a VRAM handle which is retrieved when uploading a cube mapping
    *         texture to VRAM. In contrary to BitmapTextureImage six bitmaps are uploaded instead of one. They are
    *         mapped to the sides of a cube. CubeMapTextureImage provides an interface to set the bitmaps for each
    *         face separately. Therefore the TargetFace enumerator is used in the functions GetBitmapFace and SetBitmapFace.
    *         The following sketch illustrates the mapping of the CubeMapFaces:
    *
    *                                   ___________________________
    *                                   |--->s                    |
    *                                   ||                        |
    *                                   |v                        |
    *                                   |t                        |
    *                                   |  CubeMapPositiveYFace   |
    *                                   |                         |
    *                                   |                         |
    *                                   |Top                      |
    *         __________________________|_________________________|____________________________________________________
    *         |--->s                    |--->s                    |--->s                    |--->s                    |
    *         ||                        ||                        ||                        ||                        |
    *         |v                        |v                        |v                        |v                        |
    *         |t                        |t                        |t                        |t                        |
    *         |  CubeMapNegativeXFace   |  CubeMapPositiveZFace   |  CubeMapPositiveXFace   |  CubeMapNegativeZFace   |
    *         |                         |                         |                         |                         |
    *         |                         |                         |                         |                         |
    *         |Left                     |Front                    |Right                    |Back                     |
    *         |_________________________|_________________________|_________________________|_________________________|
    *                                   |--->s                    |
    *                                   ||                        |
    *                                   |v                        |
    *                                   |t                        |
    *                                   |  CubeMapNegativeYFace   |
    *                                   |                         |
    *                                   |                         |
    *                                   |Down                     |
    *                                   |_________________________|
    *
    *         In contrast to normal textures the origin of the passed pictures has to be in the top left corner of the image.
    *         This means that the actual Bitmap data has to be passed flipped around the Y-axis, as Candera expects
    *         Bitmaps to be in the format for normal 2D textures.
    *
    *         The fact that the origin of those textures has to be on the top left corner can also be derived
    *         from the addressing algorithm used by GLSL function textureCube. Basically the addressing consists
    *         of a three-component vector that can be seen as coming from the origin of the cube. Intersecting
    *         it with the cube reveals a texel of one of the faces. This is done as follows:
    *         - The highest absolute component of the  addressing vector r and its sign determines the face, see following table.
    *         - Therefore values sc, tc and ma get assigned according values. These can also be seen in the following table.
    *
    *         | Major Axis Direction | Target               |  sc  |  tc  |  ma  |
    *         --------------------------------------------------------------------
    *         | +rx                  | CubeMapPositiveXFace | -rz  | -ry  |  rx  |
    *         | -rx                  | CubeMapNegativeXFace |  rz  | -ry  |  rx  |
    *         | +ry                  | CubeMapPositiveYFace |  rx  |  rz  |  ry  |
    *         | -ry                  | CubeMapNegativeYFace |  rx  | -rz  |  ry  |
    *         | +rz                  | CubeMapPositiveZFace |  rx  | -ry  |  rz  |
    *         | -rz                  | CubeMapNegativeZFace | -rx  | -ry  |  rz  |
    *
    *         From this table the image coordinates get calculated by the following equations:
    *
    *         s = (1/2) * ((sc/|ma|) + 1)
    *         t = (1/2) * ((tc/|ma|) + 1)
    *
    *         The behavior, if the highest component of r exists two or three times, is defined by OpenGL implementation. The need of pictures being
    *         flipped is illustrated by an example. If a pixel on the upper edge of the front face has to be selected, then for example r = (0, 0.99, 1).
    *         |rz| is one of the highest component, it's sign is positive, therefore CubeMapPositiveZFace gets selected. Then sc = 0, st = -0.99 and ma = 1.
    *         => s = (1/2) * ((0.5/1) + 1) = 0.5 and t = (1/2) * ((-0.99/1) + 1) = 0.005. As t = 0.005 is almost at the upper edge of the cube
    *         its shown that this is supposed to be almost the origin of the t-axis. The algorithm is also specified in the OpenGL specification.
    *
    *         CubeMapTextureImage implements a reference counting mechanism for handling multiple upload and unloads.
    *         Multiple textures can share a CubeMapTextureImage.
    *
    *         CubeMapTextureImages can only be uploaded, if images are set for all faces and all of them
    *         have equal and square dimensions. MipMapping can only be applied if all images have exactly the
    *         same dimensions and if the dimensions are of power of two.
    *         If mipmapping is enabled and custom mipmap chains are used, they have to be fully provided for any of the six provided pictures,
    *         otherwise mipmap generation will be triggered for the whole CubeMap texture.
    *
    */
    class CubeMapTextureImage : public TextureImage, public FeatStd::Diagnostics::Measurable {
        friend class RenderDevice;

            FEATSTD_TYPEDEF_BASE(TextureImage);

        public:
            /**
             * This enumerator describes which face of the cubemap's cube is affected by an operation.
             */
            enum TargetFace {
                CubeMapPositiveXFace = 0,   ///< Right face.
                CubeMapNegativeXFace = 1,   ///< Left face.
                CubeMapPositiveYFace = 2,   ///< Up face.
                CubeMapNegativeYFace = 3,   ///< Down face.
                CubeMapPositiveZFace = 4,   ///< Front face.
                CubeMapNegativeZFace = 5,   ///< Back face.
                TargetFaceCount = 6         ///< Number of target faces.
            };

            /**
             *  Creates an instance of this class.
             *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
             */
            static MemoryManagement::SharedPointer<CubeMapTextureImage> Create();

            /**
             *  Destructor
             */
            virtual ~CubeMapTextureImage() override;

            /**
             *  Updates data within an already uploaded CubeMapTextureImage directly in VRAM.
             *  The source data is taken from the attached bitmaps for each face.
             *  All mipmap levels are updated (if any).
             *  Note: Updating can only be done if all face bitmaps are mutable.
             *  @return True if updating this CubeMapTextureImage and setting every texture sub image succeeded. False otherwise.
             */
            bool Update() const;

            /**
             *  Updates data within an already uploaded CubeMapTextureImage face directly in VRAM.
             *  @param face        The CubeMap face to upload.
             *  @param xOffset     The x-Offset inside the Texture.
             *  @param yOffset     The y-Offset inside the Texture.
             *  @param width       The width to update. The width relates also to the width of the source data.
             *  @param height      The height to update. The height relates also to the height of the source data.
             *  @param data        The source data to update. The source data must be in continuous memory.
             *  @param level       The mipmap level to update (in case predefined mipmapping is used), default is first
             *  @param unit        The texture unit to update.
             *  @param compressedSize The size of the data array in bytes, needed for updating compressed images.
             *                     Note: Function will fail if image is compressed and size is wrong.
             *  @return            True if updating every texture sub image of the specified face succeeded. False if updating failed, texture is not uploaded,
             *                     no data is passed. Also fails when mipmapping is disabled and a level > 0 is passed.
             */
            bool Update(TargetFace face, Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt level = 0, UInt unit = 0, UInt compressedSize = 0) const;

            /**
             *  Copy content from the current frame buffer to the CubeMapTextureImage face in VRAM.
             *  @param face               The CubeMap face to copy data to.
             *  @param xTex               The x-Offset inside the Texture.
             *  @param yTex               The y-Offset inside the Texture.
             *  @param xFramebuffer       The x-Offset inside the frame buffer.
             *  @param yFramebuffer       The y-Offset inside the frame buffer.
             *  @param width              The width to copy.
             *  @param height             The height to copy.
             *  @return                   True if current frame buffer was copied to the TextureImage in VRAM. False otherwise.
             */
            bool CopyFromFramebuffer(TargetFace face, Int32 xTex, Int32 yTex, Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height) const;

            /**
             *  Sets the Bitmap for a certain CubeMap face and the corresponding Disposer for the Bitmap.
             *  Call this function before Upload. All bitmap faces must be set.
             *  It also also possible to pass a list of bitmaps for predefined mip-mapping.
             *  @param face               CubeMap face to which the set bitmap is mapped.
             *  @param bitmap             The Bitmap or list of Bitmaps which is uploaded by the CubeMapTextureImage for the specified face.
             *                            If a list of Bitmaps is passed then pass here the head of the list.
             *  @return                   False if CubeMapTextureImage is already uploaded, modification of Bitmaps after uploading is not allowed.
             *                            True if Bitmap is set.
             */
            bool SetBitmapFace(TargetFace face, const Bitmap::SharedPointer& bitmap);

            /**
             *  Retrieves the bitmap mapped to the specified CubeMap face used by this Texture object.
             *  @param face The face to retrieve the mapped bitmap from.
             *  @return Pointer to bitmap for the specified CubeMap face.
             */
            const Bitmap::SharedPointer& GetBitmapFace(TargetFace face) const;

            /**
             *  Retrieves the bitmap mapped to the specified CubeMap face used by this Texture object if it is mutable.
             *  @param face The face to retrieve the mapped bitmap from.
             *  @return  The Bitmap for the specified CubeMap face if it is mutable (Bitmap::IsMutable() == true), otherwise NULL.
             */
            Bitmap::SharedPointer GetMutableBitmapFace(TargetFace face) const;

            /**
             *  Retrieve texture handle of graphic device, associated with the active resource pool.
             *  @return the texture handle of graphic device associated with the active pool.
             *  @return Texture handle of graphic device.
             */
            virtual Handle GetVideoMemoryHandle() const override;

            /**
             *  Defines if MipMap-chain shall be attached to TextureImage.
             *  If MipMapping is enabled without passing a complete Bitmap chain (See Bitmap::SetNext) for all faces, then
             *  the MipMap-chain is generated by driver, automatically.
             *  Call this function before Texture::Upload() to achieve desired effect, see return param for further details.
             *  @param   enableMipMapping defines if MipMap chain will be generated or not if Texture will be uploaded.
             *  @return  true, if @param enableMipMapping is applied when the Texture will be uploaded.
             *  @return  false, if @param enableMipMapping is ignored, as Texture has already been uploaded.
             */
            bool SetMipMappingEnabled(bool enableMipMapping);

            /**
             *  Retrieve if MipMapping is enabled or not
             *  @return   true, if mipmapping is enabled and mipmap filters are evaluated.
             *  @return   false, if mipmapping is disabled and mipmap filters are ignored.
             */
            virtual bool IsMipMappingEnabled() const override {
                return m_isMipMappingEnabled;
            }

            /**
             *  Retrieves ImageSource3D of this CubeMapTextureImage.
             *  @return ImageSource3D of this BitmapTextureImage.
             */
            virtual ImageSource3D* ToImageSource3D() override;

            /**
             *  Retrieves the current texture target.
             *  @return The current texture target.
             */
            virtual TextureTargetType GetTextureTargetType() const override{
                return TextureImage::TextureCubeMap;
            }

            /**
             *  Overridden function of Measurable.
             *  @return estimate of the video memory used by all six bitmaps.
             */
            virtual UInt GetSize() const override;

            FEATSTD_RTTI_DECLARATION();

        protected:
            /**
             *  overridden from DeviceObject
             *  @return True if uploading of TextureImage succeeded. False otherwise.
             */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
             *  overridden from DeviceObject
             *  @return True if unloading of TextureImage succeeded. False otherwise.
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override;

        private:
            /**
             *  Wrapper class which is exposed by ToImageSource3D function.
             */
            class ImageSource3DInstance : public ImageSource3D {
                    friend class CubeMapTextureImage;

                public:
                    ImageSource3DInstance() : m_cubeMapTextureImage(0) {
                        SetTextureTarget(TextureImage::TextureCubeMap);
                    }
                    ~ImageSource3DInstance() {
                        m_cubeMapTextureImage = 0;
                    }

                    virtual Handle GetVideoMemoryHandle() const override {
                        return (m_cubeMapTextureImage != 0) ? m_cubeMapTextureImage->GetVideoMemoryHandle() : 0;
                    }
                    virtual bool IsMipMappingEnabled() const override {
                        return (m_cubeMapTextureImage != 0) ? m_cubeMapTextureImage->IsMipMappingEnabled() : false;
                    }
                    virtual Int GetHeight() const override {
                        return (m_cubeMapTextureImage != 0) ? static_cast<Int>(m_cubeMapTextureImage->GetBitmapFace(m_cubeMapTextureImage->m_lastFaceSet)->GetHeight()) : 0;
                    }
                    virtual Int GetWidth() const override {
                        return (m_cubeMapTextureImage != 0) ? static_cast<Int>(m_cubeMapTextureImage->GetBitmapFace(m_cubeMapTextureImage->m_lastFaceSet)->GetWidth()) : 0;
                    }
                    virtual void Sync() override {}
                    virtual void WaitSync() override {}

                private:
                    const CubeMapTextureImage* m_cubeMapTextureImage;

                    void SetCubeMapTextureImage(const CubeMapTextureImage* image) {
                        m_cubeMapTextureImage = image;
                    }
            };

            bool m_isMipMappingEnabled; // Defines if mipmapping is enabled or not. Only the value that was set before upload is considered.
            Bitmap::SharedPointer m_faces[TargetFaceCount]; // Bitmap that stores the image data for this TextureImage object.
            Handle m_videoMemoryHandle[CANDERA_MAX_CONTEXT_COUNT]; // Texture handle of graphics device.
            // ImageSource3DInstance is exposed by ToImageSource3D function.
            ImageSource3DInstance m_imageSource3DInstance;

            TargetFace m_lastFaceSet;

            void DisposeFace(TargetFace face) const;

            /*
             * Checks if all bitmaps are quadratic and of same size.
             * @return True if all bitmaps are set and quadratic.
             */
            bool IsCubeMapValid() const;
            bool IsCubeMapFaceValid(const TargetFace face) const;

            // Private because only ::Create() should be used to create an instance of this class.
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::CubeMapTextureImage::CubeMapTextureImage, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            CubeMapTextureImage();
            CubeMapTextureImage(const CubeMapTextureImage& material);
            CubeMapTextureImage& operator = (const CubeMapTextureImage& material);

            /**
             *  Sets the texture handle of graphic device.
             *  @param handle the texture handle
             */
            void SetVideoMemoryHandle(Handle handle);
    };
    /** @} */ // end of Core3D
} // namespace Candera
#endif  // CANDERA_CUBEMAP_TEXTURE_IMAGE_H
