//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CubeMapTextureImage.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <Candera/Engine3D/Core/Renderer.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    CubeMapTextureImage::CubeMapTextureImage() :
        Base(),
        m_isMipMappingEnabled(false),
        m_lastFaceSet(CubeMapPositiveXFace)
    {
        MemoryPlatform::Set(m_videoMemoryHandle, 0, sizeof(m_videoMemoryHandle));
        m_imageSource3DInstance.SetCubeMapTextureImage(this);

        for (UInt i = 0; i < TargetFaceCount; i++ ) {
            m_faces[i] = Bitmap::SharedPointer();
        }
    }

    SharedPointer<CubeMapTextureImage> CubeMapTextureImage::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        CubeMapTextureImage* ptr = FEATSTD_NEW(CubeMapTextureImage)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("Creation of CubeMapTextureImage failed as out of memory.");
        }
        SharedPointer<CubeMapTextureImage> sharedPointer(ptr);
        return sharedPointer;
    }

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1579, "Bitmap data is disposed by the disposer; the disposer has external lifetime")
    CubeMapTextureImage::~CubeMapTextureImage()
    {
        static_cast<void>(Unload(ForceAll));
        CubeMapTextureImage::DisposeInternal();
    }

    bool CubeMapTextureImage::SetMipMappingEnabled(bool enableMipMapping)
    {
        bool isNotUploaded = !IsUploaded();
        if (isNotUploaded) {
            m_isMipMappingEnabled = enableMipMapping;
        }
        return isNotUploaded;
    }

    bool CubeMapTextureImage::Update() const
    {
        bool isSuccessful = IsUploaded() && IsCubeMapValid();
        if (isSuccessful) {
            for (UInt i = 0; i < TargetFaceCount; i++) {
                Bitmap* current = m_faces[i].GetPointerToSharedInstance();
                isSuccessful = isSuccessful && (current != 0);
                if (isSuccessful) {
                    UInt level = 0;
                    while (current != 0) {
                        Bitmap* next = current->GetNext().GetPointerToSharedInstance();
                        Bitmap::PixelsResource pixelsResource(current->GetPixelsResourceHandle());
                        const UInt8* data = pixelsResource.GetData();
                        isSuccessful = isSuccessful && (data != 0) && RenderDevice::SetTextureSubimage(*this, static_cast<TargetFace>(i),
                                       level, 0, 0, current->GetWidth(), current->GetHeight(), data);
                        ++level;
                        current = next;
                    }
                }
            }
        }
        return isSuccessful;
    }

    bool CubeMapTextureImage::Update(TargetFace face, Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt level, UInt unit, UInt compressedSize) const
    {
        bool result = false;
        if ((!m_isMipMappingEnabled) && (level > 0)) {
            return result;
        }
        if (IsUploaded() && (data != 0)) {
            result = RenderDevice::SetTextureSubimage(*this, face, level, xOffset, yOffset, width, height, data, unit, compressedSize);
        }
        return result;
    }

    bool CubeMapTextureImage::CopyFromFramebuffer(TargetFace face, Int32 xTex, Int32 yTex, Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height) const
    {
        return RenderDevice::CopyFromFramebufferToCubeMapTextureFace(*this, face, xTex, yTex, xFramebuffer, yFramebuffer, width, height);
    }

    bool CubeMapTextureImage::SetBitmapFace(TargetFace face, const Bitmap::SharedPointer& bitmap)
    {
        if(IsUploaded()) {
            return false; // bitmap data is already uploaded; modification after upload not allowed
        }
        DisposeFace(face);
        m_faces[static_cast<UInt>(face)] = bitmap;
        m_lastFaceSet = face;
        return true;
    }

    const Bitmap::SharedPointer& CubeMapTextureImage::GetBitmapFace(TargetFace face) const
    {
        return m_faces[static_cast<UInt>(face)];
    }

    bool CubeMapTextureImage::UploadInternal(LoadingHint loadingHint)
    {
        bool uploadSuccessful = false;
        if (IsCubeMapValid()) {
            uploadSuccessful = Renderer::UploadCubeMapTextureImage(*this, 0, loadingHint);
        }
        return uploadSuccessful;
    }

    bool CubeMapTextureImage::UnloadInternal(LoadingHint loadingHint)
    {
        return Renderer::UnloadCubeMapTextureImage(*this, loadingHint);
    }

    void CubeMapTextureImage::DisposeInternal()
    {
        for (UInt i = 0; i < TargetFaceCount; i++) {
            DisposeFace(static_cast<TargetFace>(i));
        }
    }

    Bitmap::SharedPointer CubeMapTextureImage::GetMutableBitmapFace(TargetFace face) const
    {
        const Bitmap::SharedPointer& bitmap = m_faces[static_cast<UInt>(face)];
        return ((bitmap != 0) && (bitmap->GetPixelsResourceHandle().m_isMutable == 1)) ? bitmap : Bitmap::SharedPointer();
    }

    ImageSource3D* CubeMapTextureImage::ToImageSource3D()
    {
        return &m_imageSource3DInstance;
    }

    UInt CubeMapTextureImage::GetSize() const
    {
        UInt32 textureImageSize = 0;
        for (UInt i = 0; i < TargetFaceCount; i++) {
            Bitmap* bitmap = m_faces[i].GetPointerToSharedInstance();
            if (bitmap != 0) {

                UInt32 imageSize = RenderDevice::GetSize(bitmap, GetTextureMemoryPool());

                if (IsMipMappingEnabled()) {
                    if (bitmap->HasNext() != 0) {
                        SharedPointer<Bitmap> nextBitmap = bitmap->GetNext();
                        while (nextBitmap != 0) {
                            imageSize += RenderDevice::GetSize(nextBitmap.GetPointerToSharedInstance(), GetTextureMemoryPool());
                            nextBitmap = nextBitmap->GetNext();
                        }
                    }
                    else {
                        if (!bitmap->IsCompressed()) {
                            // A mipmap chain must conclude with a 1x1 sized level in OpenGL ES and has max. 1/3 more memory consumption than the greatest image.
                            imageSize = ((4 * imageSize) - 1) / 3;
                        }
                    }
                }
                textureImageSize += imageSize;
            }
        }

        return textureImageSize;
    }

void CubeMapTextureImage::DisposeFace(TargetFace face) const
{
    Bitmap::SharedPointer bitmap = m_faces[static_cast<UInt>(face)];

    while (!bitmap.PointsToNull()) {
        const Bitmap::SharedPointer& next = bitmap->GetNext();
        bitmap = next;
    }

    bitmap = Bitmap::SharedPointer();
}

bool CubeMapTextureImage::IsCubeMapValid() const
{
    bool result = true;

    UInt32 width = 0;
    for (UInt i = 0; i < TargetFaceCount; i++){
        result = result && IsCubeMapFaceValid(static_cast<TargetFace>(i));
        if (result) {
            if (width == 0 ) {
                width = m_faces[i]->GetWidth(); //It is sufficient to check width only, as equality of sides is already checked in IsCubeMapFaceValid(const TargetFace face).
            }
            else {
                if (width != m_faces[i]->GetWidth() ) {
                    result = false;
                }
            }
        }
    }

    return result;
}

bool CubeMapTextureImage::IsCubeMapFaceValid(const TargetFace face) const
{
    const Bitmap* faceBmp = m_faces[static_cast<UInt>(face)].GetPointerToSharedInstance();
    return (faceBmp != 0) ? (faceBmp->GetHeight() == faceBmp->GetWidth()) : false;
}

Handle CubeMapTextureImage::GetVideoMemoryHandle() const
{
    return m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
}

void CubeMapTextureImage::SetVideoMemoryHandle(Handle handle)
{
    m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
}

FEATSTD_RTTI_DEFINITION(CubeMapTextureImage, Base)
} // namespace Candera
