//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(ConstantAttributeNode_h)
#define ConstantAttributeNode_h

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/System/Container/LinkedList.h>
#include <CanderaPlatform/OS/StringPlatform.h>

namespace Candera {
    namespace Internal {

        /**
         *   @brief  A ConstantAttributeNode stores shader parameters for constant attributes.
         */
        class ConstantAttributeNode
        {
            public:
                /**
                 *  Constructs a ConstantAttributeNode object.
                 *  @param name Specifies the name of the constant attribute to be checked.
                 *              A check is done if name is part of ShaderParamNames. If yes,
                 *              the semantic field will be set to it.
                 *              This further indicates that the attribute location in the shader
                 *              can be taken from the Shader cache.
                 *  @param type defines the data type of the constant attribute data given.
                 *  @param data defines the pointer to the location where the data is stored.    
                 */
                ConstantAttributeNode(ShaderParamNames::AttributeSemantic semantic, Shader::ConstantAttributeType type, void* data) :
                  m_semantic(semantic), 
                  m_type(type),           
                  m_data(data)          
                {

                }   
        
                /**
                 *  Destroys this ConstantAttributeNode object.
                 */
                ~ConstantAttributeNode() { m_data = 0; }

                /**
                 *  Sets the pointer to attribute data location.
                 *  param data Pointer to attribute data location to be set.
                 *  @param data Data pointer to set.
                 */
                void SetData(void* data) { m_data = data; }

                /**  
                 *  Retrieves pointer to attribute data location.
                 *  @return Pointer to attribute data location.
                 */
                const void* GetData() const { return m_data; } 

                /**  
                 *  Retrieves pointer to attribute data location.
                 *  @return Pointer to attribute data location.
                 */
                void* GetData() { return m_data; }

                /**
                 *  Retrieves semantic of constant attribute.
                 *  @return Semantic of the constant attribute, if value is AttributeSemanticCount then it's not set.
                 */
                ShaderParamNames::AttributeSemantic GetSemantic() const { return m_semantic; }

                /**
                 *  Retrieves type of constant attribute.
                 *  @return Type of the constant attribute.
                 */
                Shader::ConstantAttributeType GetType() const { return m_type; }

                /**
                 *  Intrusive ConstantAttributeNode pointer.
                 */
                Internal::LinkedListNode<ConstantAttributeNode> ListNode;

            private:
                //Default constructor is not defined.
                ConstantAttributeNode();

                ShaderParamNames::AttributeSemantic m_semantic; // Semantic of constant attribute.
                Shader::ConstantAttributeType m_type;           // Type of constant attribute.
                void* m_data;                                   // Attribute data location.        

                //forbid assignment by commenting it out
                ConstantAttributeNode& operator=(const ConstantAttributeNode&);
        };

    }

}

#endif // ConstantAttributeNode_h

