//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if  !defined(CANDERA_COMPOSITEGROUP_H)
#define CANDERA_COMPOSITEGROUP_H

#include <FeatStd/Container/Vector.h>
#include <Candera/System/Container/ForwardIterator.h>
#include <Candera/System/MetaInfo/PropertyMetaInfo.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/EngineBase/Common/StringIdentifier.h>
#include <Candera/Engine3D/Core/Group.h>

#if defined(CANDERA_3D_CANVAS_ENABLED)
#include <Candera/Engine3D/Canvas/CanvasTransformable.h>
#include <Candera/Engine3D/Canvas/CanvasGroup.h>
#endif

namespace Candera {

    class WidgetBase;

    namespace Internal {
        class CompositeGroupFunctions;
    }

    /** @addtogroup Core3D
     *  @{
     */

    /**
     * @brief CompositeGroup is a Group that allows direct access to a number of
     *  descendants, called 'anchor points'.
     *
     * CompositeGroups are useful at design time to easily access, attach,
     *  detach or transform subtrees attached to the anchor points.
     */
#if defined(CANDERA_3D_CANVAS_ENABLED)
    class CompositeGroup: public Group, CanvasInterface
#else
    class CompositeGroup: public Group
#endif
    {
    public:
        /**
         * @brief Anchor point definition.
         *
         * An anchor point is a descendant of a CompositeGroup to which anchors
         *  (nodes) can be attached.
         */
        struct AnchorPoint {
            Id m_id;
            const Char* m_name;
            Node* m_node;
        };

        enum CompositeGroupMode {
            GroupMode,
            CanvasGroupMode
        };

    private:
        FEATSTD_TYPEDEF_BASE(Group);
        typedef FeatStd::Internal::Vector<WidgetBase*> WidgetList;
        typedef FeatStd::Internal::Vector<AnchorPoint> AnchorPointList;
        typedef FeatStd::Internal::Vector<Animation::AnimationPlayerBase::SharedPointer> AnimationList;
    public:
        FEATSTD_RTTI_DECLARATION();
        typedef ForwardIterator<AnimationList::ConstIterator> AnimationIterator;
        typedef ForwardIterator<AnchorPointList::ConstIterator> AnchorPointIterator;
        typedef ForwardIterator<WidgetList::ConstIterator> WidgetIterator;

        /**
         * Destructor.
         */
        virtual ~CompositeGroup() override {}

        // Overrides Node::Clone.
        virtual CompositeGroup* Clone() const override;

        /**
         * Retrieve a ForwardIterator that points to the first AnchorPoint.
         *
         * @return First AnchorPoint ForwardIterator.
         */
        AnchorPointIterator GetAnchorPointIterator() const;

        /**
         * Retrieve an anchor point node by name.
         *
         * @param name Name of the anchor point.
         * @return anchor point node if it exists.
         */
        Node* GetAnchorPointNodeByName(const Char* name) const;

        /**
         * Retrieve an anchor point node by id.
         *
         * @param id Id of the anchor point.
         * @return anchor point node if it exists.
         */
        Node* GetAnchorPointNode(Id id) const;

        /**
         * Retrieve a ForwardIterator that points to the first animation.
         *
         * @return First animation ForwardIterator.
         */
        AnimationIterator GetAnimationIterator() const;

        /**
         * Retrieve an animation by name.
         *
         * @param name Name of the animation.
         * @return animation with the given name.
         */
        Animation::AnimationPlayerBase::SharedPointer GetAnimationByName(const Char*) const;

        /**
         * Retrieve an animation by id.
         *
         * @param id Id of the animation.
         * @return animation with the given id.
         */
        Animation::AnimationPlayerBase::SharedPointer GetAnimation(Id id) const;

        /**
         * Retrieve a ForwardIterator that points to the first widget.
         *
         * @return First widget ForwardIterator.
         */
        WidgetIterator GetWidgetIterator() const;

        /**
         * Retrieve an widget by name.
         *
         * @param name Name of the widget.
         * @return widget with the given name.
         */
        WidgetBase* GetWidgetByName(const Char* name) const;

        /**
         * Retrieve an widget by id.
         *
         * @param id Id of the widget.
         * @return widget with the given id.
         */
        WidgetBase* GetWidget(Id id) const;

        /**
         * Retrieve the string identifier.
         *
         * @param id Id of the widget.
         * @return widget with the given id.
         */
        const StringIdentifier* GetStringId() const { return &m_stringIdentifier; }

        /**
         * Gets whether this CompositeGroup serves as Group or CanvasGroup.
         * @return Whether this CompositeGroup serves as Group or CanvasGroup.
         */
        CompositeGroupMode GetCompositeGroupMode() const { return m_compositeGroupMode; }


#if defined(CANDERA_3D_CANVAS_ENABLED)
        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center of a bitmap,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector2& basePoint) const override;


        /**
        *  Called when one of the transforms (translation, rotation, scale, general transform) has changed.
        *  Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
        */
        virtual void OnCompositeTransformChanged() override;

#ifdef CANDERA_LAYOUT_ENABLED
        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector3& basePoint) const override;
#endif
        /**
        *  Gets the canvas axis-aligned bounding rectangle in local coordinate space which is computed by its
        *  actual content (e.g. boundaries of the image, text...).
        *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
        */
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle) const override;

        /**
        *  Gets the CanvasTransformable of this Canvas node.
        *  @return  This nodes CanvasTransformable.
        */
        virtual CanvasTransformable& GetCanvasTransformable() override { return m_canvasTransformable; }


        /**
        * Override from Node.
        * Invalidates parent canvas.
        */
        virtual void OnAncestorAdded(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas(); }

        /**
        * Override from Node.
        * Invalidates parent canvas.
        */
        virtual void OnAncestorRemoved(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas(); }

#endif

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        CompositeGroup(CompositeGroupMode mode);
        CompositeGroup(const CompositeGroup& other);
        CompositeGroup& operator = (const CompositeGroup& other);

        // overrides Node::DisposeSelf
        virtual void DisposeSelf() override;

        // overrides Node::Render
        virtual void Render() override {}

        // overrides Node::OnDescendantRemoved
        virtual void OnDescendantRemoved(Node* descendant) override;
        
        /**
         * The anchor point Node must be a descendant of the CompositeGroup in
         *  order to be successfully defined.
         *
         * @param id Id of the anchor point.
         * @param name Name of the anchor point.
         * @param node Node to be marked as anchor point.
         * @return true if anchor point is defined.
         * @return false if anchor point could not be defined.
         */
        bool DefineAnchorPoint(Id id, const Char* name, Node* node);

        /**
         * Update AnchorPoint id.
         * 
         * @param oldId id of the anchor point of which the id should be changed.
         * @param newId new id for the anchor point.
         * @return true if anchor point with oldId is found and id is changed, false otherwise.
         */
        bool UpdateAnchorPointId(Id oldId, Id newId);

        /**
         * @param widget Widget to be associated with this CompositeGroup.
         * @return whether the widget was added successfully or not.
         */
        bool AddWidget(WidgetBase* widget);

        /**
         * @param animation AnimationPlayerBase to be associated with this CompositeGroup.
         * @return whether the animation was added successfully or not.
         */
        void AddAnimation(const Animation::AnimationPlayerBase::SharedPointer& animation);

    private:
        // The function for creating composite groups is internal.
        // See Internal::CompositeGroupFunctions.
        static CompositeGroup* Create(CompositeGroupMode mode = GroupMode);

        AnchorPointList m_anchorPointList;
        AnimationList m_animationList;
        WidgetList m_widgetList;
        StringIdentifier m_stringIdentifier;

        CompositeGroupMode m_compositeGroupMode;

        friend class Candera::Internal::CompositeGroupFunctions;

#if defined(CANDERA_3D_CANVAS_ENABLED)
        CanvasTransformable m_canvasTransformable;
#endif

    };

/** @} */ // end of Core3D

    namespace Internal {
        class CompositeGroupFunctions {
        public:
            static CompositeGroup* Create(CompositeGroup::CompositeGroupMode mode = CompositeGroup::GroupMode) {
                return CompositeGroup::Create(mode);
            }

            static bool DefineAnchorPoint(CompositeGroup& compositeGroup, Id id, const Char* name, Node* node) {
                return compositeGroup.DefineAnchorPoint(id, name, node);
            }

            static bool AddWidget(CompositeGroup& compositeGroup, WidgetBase* widget) {
                return compositeGroup.AddWidget(widget); 
            }

            static void AddAnimation(CompositeGroup& compositeGroup, const Animation::AnimationPlayerBase::SharedPointer& animation) {
                compositeGroup.AddAnimation(animation);
            }

            static bool UpdateAnchorPointId(CompositeGroup& compositeGroup, Id oldId, Id newId) {
                return compositeGroup.UpdateAnchorPointId(oldId, newId);
            }

            static StringIdentifier& CompositeStringIdentifier(CompositeGroup& compositeGroup) {
                return compositeGroup.m_stringIdentifier;
            }
        };
    }
}

#endif //CANDERA_COMPOSITEGROUP_H

