//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CompositeGroup.h"

#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <Candera/System/Diagnostics/Log.h>
#if defined(CANDERA_3D_CANVAS_ENABLED)
#include <Candera/Engine3D/Canvas/Canvas.h>
#include <Candera/Engine3D/Canvas/CanvasGroup.h>
#endif

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);
    FEATSTD_RTTI_DEFINITION(CompositeGroup, Base)

    static bool IsDescendant(const Node* parent, const Node* child)
    {
        while ((child != 0) && (child != parent)) {
            child = child->GetParent();
        }

        return (child != 0);
    }

    void CompositeGroup::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    CompositeGroup* CompositeGroup::Clone() const
    {
        return FEATSTD_NEW(CompositeGroup)(*this);
    }
    
    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CompositeGroup::CompositeGroup(const Candera::CompositeGroup &other) :
        Base(other),
        m_compositeGroupMode(other.m_compositeGroupMode)
#if defined(CANDERA_3D_CANVAS_ENABLED)
        , m_canvasTransformable(*this, *this)
#endif
    {
        m_anchorPointList.Clear();
        m_animationList.Clear();
        m_widgetList.Clear();
    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CompositeGroup::CompositeGroup(CompositeGroupMode mode):
        m_compositeGroupMode(mode) 
#if defined(CANDERA_3D_CANVAS_ENABLED)
        , m_canvasTransformable(*this, *this)
#endif
    {
    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    bool CompositeGroup::DefineAnchorPoint(Id id, const Char* name, Node* node)
    {
        bool isAnchorPointValid =
            ((name != 0) &&
             (node != 0) &&
             (IsDescendant(this, node)) &&
             (GetAnchorPointNode(id) == 0));

        if (!isAnchorPointValid) {
            return false;
        }

        AnchorPoint anchorEntry = {id, name, node};
        return m_anchorPointList.Add(anchorEntry);
    }

    void CompositeGroup::OnDescendantRemoved(Node* descendant)
    {
        for (Int index = static_cast<Int>(m_anchorPointList.Size()) - 1; index >= 0; --index) {
            if (IsDescendant(descendant, m_anchorPointList[index].m_node)) {
                static_cast<void>(m_anchorPointList.Remove(index));
            }
        }
    }

    CompositeGroup::AnchorPointIterator CompositeGroup::GetAnchorPointIterator() const
    {
        return AnchorPointIterator(m_anchorPointList.ConstBegin(), m_anchorPointList.ConstEnd());
    }

    Node* CompositeGroup::GetAnchorPointNodeByName(const Char* name) const
    {
        if (name == 0) {
            return 0;
        }

        for (AnchorPointIterator iterator = GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings(iterator->m_name, name) == 0) {
                return iterator->m_node;
            }
        }

        return 0;
    }

    Node* CompositeGroup::GetAnchorPointNode(Id id) const
    {
        for (AnchorPointIterator iterator = GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
            if (iterator->m_id == id) {
                return iterator->m_node;
            }
        }

        return 0;
    }

    CompositeGroup::AnimationIterator CompositeGroup::GetAnimationIterator() const
    {
        return AnimationIterator(m_animationList.ConstBegin(), m_animationList.ConstEnd());
    }

    Animation::AnimationPlayerBase::SharedPointer CompositeGroup::GetAnimationByName(const Char* name) const
    {
        Animation::AnimationPlayerBase::SharedPointer nullAnimation(0);

        if (name == 0) {
            return nullAnimation;
        }

        for (AnimationIterator iterator = GetAnimationIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings((*iterator)->GetName(), name) == 0) {
                return *iterator;
            }
        }

        return nullAnimation;
    }
    
    Animation::AnimationPlayerBase::SharedPointer CompositeGroup::GetAnimation(Id id) const
    {
        for (AnimationIterator iterator = GetAnimationIterator(); iterator.IsValid(); ++iterator) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, const Candera::StringIdentifier &, False positive because GetId is from Candera::CanderaObject and not Candera::StringIdentifier.);
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::StringIdentifier = Candera::Animation::AnimationPlayerBase, False positive because GetId is from Candera::CanderaObject and not Candera::StringIdentifier.);
            if ((*iterator)->GetId() == id) {
                return *iterator;
            }
        }

        return Animation::AnimationPlayerBase::SharedPointer();
    }

    CompositeGroup::WidgetIterator CompositeGroup::GetWidgetIterator() const
    {
        return WidgetIterator(m_widgetList.ConstBegin(), m_widgetList.ConstEnd());
    }

    WidgetBase* CompositeGroup::GetWidgetByName(const Char* name) const
    {
        if (name == 0) {
            return 0;
        }

        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings((*iterator)->GetName(), name) == 0) {
                return *iterator;
            }
        }

        return 0;
    }

    WidgetBase* CompositeGroup::GetWidget(Id id) const
    {
        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if ((*iterator)->GetId() == id) {
                return *iterator;
            }
        }

        return 0;
    }

    CompositeGroup* CompositeGroup::Create(CompositeGroupMode mode)
    {
        return FEATSTD_NEW(CompositeGroup)(mode);
    }

    bool CompositeGroup::UpdateAnchorPointId(Id oldId, Id newId)
    {
        for (AnchorPointList::Iterator it = m_anchorPointList.Begin(); it != m_anchorPointList.End(); ++it) {
            if ((*it).m_id == oldId) {
                (*it).m_id = newId;
                return true;
            }
        }

        return false;
    }

    bool CompositeGroup::AddWidget(WidgetBase* widget)
    {
        widget->SetOwnerId(&m_stringIdentifier);
        return m_widgetList.Add(widget);
    }

    void CompositeGroup::AddAnimation(const Animation::AnimationPlayerBase::SharedPointer& animation)
    {
        animation->SetOwnerId(&m_stringIdentifier);
        static_cast<void>(m_animationList.Add(animation));
    }
#if defined(CANDERA_3D_CANVAS_ENABLED)
    void  CompositeGroup::GetBasePoint(Vector2& basePoint) const  {
        Rectangle boundingRect;
        m_canvasTransformable.GetEffectiveBoundingRectangle(boundingRect);

        basePoint = boundingRect.GetPosition() + (boundingRect.GetSize() * 0.5F);
    }

    void CompositeGroup::OnCompositeTransformChanged()
    {
        Base::OnCompositeTransformChanged();
        Canvas* parentalCanvas = m_canvasTransformable.GetParentCanvas();
        if (0 != parentalCanvas) {
            parentalCanvas->SetChanged();
        }
    }
#ifdef CANDERA_LAYOUT_ENABLED

    void CompositeGroup::GetBasePoint(Vector3& basePoint) const
    {
        Vector2 basePoint2D;
        GetBasePoint(basePoint2D);

        basePoint.SetX(basePoint2D.GetX());
        basePoint.SetY(basePoint2D.GetY());
        const Vector3& currPos = GetPosition();
        basePoint.SetZ(currPos.GetZ());
    }
#endif

    void  CompositeGroup::GetComputedBoundingRectangle(Rectangle& boundingRectangle) const
    {
        Internal::BoundingRectangleTraverser traverser(*this);
        traverser.Traverse(*this);
        traverser.GetBoundingRectangle(boundingRectangle);
    }
#endif

}
