//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CLEARMODE_H)
#define CANDERA_CLEARMODE_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/SkyBox.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

class Camera;
class RenderTarget3D;
class Rectangle;

/**
 * @brief   Describes how a viewport is cleared.
 * The color buffer may be cleared using a given color.
 * The depth buffer can be cleared using a given depth value.
 * Color and depth buffer can be enabled separately for clearing.
 * Note: ClearMode only represents the clearing strategy. It does not do any clearing itself.
 */
class ClearMode
{
    public:

        /**
         * Constructs a ClearMode object.
         */
        ClearMode();
        /**
         * Destructs a ClearMode object.
         */
        ~ClearMode() {}

        /**
         * Sets the clear color.
         * @param color Clear color to be set.
         */
        void SetClearColor(const Color& color) { m_clearColor = color; }
        /**
         * Gets the clear color.
         * @return Clear color of this ClearMode.
         */
        const Color& GetClearColor() const { return m_clearColor; }
        /**
         * Sets if the color buffer is cleared.
         * @param enabled Enables(true)/Disables(false) if the color buffer is cleared.
         */
        void SetColorClearEnabled(bool enabled) { m_isColorClearEnabled = enabled; }
        /**
         * Gets if the color buffer is cleared.
         * @return True if the color buffer is cleared, false otherwise.
         */
        bool IsColorClearEnabled() const { return m_isColorClearEnabled; }
        /** Sets color write flags for all 4 components of RGBA.
        * @param enableRed true enables and false disables writing of red value to color buffer.
        * @param enableGreen true enables and false disables writing of green value to color buffer.
        * @param enableBlue true enables and false disables writing of blue value to color buffer.
        * @param enableAlpha true enables and false disables writing of alpha value to color buffer.
        * Per default writing to each color component is enabled.
        */
        void SetColorWriteEnabled(bool enableRed, bool enableGreen, bool enableBlue, bool enableAlpha);
        /**
         * Sets the color write flag for the red component of RGBA. Compare similar function in RenderMode.
         * @param enableRed true enables and false disables writing of red value to color buffer.
         */
        void SetColorWriteRedEnabled(bool enableRed) { m_isColorWriteRedEnabled = enableRed; }
        /**
         * Gets the color write flag for the red component of RGBA. Compare similar function in RenderMode.
         * @return Whether writing of red color value is enabled(true) or not(false);
         */
        bool IsColorWriteRedEnabled() const { return m_isColorWriteRedEnabled; }
        /**
         * Sets the color write flag for the green component of RGBA. Compare similar function in RenderMode.
         * @param enableGreen true enables and false disables writing of green value to color buffer.
         */
        void SetColorWriteGreenEnabled(bool enableGreen) { m_isColorWriteGreenEnabled = enableGreen; }
        /**
         * Gets the color write flag for the green component of RGBA. Compare similar function in RenderMode.
         * @return Whether writing of green color value is enabled(true) or not(false);
         */
        bool IsColorWriteGreenEnabled() const { return m_isColorWriteGreenEnabled; }
        /**
         * Sets the color write flag for the blue component of RGBA. Compare similar function in RenderMode.
         * @param enableBlue true enables and false disables writing of blue value to color buffer.
         */
        void SetColorWriteBlueEnabled(bool enableBlue) { m_isColorWriteBlueEnabled = enableBlue; }
        /**
         * Gets the color write flag for the blue component of RGBA. Compare similar function in RenderMode.
         * @return Whether writing of blue color value is enabled(true) or not(false);
         */
        bool IsColorWriteBlueEnabled() const { return m_isColorWriteBlueEnabled; }
        /**
         * Sets the color write flag for the alpha component of RGBA. Compare similar function in RenderMode.
         * @param enableAlpha true enables and false disables writing of alpha value to color buffer.
         */
        void SetColorWriteAlphaEnabled(bool enableAlpha) { m_isColorWriteAlphaEnabled = enableAlpha; }
        /**
         * Gets the color write flag for the alpha component of RGBA. Compare similar function in RenderMode.
         * @return Whether writing of alpha color value is enabled(true) or not(false);
         */
        bool IsColorWriteAlphaEnabled() const { return m_isColorWriteAlphaEnabled; }

        /**
         * Sets the depth clear value.
         * @param depth Depth clear value to be set.
         */
        void SetClearDepth(Float depth) { m_clearDepth = depth; }
        /**
         * Gets the depth clear value.
         * @return Depth clear value of this ClearMode.
         */
        Float GetClearDepth() const { return m_clearDepth; }
        /**
         * Sets if the depth buffer is cleared.
         * @param enabled Enables(true)/Disables(false) if depth buffer is cleared.
         */
        void SetDepthClearEnabled(bool enabled) { m_isDepthClearEnabled = enabled; }
        /**
         * Gets if the depth buffer is cleared.
         * @return Whether depth buffer is cleared (true) or not(false).
         */
        bool IsDepthClearEnabled() const { return m_isDepthClearEnabled; }

        /**
         * Sets the stencil clear value.
         * @param stencilClearValue stencil clear value to be set.
         */
        void SetStencilClearValue(Int32 stencilClearValue) { m_stencilClearValue = stencilClearValue; }
        /**
         * Gets the stencil clear value.
         * @return Stencil clear value of this ClearMode.
         */
        Int32 GetStencilClearValue() const { return m_stencilClearValue; }
        /**
         * Sets if the stencil buffer is cleared.
         * @param enabled Enables(true) / Disables(false) clearing of stencil buffer.
         */
        void SetStencilClearEnabled(bool enabled) { m_isStencilClearEnabled = enabled; }
        /**
         * Gets if the stencil buffer is cleared.
         * @return Whether stencil buffer clearing is enabled(true) or not(false);
         */
        bool IsStencilClearEnabled() const { return m_isStencilClearEnabled; }
        /**
         * Sets the stencil write mask (front and back).
         * @param writeMask Stencil write mask to be set.
         */
        void SetStencilWriteMask(const UInt32 writeMask) { m_stencilWriteMask = writeMask; }
        /**
         * Gets the stencil write mask.
         * @return Stencil write mask of this clear mode.
         */
        UInt32 GetStencilWriteMask() const { return m_stencilWriteMask; }

        /**
         * Sets if SkyBox is used as background.
         * Note: If SkyBox is enabled it overwrites the entire background. Thus, the color clear settings are ignored and color buffer clear is omitted.
         * @param enable defines whether sky box shall be rendered or not.
         */
        void SetSkyBoxEnabled(bool enable) { m_isSkyBoxEnabled = enable; }

        /**
         * Gets if rendering of SkyBox is enabled.
         * @return Whether rendering of SkyBox is enabled or not.
         */
        bool IsSkyBoxEnabled() const { return m_isSkyBoxEnabled; }

        /**
         * Sets the SkyBox to render, if enabled (See SetSkyBoxEnabled).
         * @param skybox specifies the skybox.
         */
        void SetSkyBox(const MemoryManagement::SharedPointer<SkyBox>& skybox) { m_skyBox = skybox; }

        /**
         * Gets the SkyBox that is rendered, if enabled (See SetSkyBoxEnabled).
         * @return SkyBox associated to this ClearMode object.
         */
        const MemoryManagement::SharedPointer<SkyBox>& GetSkyBox() const { return m_skyBox; }

        /**
         * Update camera hash info
         * @param hash new hash value
         */
        void UpdateHash(UInt32 &hash) const;

    private:
        bool m_isColorClearEnabled;
        bool m_isDepthClearEnabled;
        bool m_isStencilClearEnabled;
        bool m_isColorWriteRedEnabled;
        bool m_isColorWriteGreenEnabled;
        bool m_isColorWriteBlueEnabled;
        bool m_isColorWriteAlphaEnabled;
        bool m_isSkyBoxEnabled;
        Int32 m_stencilClearValue;
        UInt32 m_stencilWriteMask;
        Float m_clearDepth;
        Color m_clearColor;
        MemoryManagement::SharedPointer<SkyBox> m_skyBox;
};

/** @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_CLEARMODE_H
