//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CAMERA_H)
#define CANDERA_CAMERA_H

#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/ClearMode.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Projection.h>
#include <Candera/Engine3D/Core/ProjectionListener.h>
#include <Candera/Engine3D/Core/ViewingFrustum.h>
#include <Candera/EngineBase/Common/ListenerContainer.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetLoader3DDataTypes.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

// Forward declarations.
class AbstractRenderOrder;
class CameraListener;
class CameraRenderStrategy;
class RenderTarget3D;
class Scene;
class StereoCamera;

namespace Monitor {
    class RenderBenchmarkMeasurement;
}

/**
 *  @brief   A Camera provides Modelview and Projection matrices. Camera is a Transformable.*
 *  The eye point is set using the "position" part of the Camera as a Transformable.
 *  Viewing orientation is controlled using the following:
 *   - the local "look" vectors; functions SetLookAtVector, LookAtWorldPoint, RotateAroundWorldAxis
 *   - the local "up"  vector; function SetUpVector
 *   - rotation of the Camera as a Transformable, by modifying the "rotation" part of Transformable.
 *
 *  The initial values of the local "look" vectors are
 *    LookAt vector is x:0, y:0, z: -1
 *    Up vector is x:0, y:1, z:0
 *    Right vector is (as a result) x:1, y:0, z:0
 *
 *  Especially for multi pass rendering, it might be required that each camera (= render pass) renders the captured content with a different
 *  default render mode. To enable this, the Camera's RenderMode - if set - simply overrules the DefaultRenderMode (see Renderer::SetDefaultRenderMode)
 *  during its render pass. Thus, all Nodes rendered by the Camera that do not have their own RenderMode, use the RenderMode  attached to the Camera.
 *  Further, if a Node's RenderMode  has set an inheritance bit for a certain render attribute, then the property of the base render mode is used,
 *  this is either the Camera's RenderMode or the DefaultRenderMode, if the Camera has no RenderMode attached.
 */
class Camera : public Node
{
    FEATSTD_TYPEDEF_BASE(Node);

    friend class BatchOrderCriterion;
    friend class CameraProjectionListener;
    friend class Renderer;
    friend class Monitor::RenderBenchmarkMeasurement;

    public:

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to created Camera object.
         */
        static Camera* Create();

         /**
          * Creates an instance of Camera, using the data provided by
          * the passed StereoCamera.
          * Basically one of the StereoCamera's eyes is cloned,
          * its asymmetric StereoProjection is exchanged with symmetric PerspectiveProjection,
          * and finally the StereoCamera's transformation and other node properties are applied to the new
          * Camera node.
          * @param camera StereoCamera to extract data from.
          * @param left Whether the left (true) or the right (false) eye shall be used to retrieve data from.
          * @return Pointer to the created Camera object. If the provided StereoCamera is 0,
          *         a simple created Camera without any settings made is returned.
          */
        static Camera* Create(StereoCamera* camera, bool left = true);

        /**
         *  Destructor
         */
        virtual ~Camera() override;

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual Camera* Clone() const override;

        /**
         *  Sets projection of Camera, this might be a orthographic, perspective or generic projection.
         *  The projection is analogous to choosing a lens for the camera.
         *  @param projection Projection to be set.
         */
        void SetProjection(MemoryManagement::SharedPointer<Projection> projection);

        /**
         *  Retrieves projection of this Camera.
         *  @return Projection of this Camera.
         */
        MemoryManagement::SharedPointer<Projection> GetProjection() const;

        /**
         *  Sets generic view matrix.
         *  The view matrix translates and rotates objects to place them in camera space, where the camera is at the origin.
         *  If LookAtNode is set, orientation intended via this function call may become overwritten.
         *  @param viewMatrix View matrix to be set.
         */
        void SetViewMatrix(const Matrix4& viewMatrix);

        /**
         *  Gets view matrix of this Camera.
         *  @return View matrix of this Camera.
         */
        const Matrix4& GetViewMatrix() const;

        /**
         *  Gets view-projection matrix of this Camera.
         *  @return product of view matrix * projection matrix.
         */
        const Matrix4& GetViewProjectionMatrix() const;

        /**
         *  Sets this Camera's look-at vector. Initial value is x:0, y:0, z:-1.
         *  The stored look-at vector will be normalized.
         *  The up-vector will be recalculated based on current up-vector (see SetUpVector).
         *  If the passed look-at vector is collinear with the up vector, the up vector is set to - or + old lookat vector,
         *  depending on whether the new lookat is in the same or the opposite direction as the old up vector.
         *  If LookAtNode is set, orientation intended via this function call may become overwritten.
         *  @param lookAt Look-at vector to be set.
         *  @return false if the passed lookat vector is singular (close to 0)
         */
        bool SetLookAtVector(const Vector3& lookAt);

        /**
         *  Retrieves normalized look-at vector of this camera.
         *  @return Look-at vector of this camera.
         */
        const Vector3& GetLookAtVector() const;

        /**
         *  Retrieves normalized look-at vector in world space.
         *  @return Look-at vector in world space.
         */
        Vector3 GetWorldLookAtVector() const;

        /**
         *  Sets the up vector of this Camera. The up vector is made orthogonal (calculated as the vector
         *  in the same plane as the look-at vector and the input up vector), and normalized. Subsequent calls to SetLookAtVector
         *  will also perform this "make orthogonal" operation.
         *  If LookAtNode is set, orientation intended via this function call may become overwritten.
         *  @param up input up vector; Note: GetUpVector will probably NOT return this value
         *  @return False if up is singular or collinear to already set up vector. True if up was set.
         */
        bool SetUpVector(const Vector3& up);

        /**
         *  Retrieves normalized up-vector of this camera.
         *  @return Up vector of this camera.
         */
        const Vector3& GetUpVector() const;

        /**
         *  Retrieves normalized up vector in world space.
         *  @return Up vector in world space.
         */
        Vector3 GetWorldUpVector() const;

        /**
         *  Retrieves normalized right vector of this camera.
         *  @return Right vector of this camera.
         */
        Vector3 GetRightVector() const;

        /**
         *  Retrieves normalized right-vector in world space.
         *  @return Right-vector in world space.
         */
        Vector3 GetWorldRightVector() const;

        /**
         * @param up New up vector.
         * @param lookAt New lookAt vector.
         * @see SetUpVector
         * @see SetLookAtVector
         */
        bool SetUpAndLookAtVectors(const Vector3& up, const Vector3& lookAt);

        /**
         *  Points the camera at a location in world space.
         *  This helper method generates the look-at-vector for the camera, based on its current position and the target point.
         *  If LookAtNode is set, orientation intended via this function call may become overwritten.
         *  @param targetPoint specifies the point in world space to look at.
         *  @return true on success; success/failure determined by the resulting call to SetLookAtVector.
         */
        bool LookAtWorldPoint(const Vector3& targetPoint);

        /**
         *  Rotates the camera around an arbitrary world axis.
         *  If LookAtNode is set, orientation intended via this function may become overwritten.
         *  @param axis           can be any direction vector.
         *  @param angleDegrees   specifies the rotation around the axis in degrees.
         *  @return true          on success; false if axis is singular.
         */
        bool RotateAroundWorldAxis(const Vector3& axis, Float angleDegrees);

        /**
         *  Sets the Render Target for this camera.
         *
         * If the render pass of the render strategy of this camera is not yet
         *  completed the RenderTarget should not be changed. A listener should be
         *  used to check when the RenderTarget can be changed in case of multi
         *  pass rendering.
         *
         * @param renderTarget Render target of this camera to be set.
         */
        void SetRenderTarget(RenderTarget3D* renderTarget);

        /**
         *  Retrieves the render target from this camera.
         *  @return the render target from this camera.
         */
        RenderTarget3D* GetRenderTarget() const { return m_renderTarget; }
        /**
         *  Sets the Clear Mode for this camera. Per default a ClearMode is set with ClearMode's default values (see class ClearMode).
         *  @param clearMode Clear Mode to be set.
         */
        void SetClearMode(const ClearMode& clearMode) { m_clearMode = clearMode; }

        /**
         *  Retrieves the clear mode from this camera.
         *  @return The clear mode from this camera.
         */
        const ClearMode& GetClearMode() const { return m_clearMode; }

        /**
         *  The Viewport specifies a rectangular region within the Camera's RenderTarget.
         *  The Viewport settings transform the vertex positions from normalized device coordinates [-1..1] into
         *  normalized screen coordinates [0..1]. The upper left corner of the Render Target is addressed with left:0, top:0.
         *  The dimension of the Render Target is specified with width 1.0F, height 1.0F.
         *  The Viewport's width and height must be greater than 0.
         *  A Viewport that intersects the RenderTarget is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
         *  If the viewport aspect ratio differs from the aspect ratio of the Camera's projection matrix, the final render result will
         *  appear stretched or compressed.
         *  The default value for viewport is: left 0, top 0, width 1, height 1.
         *  @param viewport Rectangle specifying the cameras viewport.
         */
        void SetViewport(const Rectangle& viewport) { m_viewport = viewport; }

        /**
         *  The Viewport specifies a rectangular region within the Camera's RenderTarget.
         *  The Viewport settings transform the vertex positions from normalized device coordinates [-1..1] into
         *  normalized screen coordinates [0..1]. The upper left corner of the Render Target is addressed with left:0, top:0.
         *  The dimension of the Render Target is specified with width 1.0F, height 1.0F.
         *  The Viewport's width and height must be greater than 0.
         *  A Viewport that intersects the RenderTarget is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
         *  If the viewport aspect ratio differs from the aspect ratio of the Camera's projection matrix, the final render result will
         *  appear stretched or compressed.
         *  The default value for viewport is: left 0, top 0, width 1, height 1.
         *  @return The rectangle specifying the cameras viewport.
         */
        const Rectangle& GetViewport() const { return m_viewport; }

        /**
         *  The scissor rectangle specifies a rectangular region within the Camera's RenderTarget in normalized screen coordinates [0..1].
         *  The upper left corner of the Render Target is addressed with left:0, top:0.
         *  The dimension of the Render Target is specified with width 1.0F, height 1.0F.
         *  The scissor rectangles width and height must be greater than 0.
         *  A scissor rectangle that intersects the RenderTarget is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
         *  If scissoring is enabled (see IsScissoringEnabled) pixels outside of the intersection area of viewport and scissor rectangle are clipped.
         *  The ClearMode clears the entire area defined by scissor rectangle if scissoring is enabled, the viewport otherwise.
         *  The default value for scissor rectangle is: left 0, top 0, width 1, height 1.
         *  @param scissorRectangle Scissor rectangle to be set.
         */
        void SetScissorRectangle(const Rectangle& scissorRectangle) { m_scissorRectangle = scissorRectangle; }

        /**
         *  The scissor rectangle specifies a rectangular region within the Camera's RenderTarget in normalized screen coordinates [0..1].
         *  The upper left corner of the Render Target is addressed with left:0, top:0.
         *  The dimension of the Render Target is specified with width 1.0F, height 1.0F.
         *  The scissor rectangles width and height must be greater than 0.
         *  A scissor rectangle that intersects the RenderTarget is allowed, for instance values like: left -1, top -1, width 2.0, height 2.0.
         *  If scissoring is enabled (see IsScissoringEnabled) pixels outside of the intersection area of viewport and scissor rectangle are clipped.
         *  The ClearMode clears the entire area defined by scissor rectangle if scissoring is enabled, the viewport otherwise.
         *  The default value for scissor rectangle is: left 0, top 0, width 1, height 1.
         *  @return The scissor rectangle.
         */
        const Rectangle& GetScissorRectangle() const { return m_scissorRectangle; }

        /**
         *  Defines if scissoring is enabled or not. Per default, scissoring is disabled.
         *  If scissoring is enabled, the scissor rectangle (see SetScissorRectangle) is used by ClearMode and scissor test.
         *  @param enable Enables(true)/Disables(false) scissoring.
         */
        void SetScissoringEnabled(bool enable) { m_isScissoringEnabled = enable; }

        /**
         *  Retrieves whether scissoring is enabled or not.
         *  If scissoring is enabled, the scissor rectangle (see SetScissorRectangle) is used by ClearMode and scissor test.
         *  @return True if scissoring is enabled, otherwise false
         */
        bool IsScissoringEnabled() const { return m_isScissoringEnabled; }

        /**
         *  Sets render sequence of this camera in an ascending order, used if Renderer::RenderAllCameras is invoked.
         *  @param sequenceNumber Sequence number within the render sequence to be set to this camera.
         */
        void SetSequenceNumber(Int32 sequenceNumber);

        /**
         *  Retrieves render sequence of this camera.
         *  @return Sequence number of this camera within the render sequence.
         */
        Int32 GetSequenceNumber() const;

        /**
         *  Returns the viewing frustum of the camera.
         *  @return Viewing frustum of the camera.
         */
        const ViewingFrustum& GetViewingFrustum() const ;

        /**
         *  Enables or disables viewing frustum culling. Enabled per default.
         *  @param enable Enables(true)/Disables(false) viewing frustum culling of this camera.
         */
        void SetViewingFrustumCullingEnabled(bool enable) { m_isViewingFrustumCullingEnabled = enable; }

        /**
         *  Retrieves whether viewing frustum culling is enabled or not.
         *  @return True if viewing frustum culling is enabled otherwise false.
         */
        bool IsViewingFrustumCullingEnabled() const { return m_isViewingFrustumCullingEnabled; }

        /**
         *  Enables or disables a swap of Camera's render target after rendering has been processed. Disabled per default.
         *  @param enable Enables(true)/Disables(false) swapping of this camera.
         */
        void SetSwapEnabled(bool enable) { m_isSwapEnabled = enable; }

        /**
         *  Retrieves whether Camera's swap is enabled or not.
         *  @return True if swap is enabled, otherwise false.
         */
        bool IsSwapEnabled() const { return m_isSwapEnabled; }

        /**
         *  Adds a CameraListener object. Event/timing of the calls: before/after this Camera is rendered.
         *  @param listener The lifetime of the listener is managed by the calling code. Camera does not take ownership.
         *  @return True if successful, false otherwise.
         */
        bool AddCameraListener(CameraListener* listener);

        /**
         *  Removes a CameraListener object.
         *  @param listener Specifies the CameraListener object to remove from this Camera.
         *  @return True if successful, false otherwise.
         */
        bool RemoveCameraListener(CameraListener* listener);

        /**
         *  Overrides IsRenderPreconditionFulfilled from base class. A Camera object itself cannot be rendered.
         *  @returns Always returns false because a camera cannot be rendered.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override { return false; }

        /**
         *  Assigns a CameraRenderStrategy to the Camera. The Camera does not take ownership.
         *  @param cameraRenderStrategy specifies the CameraRenderStrategy object assigned to this Camera. A CameraRenderStrategy
         *  may interrupt one camera's render pass for special use cases like partitioning a camera's render pass into
         *  multiple sub render passes.
         *  If no CameraRenderStrategy object is assigned the camera attempts to render all nodes within its parent scene in one render pass.
         */
        void SetCameraRenderStrategy(CameraRenderStrategy* cameraRenderStrategy);

        /**
         *  Retrieves the camera render strategy for this camera.
         *  @return The camera render strategy for this camera.
         */
        CameraRenderStrategy* GetCameraRenderStrategy() const { return m_cameraRenderStrategy; }

        /**
         *  Uploads the ClearMode's SkyBox, if set.
         *  @return true if upload of SkyBox succeeded or no SkyBox is set, false if upload failed.
         */
        virtual bool Upload() override;

        /**
         *  Unloads the ClearMode's SkyBox if there is any.
         *  @return true if upload of SkyBox succeeded or no SkyBox is set, false if unload failed.
         */
        virtual bool Unload() override;

        /**
         *  Set an AbstractRenderOrder for this Camera. AbstractRenderOrder objects must not be shared between Cameras.
         *  Every camera instance requires its own AbstractRenderOrder instance. If no AbstractRenderOrder is set, Camera uses
         *  its own RenderOrder object internally. Camera takes ownership of the AbstractRenderOrder object and calls
         *  AbstractRenderOrder::Dispose() on the destruction of the Camera. Please consider that only the last AbstractRenderOrder
         *  set on the Camera will be disposed upon Camera destruction. External ownership is assumed for the rest.
         *  @param renderOrder  Abstract render order to be set.
         */
        void SetRenderOrder(AbstractRenderOrder* renderOrder);

        /**
         *  Retrieve the AbstractRenderOrder object used by this Camera.
         *  @return  A pointer to the abstract render order used by this Camera.
         */
        AbstractRenderOrder* GetRenderOrder() { return m_renderOrder; }
        const AbstractRenderOrder* GetRenderOrder() const { return m_renderOrder; }

        /**
         *  Traverse the scene that this camera is part of, cull all nodes, assign them to the render order of the camera,
         *  calculate the order criterion value for each node in the render order and sort them.
         */
        void PrepareRenderOrder(const Rectangle& dirtyArea = Rectangle(0.0F, 0.0F, 1.0F, 1.0F));

        /**
         *  Sets a look-at-node. If such a node is set the camera will always look at that node.
         *  This behavior overwrites rotations set by the other methods.
         *  The look-at-vector is calculated in CameraLookAtNodeController::LateUpdate whenever (false == m_isViewUpToDate) or camera or node move.
         *  Function call chain: CameraLookAtNodeController::LateUpdate calls LookAtWorldPoint that calls SetLookAtVector and the up-vector will be recalculated
         *  based on current up-vector (see SetUpVector).
         *  @param lookAtNode The node to look at.
         */
        bool SetLookAtNode(Node* lookAtNode);

        /**
         *  Gets the set look-at-node.
         *  @return The set look-at-node or null if none was set.
         */
        Node* GetLookAtNode() const { return GetValue(CdaDynamicPropertyInstance(LookAtNode)); }

        /**
         * Check if camera setting were updated: scissoring, clear mode, viewport
         */
        bool WasUpdated() const;

        /**
         * Sets whether the camera effective alpha is multiplied to the effective alpha of a Node when being rendered.
         * Note: The camera alpha value is only applicable in case a Material with diffuse color is used.
         * @param enable Enables or disables the camera effective alpha.
         */
        void SetCameraEffectiveAlphaEnabled(bool enable) { m_isCameraEffectiveAlphaEnabled = enable; }

        /**
        * Gets whether the camera effective alpha is multiplied to the effective alpha of a Node when being rendered.
        * Note: The camera alpha value is only applicable in case a Material with diffuse color is used.
        * @return Whether the camera effective alpha is enabled (true) or disabled (false).
        */
        bool IsCameraEffectiveAlphaEnabled() const { return m_isCameraEffectiveAlphaEnabled; }

    protected:
        // Deliberately non-public Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Camera();
        FEATSTD_LINT_NEXT_EXPRESSION(1704, "Copy contructor is only used internaly by Clone")
        Camera(const Camera& camera);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;
        /**
         *  Overrides Render from base class to allow instantiation.
         */
        virtual void Render() override {}

        /**
         *  Activates this Camera and its ClearMode states in RenderDevice.
         *  @return False if renderContext could not begin drawing. This occurs if
         *                if CameraRenderContext is 0. Otherwise camera is activated and returns true.
         */
        virtual bool Activate();

        /**
         *  Deactivates this Camera.
         *  @return False if renderContext could not end drawing. This occurs if
         *                if CameraRenderContext is 0. Otherwise camera is deactivated and returns true.
         */
        virtual bool Deactivate();

        //override (Node)
        virtual void OnAncestorAdded(Scene* scene) override;

        // override (Node)
        virtual void OnAncestorRemoved(Scene* scene) override;

        // override (Node)
        virtual void OnAncestorTransformChanged() override;

        /**
         *  Called before this Camera is rendered. Notifies the RenderCameraListeners.
         */
        void NotifyListenersOnPreRender();

        /**
         *  Called after this Camera is rendered. Notifies the RenderCameraListeners.
         */
        void NotifyListenersOnPostRender();

        /**
         *  Called after this Camera's projection was changed. Notifies the RenderCameraListeners.
         */
        void NotifyListenersOnProjectionChanged();

        /**
         *  Called after Camera's View was changed. Notifies the CameraListeners.
         */
        void NotifyListenersOnViewChanged();

        /**
         *  Called after this Camera's RenderTarget3D was changed. Notifies the RenderCameraListeners.
         */
        void NotifyListenersOnRenderTargetChanged(RenderTarget3D* previousRenderTarget);

        /**
         *  Called after Camera's CameraRenderStrategy changed. Notifies the CameraListeners.
         */
        void NotifyListenersOnRenderStrategyChanged(CameraRenderStrategy* previousRenderStrategy);

        /**
        *  Called before a camera is activated.
        */
        void NotifyListenersOnPreActivate();

        /**
        *  Called after a camera was activated.
        */
        void NotifyListenersOnPostActivate();

        /**
        *  Called before the framebuffer is cleared.
        */
        void NotifyListenersOnPreClear();

        /**
        *  Called after the framebuffer clear call has been sent to the device.
        */
        void NotifyListenersOnPostClear();


        /**
         * Enable/disable clearing on activation, using the configured ClearMode. Enabled by default.
         */
        void SetClearingEnabled(bool enable) { m_isClearingEnabled = enable; }

        /**
         * Check if clearing in activation is enabled.
         */
        bool IsClearingEnabled() const { return m_isClearingEnabled; }

        /**
         * Compute the internal hash from the camera settings
         */
        bool ComputeHash() const;

    private:
        class OnPreRenderEvent;
        class OnPostRenderEvent;
        class OnProjectionChangedEvent;
        class OnViewChangedEvent;
        class OnRenderTargetChangedEvent;
        class OnRenderStrategyChangedEvent;
        class OnPreActivateEvent;
        class OnPostActivateEvent;
        class OnPreClearEvent;
        class OnPostClearEvent;

        Camera& operator = (const Camera& camera);

        mutable bool m_isViewUpToDate;
        mutable bool m_isProjUpToDate;
        mutable bool m_isViewProjUpToDate;
        mutable bool m_isViewFrustumUpToDate;
        bool m_isScissoringEnabled;                     // Defines if scissoring is enabled or not. Default value is false.
        bool m_isViewingFrustumCullingEnabled;          // Used by Renderer. Default value is true.
        bool m_isSwapEnabled;                           // Used by Renderer. Default value is false.
        bool m_isClearingEnabled;                       // Used by Renderer. Default value is true.
        bool m_isCameraEffectiveAlphaEnabled;
        Int32 m_sequenceNumber;
        AbstractRenderOrder* m_internalRenderOrder;     // Identifies the default render order created by the camera itself.
        AbstractRenderOrder* m_renderOrder;             // Render order to be used by the camera. Can originate from an external source.
        CameraRenderStrategy* m_cameraRenderStrategy;   // Used by Renderer. Default value is null, thus no CameraRenderStrategy is applied.
        RenderTarget3D* m_renderTarget;
        Vector3 m_lookatVector;
        Vector3 m_upVector;
        Rectangle m_viewport;               // Default value is: left 0.0F, top 0.0F, width 1.0F, height 1.0F.
        Rectangle m_scissorRectangle;       // Default value is: left 0.0F, top 0.0F, width 1.0F, height 1.0F.
       /* "mutable" justification:
        *   - the view matrix is functionally dependent on the look-at parameters (camera position, target point, up vector)
        *   - the view projection matrix is functionally dependent on the view and projection matrices
        *   - the "is up to date" flags are caching helpers
        *       (and semantically invisible to clients, like any cache should be)
        */
        mutable Matrix4 m_view;
        MemoryManagement::SharedPointer<Projection> m_projection;
        mutable Matrix4 m_viewProjection;
        mutable ViewingFrustum m_viewingFrustum;
        ClearMode m_clearMode;

        typedef Internal::ListenerContainer<CameraListener> CameraListenerContainer;
        CameraListenerContainer m_cameraListeners;

        bool RotateAroundAxis(const Vector3& axis, Float angleDeg, Vector3& binormal) const;

        bool IsViewUpToDate() const { return m_isViewUpToDate; }
        bool IsProjUpToDate() const { return m_isProjUpToDate; }
        bool IsViewProjUpToDate() const { return m_isViewProjUpToDate; }
        bool IsViewingFrustumUpToDate() const { return m_isViewFrustumUpToDate; }

        void UpdateViewProjection() const;
        void RecalculateView() const;

        void MakeUpOrthogonalToLookat();

        void CreateInternalRenderOrder();
        static void CloneRenderOrder(const Camera& source, Camera& destination);

        CdaDynamicProperties(Candera::Camera, Candera::Node);

            CdaDynamicProperty(LookAtNode, Node*);
                CdaDynamicPropertyDefaultValue(0);
                CdaDynamicPropertyDescription("A node that the camera shall constantly look at.")
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();

        class CameraProjectionListener : public ProjectionListener
        {
            friend class Camera;

            public:
                CameraProjectionListener():ProjectionListener(), m_camera(0) { };
                ~CameraProjectionListener() {
                    m_camera = 0;
                };

                virtual void OnProjectionChanged(Projection* projection) {
                    FEATSTD_UNUSED(projection);

                    if (m_camera!=0) {
                        m_camera->m_isProjUpToDate = false;
                        m_camera->m_isViewProjUpToDate = false;
                        m_camera->m_isViewFrustumUpToDate = false;

                        m_camera->NotifyListenersOnProjectionChanged();
                    }
                }

            protected:
                void SetCamera(Camera* camera) { m_camera = camera; }

            private:
                Camera* m_camera;
        } m_projectionListener;

        mutable UInt32 m_hash;
};
/** @} */ // end of Core3D
} // namespace Candera
#endif    // CANDERA_CAMERA_H
