//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Camera.h"
#include <Candera/Engine3D/Core/CameraLookAtNodeController.h>
#include <Candera/Engine3D/Cloning/TreeCloner.h>
#include <Candera/Engine3D/Core/CameraListener.h>
#include <Candera/Engine3D/Core/CameraRenderStrategy.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/StereoCamera.h>
#include <Candera/Engine3D/Core/StereoProjection.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/RenderOrder/RenderOrder.h>
#include <Candera/System/ControllerSystem/ControllerSystem.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Matrix3.h>
//#include <Candera/System/Monitor/MonitorPublicIF.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <FeatStd/Util/Hash.h>


namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    Camera::Camera() :
        Base(),
        m_isViewUpToDate(false),
        m_isProjUpToDate(false),
        m_isViewProjUpToDate(false),
        m_isViewFrustumUpToDate(false),
        m_isScissoringEnabled(false),
        m_isViewingFrustumCullingEnabled(true),
        m_isSwapEnabled(false),
        m_isClearingEnabled(true),
        m_isCameraEffectiveAlphaEnabled(false),
        m_sequenceNumber(0),
        m_internalRenderOrder(0),
        m_renderOrder(0),
        m_cameraRenderStrategy(0),
        m_renderTarget(0),
        m_lookatVector(0.0F, 0.0F, -1.0F),
        m_upVector(0.0F, 1.0F, 0.0F),
        m_viewport(Candera::Rectangle(0.0F, 0.0F, 1.0F, 1.0F)),
        m_scissorRectangle(Candera::Rectangle(0.0F, 0.0F, 1.0F, 1.0F)),
        m_view(),
        m_projection(0),
        m_viewProjection(),
        m_projectionListener(),
        m_hash(FeatStd::Hash::cInvalidHashValue)
    {
        m_viewingFrustum.SetCamera(this);
        m_projectionListener.SetCamera(this);

        CreateInternalRenderOrder();

        ComputeHash();
    }

    Camera* Camera::Create()
    {
        return FEATSTD_NEW(Camera);
    }

    Camera* Camera::Create(StereoCamera* camera, bool left)
    {
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, camera, CANDERA_LINT_REASON_NONCONST)

        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        Camera* ptr = FEATSTD_NEW(Camera)();
        if ((ptr != 0) && (camera != 0)) {
            const Camera* toClone = 0;
            if (left) {
                toClone = camera->GetLeftEye();
            }
            else {
                toClone = camera->GetRightEye();
            }

            //Camera properties.
            ptr->SetCameraRenderStrategy(toClone->GetCameraRenderStrategy());
            ptr->SetClearMode(toClone->GetClearMode());
            ptr->SetClearingEnabled(toClone->IsClearingEnabled());
            static_cast<void>(ptr->SetLookAtVector(toClone->GetLookAtVector()));
            SharedPointer<Projection> projection = toClone->GetProjection();
            if (projection->IsTypeOf(Candera::StereoProjection::GetTypeId())) {
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                StereoProjection* stereoProjection = static_cast<StereoProjection*>(projection.GetPointerToSharedInstance());
                stereoProjection->SetEyeSeparation(0.0F);
            }
            ptr->SetProjection(projection);
            ptr->SetRenderTarget(0);
            ptr->SetScissoringEnabled(toClone->IsScissoringEnabled());
            ptr->SetScissorRectangle(toClone->GetScissorRectangle());
            ptr->SetSequenceNumber(toClone->GetSequenceNumber());
            ptr->SetSwapEnabled(toClone->IsSwapEnabled());
            static_cast<void>(ptr->SetUpVector(toClone->GetUpVector()));
            ptr->SetViewMatrix(toClone->GetViewMatrix());
            ptr->SetViewingFrustumCullingEnabled(toClone->IsViewingFrustumCullingEnabled());
            ptr->SetViewport(toClone->GetViewport());
            static_cast<void>(ptr->SetLookAtNode(toClone->GetLookAtNode()));

            //Node properties.
            ptr->SetAlphaValue(camera->GetAlphaValue());
            ptr->SetAppearance(toClone->GetAppearance());
            Vector3 minBounds;
            Vector3 maxBounds;
            toClone->GetBoundingBox(minBounds, maxBounds);
            ptr->SetBoundingBox(minBounds, maxBounds);
            ptr->SetCenter(camera->GetCenter());
            ptr->SetGenericTransform(camera->GetGenericTransform());
            ptr->SetIntersectionTestEnabled(camera->IsIntersectionTestEnabled());
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, Candera::Rectangle::SetPosition, False positive because here SetPosition is not called from Candera::Rectangle.)
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::Rectangle = Candera::Camera, False positive because here Candera::Rectangle is not used.)
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, float = struct, False positive because an overloaded function SetPosition is called that accepts Vector3.)
            ptr->SetPosition(camera->GetPosition());
            ptr->SetRadius(camera->GetRadius());
            ptr->SetRenderBenchmark(camera->GetRenderBenchmark());
            ptr->SetRenderingEnabled(camera->IsRenderingEnabled());
            ptr->SetRenderOrderBinAssignment(0);
            ptr->SetRenderOrderRank(camera->GetRenderOrderRank());
            ptr->SetRotation(camera->GetRotation());
            ptr->SetScale(camera->GetScale());
            ptr->SetScopeMask(camera->GetScopeMask());
            for (const Node* child = camera->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                //Deep clone children.
                TreeCloner traverser;
                Node* clone = traverser.CreateClone(*child);
                static_cast<void>(ptr->AddChild(clone));
            }

            CloneRenderOrder(*toClone, *ptr);
        }

        return ptr;
    }

    void Camera::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    Camera::~Camera()
    {
        static_cast<void>(SetLookAtNode(0));

        if (Renderer::GetActiveCamera() == this) {
            Renderer::SetActiveCamera(0);
        }

        m_cameraRenderStrategy = 0;
        m_renderTarget = 0;
        if (m_projection!=0){
            static_cast<void>(m_projection->RemoveProjectionListener(&m_projectionListener));
        }

        if (0 != m_renderOrder) {
            m_renderOrder->Dispose();
        }
        m_internalRenderOrder = 0;
        m_renderOrder = 0;
    }

    Camera* Camera::Clone() const
    {
        return FEATSTD_NEW(Camera)(*this);
    }

    Camera::Camera(const Candera::Camera& camera) :
        Base(camera),
        m_isViewUpToDate(camera.m_isViewUpToDate),
        m_isProjUpToDate(camera.m_isProjUpToDate),
        m_isViewProjUpToDate(camera.m_isViewProjUpToDate),
        m_isViewFrustumUpToDate(camera.m_isViewFrustumUpToDate),
        m_isScissoringEnabled(camera.m_isScissoringEnabled),
        m_isViewingFrustumCullingEnabled(camera.m_isViewingFrustumCullingEnabled),
        m_isSwapEnabled(camera.m_isSwapEnabled),
        m_isClearingEnabled(camera.m_isClearingEnabled),
        m_isCameraEffectiveAlphaEnabled(camera.m_isCameraEffectiveAlphaEnabled),
        m_sequenceNumber(camera.m_sequenceNumber),
        m_internalRenderOrder(0),
        m_renderOrder(0),
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1554, Candera::Camera::m_cameraRenderStrategy, "pause render strategy has external lifetime")
        m_cameraRenderStrategy(0),
        m_renderTarget(0),
        m_lookatVector(camera.m_lookatVector),
        m_upVector(camera.m_upVector),
        m_viewport(camera.m_viewport),
        m_scissorRectangle(camera.m_scissorRectangle),
        m_view(camera.m_view),
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1554, Candera::Camera::m_projection, CANDERA_LINT_REASON_ASSOCIATION)
        m_projection(camera.m_projection),
        m_viewProjection(camera.m_viewProjection),
        m_clearMode(camera.m_clearMode),
        m_projectionListener(),
        m_hash(camera.m_hash)
    {
        m_viewingFrustum.SetCamera(this);
        m_projectionListener.SetCamera(this);
        static_cast<void>(SetLookAtNode(camera.GetLookAtNode()));
        if (m_projection != 0) {
            static_cast<void>(m_projection->AddProjectionListener(&m_projectionListener));
        }

        CloneRenderOrder(camera, *this);
    }

    void Camera::SetProjection(SharedPointer<Projection> projection)
    {
        if (m_projection != 0) {
            static_cast<void>(m_projection->RemoveProjectionListener(&m_projectionListener));
        }
        m_projection = projection;
        if (m_projection != 0) {
            static_cast<void>(m_projection->AddProjectionListener(&m_projectionListener));
        }
        m_isProjUpToDate = true;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;

        NotifyListenersOnProjectionChanged();
    }

    SharedPointer<Projection> Camera::GetProjection() const
    {
        return m_projection;
    }

    void Camera::SetViewMatrix(const Matrix4& viewMatrix)
    {
        m_view = viewMatrix;
        m_isViewUpToDate = true;

        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;

        NotifyListenersOnViewChanged();
    }

    const Matrix4& Camera::GetViewProjectionMatrix() const
    {
        if (!IsViewProjUpToDate()) {
            UpdateViewProjection();
        }
        return m_viewProjection;
    }

    const ViewingFrustum& Camera::GetViewingFrustum() const
    {
        if (!IsViewingFrustumUpToDate()) {
            m_viewingFrustum.Recalculate();
            m_isViewFrustumUpToDate = true;
        }
        return m_viewingFrustum;
    }

    static bool IsSingular(const Vector3& v)
    {
        const Float tooShortSquared = 0.0000001F;
        return v.GetSquaredLength() < tooShortSquared;
    }

    // v1 and v2 both assumed to be normalized
    static bool AreCollinear(const Vector3& v1, const Vector3& v2)
    {
        return IsSingular(v1 - v2) || IsSingular(v1 + v2);
    }

    bool Camera::RotateAroundAxis(const Vector3& axis, Float angleDeg, Vector3& binormal) const
    {
        if (IsSingular(axis)) {
            FEATSTD_LOG_ERROR("Rotate around axis failed, axis is singular.");
            return false;
        }
        Matrix4 rot;
        rot.SetRotationAxis(angleDeg, axis.GetX(), axis.GetY(), axis.GetZ());
        binormal.TransformCoordinate(rot);
        return true;
    }

    bool Camera::SetLookAtVector(const Vector3& lookAt)
    {
        if (IsSingular(lookAt)) {
            FEATSTD_LOG_ERROR("Set lookAt vector failed, lookAt param is singular.");
            return false;
        }

        Vector3 newLookAtNormalized = lookAt;
        static_cast<void>(newLookAtNormalized.Normalize());

        if (AreCollinear(newLookAtNormalized, m_upVector)) {
            FEATSTD_LOG_ERROR("New lookAt almost collinear with up vector.");
            const Float dp = m_upVector.GetDotProduct(newLookAtNormalized);
            const bool sameDirection = dp > 0.0F;
            if (sameDirection) {
                m_upVector = -m_lookatVector;
            }
            else {
                m_upVector = m_lookatVector;
            }
        }
        m_lookatVector = newLookAtNormalized;

        m_isViewUpToDate = false;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;

        MakeUpOrthogonalToLookat();

        NotifyListenersOnViewChanged();
        return true;
    }

    const Vector3& Camera::GetLookAtVector() const
    {
        return m_lookatVector;
    }

    Vector3 Camera::GetWorldLookAtVector() const
    {
        Vector3 worldLookat = m_lookatVector;
        worldLookat.TransformCoordinate(GetWorldRotation());
        return worldLookat;
    }

    void Camera::MakeUpOrthogonalToLookat()
    {
        Vector3 right = GetRightVector();
        Vector3 orthoUp = right.GetCrossProduct(m_lookatVector);
        static_cast<void>(orthoUp.Normalize());
        m_upVector = orthoUp;
    }

    bool Camera::SetUpVector(const Vector3& up)
    {
        if (IsSingular(up)) {
            FEATSTD_LOG_ERROR("Set UpVector failed, param is singular.");
            return false;
        }

        Vector3 newUpNormalized = up;
        static_cast<void>(newUpNormalized.Normalize());

        if (AreCollinear(m_lookatVector, newUpNormalized)) {
            FEATSTD_LOG_ERROR("New up almost collinear with lookat vector.");
            return false;
        }

        m_isViewUpToDate = false;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;

        m_upVector = newUpNormalized;
        MakeUpOrthogonalToLookat();

        NotifyListenersOnViewChanged();
        return true;
    }

    const Vector3& Camera::GetUpVector() const
    {
        return m_upVector;
    }

    Vector3 Camera::GetWorldUpVector() const
    {
        Vector3 worldUp = m_upVector;
        worldUp.TransformCoordinate(GetWorldRotation());
        return worldUp;
    }

    Vector3 Camera::GetRightVector() const
    {
        Vector3 right = m_lookatVector.GetCrossProduct(m_upVector);
        static_cast<void>(right.Normalize());
        return right;
    }

    Vector3 Camera::GetWorldRightVector() const
    {
        Vector3 worldRight = GetRightVector();
        worldRight.TransformCoordinate(GetWorldRotation());
        return worldRight;
    }

    bool Camera::SetUpAndLookAtVectors(const Vector3& up, const Vector3& lookAt)
    {
        if (IsSingular(lookAt)) {
            FEATSTD_LOG_ERROR("Set up and lookAt vectors failed, lookAt param is singular.");
            return false;
        }

        if (IsSingular(up)) {
            FEATSTD_LOG_ERROR("Set up and lookAt vectors failed, up param is singular.");
            return false;
        }

        Vector3 normalizedUp = up;
        static_cast<void>(normalizedUp.Normalize());
        Vector3 normalizedLookAt = lookAt;
        static_cast<void>(normalizedLookAt.Normalize());

        if (AreCollinear(normalizedLookAt, normalizedUp)) {
            FEATSTD_LOG_ERROR("New up vector almost collinear with new lookat vector.");
            return false;
        }

        m_isViewUpToDate = false;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;

        m_upVector = normalizedUp;
        m_lookatVector = normalizedLookAt;
        MakeUpOrthogonalToLookat();

        NotifyListenersOnViewChanged();
        return true;
    }

    bool Camera::LookAtWorldPoint(const Vector3& targetPoint)
    {
        Matrix4 worldRotationInverted = GetWorldRotation();
        worldRotationInverted.Inverse();
        Vector3 lookat = targetPoint - GetWorldPosition();
        lookat.TransformCoordinate(worldRotationInverted);
        return SetLookAtVector(lookat);
    }

    bool Camera::RotateAroundWorldAxis(const Vector3& axis, Float angleDegrees)
    {
        Vector3 localAxis = axis;
        Matrix4 worldRotationInverted = GetWorldRotation();
        worldRotationInverted.Inverse();
        localAxis.TransformCoordinate(worldRotationInverted);
        bool rotOk = RotateAroundAxis(localAxis, -angleDegrees, m_lookatVector);
        rotOk = rotOk && RotateAroundAxis(localAxis, -angleDegrees, m_upVector);
        m_isViewUpToDate = false;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;
        NotifyListenersOnViewChanged();
        return rotOk;
    }

    const Matrix4& Camera::GetViewMatrix() const
    {
        if (! IsViewUpToDate()) {
            RecalculateView();
        }
        return m_view;
    }

    bool Camera::Activate()
    {
        return Renderer::ActivateCamera(this);
    }

    bool Camera::Deactivate()
    {
        return Renderer::DeactivateCamera(this);
    }

    void Camera::SetSequenceNumber(Int32 sequenceNumber)
    {
        m_sequenceNumber = sequenceNumber;
        if (GetScene() != 0) {
            static_cast<void>(Renderer::RemoveCamera(this));
            static_cast<void>(Renderer::AddCamera(this));
        }
    }

    Int32 Camera::GetSequenceNumber() const
    {
        return m_sequenceNumber;
    }

    void Camera::OnAncestorAdded(Scene* scene)
    {
        if (scene != 0) {
            static_cast<void>(Renderer::AddCamera(this));
        }
    }

    void Camera::OnAncestorRemoved(Scene* /*scene*/)
    {
        static_cast<void>(Renderer::RemoveCamera(this));
    }

    void Camera::OnAncestorTransformChanged()
    {
        m_isViewUpToDate = false;
        m_isViewProjUpToDate = false;
        m_isViewFrustumUpToDate = false;
        NotifyListenersOnViewChanged();
    }

    bool Camera::AddCameraListener(CameraListener* listener)
    {
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_cameraListeners.Add(listener);
        }
        return isSuccessful;
    }

    bool Camera::RemoveCameraListener(CameraListener* listener)
    {
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_cameraListeners.Remove(listener);
        }
        return isSuccessful;
    }

    class Camera::OnPreRenderEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPreRenderEvent(Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPreRender(m_camera);
        }
    private:
        Camera* m_camera;
    };


    void Camera::NotifyListenersOnPreRender()
    {
        OnPreRenderEvent event(this);
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnPostRenderEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPostRenderEvent(Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPostRender(m_camera);
        }
    private:
        Camera* m_camera;
    };

    void Camera::NotifyListenersOnPostRender()
    {
        OnPostRenderEvent event(this);
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnProjectionChangedEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnProjectionChangedEvent(Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnProjectionChanged(m_camera);
        }
    private:
        Camera* m_camera;
    };

    void Camera::NotifyListenersOnProjectionChanged()
    {
        OnProjectionChangedEvent event(this);
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnViewChangedEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnViewChangedEvent(Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnViewChanged(m_camera);
        }
    private:
        Camera* m_camera;
    };

    void Camera::NotifyListenersOnViewChanged()
    {
        OnViewChangedEvent event(this);
        m_cameraListeners.Iterate(event);
    }



    class Camera::OnRenderTargetChangedEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnRenderTargetChangedEvent(Camera* camera, RenderTarget3D* renderTarget) : m_camera(camera), m_renderTarget(renderTarget) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnRenderTargetChanged(m_camera, m_renderTarget);
        }
    private:
        Camera* m_camera;
        RenderTarget3D* m_renderTarget;
    };

    void Camera::NotifyListenersOnRenderTargetChanged(RenderTarget3D* previousRenderTarget)
    {
        OnRenderTargetChangedEvent event(this, previousRenderTarget);
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnRenderStrategyChangedEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnRenderStrategyChangedEvent(Camera* camera, CameraRenderStrategy* renderStrategy) : m_camera(camera), m_renderStrategy(renderStrategy) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnCameraRenderStrategyChanged(m_camera, m_renderStrategy);
        }
    private:
        Camera* m_camera;
        CameraRenderStrategy* m_renderStrategy;
    };

    void Camera::NotifyListenersOnRenderStrategyChanged(CameraRenderStrategy* previousRenderStrategy)
    {
        OnRenderStrategyChangedEvent event(this, previousRenderStrategy);
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnPreActivateEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPreActivateEvent(const Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPreActivate(m_camera);
        }
    private:
        const Camera* m_camera;
    };

    void Camera::NotifyListenersOnPreActivate()
    {
        OnPreActivateEvent event(const_cast<const Camera*>(this));
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnPostActivateEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPostActivateEvent(const Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPostActivate(m_camera);
        }
    private:
        const Camera* m_camera;
    };

    void Camera::NotifyListenersOnPostActivate()
    {
        OnPostActivateEvent event(const_cast<const Camera*>(this));
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnPreClearEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPreClearEvent(const Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPreClear(m_camera);
        }
    private:
        const Camera* m_camera;
    };

    void Camera::NotifyListenersOnPreClear()
    {
        OnPreClearEvent event(const_cast<const Camera*>(this));
        m_cameraListeners.Iterate(event);
    }

    class Camera::OnPostClearEvent : public Camera::CameraListenerContainer::Event
    {
    public:
        OnPostClearEvent(const Camera* camera) : m_camera(camera) {}
        virtual void Notify(CameraListener* listener) override
        {
            listener->OnPostClear(m_camera);
        }
    private:
        const Camera* m_camera;
    };

    void Camera::NotifyListenersOnPostClear()
    {
        OnPostClearEvent event(const_cast<const Camera*>(this));
        m_cameraListeners.Iterate(event);
    }

    bool Camera::WasUpdated() const
    {
        return ComputeHash();
    }

    bool Camera::ComputeHash() const
    {
       using namespace FeatStd;

        UInt32 hash = Hash::cInitialHashValue;
        UInt32 seed = Hash::cInitialSeed;

        do {
            if (m_isClearingEnabled)
            {
                // Add clear mode
                m_clearMode.UpdateHash(hash);
            }

            // Add viewport rectangle
            static_cast<void>(Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(&m_viewport),
                reinterpret_cast<const UInt8*>(&m_viewport) + sizeof(m_viewport),
                hash));

            // Add scissoring rectangle
            if (m_isScissoringEnabled)
            {
                static_cast<void>(Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(&m_scissorRectangle),
                    reinterpret_cast<const UInt8*>(&m_scissorRectangle) + sizeof(m_scissorRectangle),
                    hash));
            }
        } while (!Hash::FinishBinBufferHash(hash, seed));

        if (hash != m_hash)
        {
            m_hash = hash;
            return true;
        }

        return false;
    }

    void Camera::RecalculateView() const
    {
        const Vector3 eye = GetWorldPosition();

        Vector3 forward = m_lookatVector;
        Vector3 up = m_upVector;
        Vector3 right = GetRightVector();

        Matrix4 worldRotation = GetWorldRotation();
        forward.TransformCoordinate(worldRotation);
        up.TransformCoordinate(worldRotation);
        right.TransformCoordinate(worldRotation);

        Matrix4 m4;

        m4.Set(0, 0, right.GetX());
        m4.Set(1, 0, right.GetY());
        m4.Set(2, 0, right.GetZ());

        m4.Set(0, 1, up.GetX());
        m4.Set(1, 1, up.GetY());
        m4.Set(2, 1, up.GetZ());

        m4.Set(0, 2, -forward.GetX());
        m4.Set(1, 2, -forward.GetY());
        m4.Set(2, 2, -forward.GetZ());

        Matrix4 translation;
        translation.SetTranslation(-eye.GetX(), -eye.GetY(), -eye.GetZ());

        m_view = translation * m4;

        m_isViewUpToDate = true;
    }

    void Camera::UpdateViewProjection() const
    {
        if (m_projection != 0) {
            m_viewProjection = GetViewMatrix() * m_projection->GetProjectionMatrix();
        }
        m_isViewProjUpToDate = true;
    }

    bool Camera::Upload()
    {
        return (m_clearMode.GetSkyBox() != 0) ? (m_clearMode.GetSkyBox()->Upload()) : true;
    }

    bool Camera::Unload()
    {
        return (m_clearMode.GetSkyBox() != 0) ? (m_clearMode.GetSkyBox()->Unload()) : true;
    }

    FEATSTD_RTTI_DEFINITION(Camera, Node)

    void Camera::SetRenderTarget(RenderTarget3D* renderTarget)
    {
        RenderTarget3D* previousRenderTarget = m_renderTarget;
        m_renderTarget = renderTarget;
        NotifyListenersOnRenderTargetChanged(previousRenderTarget);
    }

    void Camera::SetCameraRenderStrategy(CameraRenderStrategy* cameraRenderStrategy)
    {
        if (m_cameraRenderStrategy != cameraRenderStrategy) {
            CameraRenderStrategy* previousRenderStrategy = m_cameraRenderStrategy;
            m_cameraRenderStrategy = cameraRenderStrategy;
            NotifyListenersOnRenderStrategyChanged(previousRenderStrategy);
        }
    }

    void Camera::SetRenderOrder(AbstractRenderOrder* renderOrder)
    {
        if (m_renderOrder == renderOrder) {
            return;
        }

        if (0 != renderOrder) {
            if (m_internalRenderOrder == m_renderOrder) {
                FEATSTD_DEBUG_ASSERT(0 != m_internalRenderOrder);
                m_internalRenderOrder->Dispose();
                m_internalRenderOrder = 0;
            }

            m_renderOrder = renderOrder;
        }
        else {
            if (m_internalRenderOrder != m_renderOrder) {
                CreateInternalRenderOrder();
            }
        }
    }

    void Camera::CreateInternalRenderOrder()
    {
        m_internalRenderOrder = RenderOrder::Create(300, 30);
        m_renderOrder = m_internalRenderOrder;
    }

    void Camera::CloneRenderOrder(const Camera& source, Camera& destination)
    {
        destination.m_internalRenderOrder = 0;
        destination.m_renderOrder = source.m_renderOrder->Clone();
        if (source.m_internalRenderOrder == source.m_renderOrder) {
            destination.m_internalRenderOrder = destination.m_renderOrder;
        }
    }

    class AssignToRenderOrderBinTraverser : public TreeTraverser {
        typedef TreeTraverser Base;

    public:
        AssignToRenderOrderBinTraverser(AbstractRenderOrder* renderOrder, const Camera* camera, const Rectangle& dirtyArea)
            :
            Base(),
            m_renderOrder(renderOrder),
            m_camera(camera),
            m_dirtyArea(dirtyArea),
            m_useDirtyArea((m_dirtyArea.GetLeft() != 0.0F) || (m_dirtyArea.GetTop() != 0.0F) || (m_dirtyArea.GetWidth() != 1.0F) || (m_dirtyArea.GetHeight() != 1.0F))
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            if (!node.IsRenderingEnabled()) {
                return StopTraversingForDescendants;
            }

            // Visibility Culling. Cull Nodes that are not in scope of Camera.
            // This is less expensive than frustum culling, so we do it fist.
            if (!node.IsInScopeOf(m_camera->GetScopeMask())) {
                return ProceedTraversing;
            }

            // Viewing Frustum Culling.
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::AssignToRenderOrderBinTraverser::m_camera, CANDERA_LINT_REASON_OPERANDNOTNULL)
                if (m_camera->IsViewingFrustumCullingEnabled() && node.IsBoundingSphereValid()) {
                    if (!m_camera->GetViewingFrustum().IsSphereInFrustum(node.GetWorldCenter(), node.GetWorldRadius())) {
                        return ProceedTraversing;
                    }
                }

            // Cull projected world OBB against user specified dirty area (includes camera's scissor rectangle)
            if (m_useDirtyArea && node.IsBoundingBoxValid()) {
                // Transform world oriented bounding box to to normalized device coordinates [-1..1],
                // and determine its bounding rectangle.
                const UInt boundingVertexCount = 8;
                Vector3 boundingBox[boundingVertexCount];
                node.GetWorldOrientedBoundingBox(boundingBox);
                const Matrix4& viewProjectionMatrix = m_camera->GetViewProjectionMatrix();
                Float minX = Math::MaxFloat();
                Float maxX = -Math::MaxFloat();
                Float minY = Math::MaxFloat();
                Float maxY = -Math::MaxFloat();
                for (UInt i = 0; i < boundingVertexCount; ++i) {
                    boundingBox[i].TransformCoordinate(viewProjectionMatrix);
                    if (minX > boundingBox[i].GetX()) {
                        minX = boundingBox[i].GetX();
                    }

                    if (maxX < boundingBox[i].GetX()) {
                        maxX = boundingBox[i].GetX();
                    }

                    if (minY > boundingBox[i].GetY()) {
                        minY = boundingBox[i].GetY();
                    }

                    if (maxY < boundingBox[i].GetY()) {
                        maxY = boundingBox[i].GetY();
                    }
                }

                // Transform normalized device coordinates [-1..1] to normalized screen coordinates [0..1].
                // (Requires swapping Top and Bottom)
                Float left = (minX + 1.0F) * 0.5F;
                Float right = (maxX + 1.0F) * 0.5F;
                Float top = 1.0F - ((maxY + 1.0F) * 0.5F);
                Float bottom = 1.0F - ((minY + 1.0F) * 0.5F);

                // Transform from viewport space to screen space
                const Rectangle& viewport = m_camera->GetViewport();
                left = left * viewport.GetWidth() + viewport.GetLeft();
                right = right * viewport.GetWidth() + viewport.GetLeft();
                top = top * viewport.GetHeight() + viewport.GetTop();
                bottom = bottom * viewport.GetHeight() + viewport.GetTop();

                // Cull screen space bounding rectangle against dirty area
                Rectangle boundingRectangle(left, top, right - left, bottom - top);
                boundingRectangle.Intersect(m_dirtyArea);
                if (0.0F >= (boundingRectangle.GetHeight() * boundingRectangle.GetWidth())) {
                    Renderer::Statistics& statistics = const_cast<Renderer::Statistics&>(Renderer::GetStatistics());
                    ++statistics.CulledNodesUsingDirtyAreaScissor;
                    return ProceedTraversing;
                }
            }

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::AssignToRenderOrderBinTraverser::m_renderOrder, CANDERA_LINT_REASON_OPERANDNOTNULL)
                static_cast<void>(m_renderOrder->AssignNodeToBin(&node));
            return ProceedTraversing;
        }

    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(AssignToRenderOrderBinTraverser); // eliminates warning (AbstractRenderOrder is not assignable)
        AbstractRenderOrder* m_renderOrder;
        const Camera* m_camera;
        const Rectangle& m_dirtyArea;
        bool m_useDirtyArea;
    };

    void Camera::PrepareRenderOrder(const Rectangle& dirtyArea)
    {
        Scene* const scene = GetScene();
        if (0 == scene) {
            return;
        }

        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::PrepareRenderOrder));
        m_renderOrder->Clear();
        AssignToRenderOrderBinTraverser assignTraverser(m_renderOrder, this, dirtyArea);
        assignTraverser.Traverse(*scene);
        m_renderOrder->SortBins();
    }

    bool Camera::SetLookAtNode(Node* lookAtNode)
    {
        Node* currentLookAtNode = GetLookAtNode();
        if (currentLookAtNode == lookAtNode) {
            return true;
        }
        ControllerSystem* controllerSystem = EntityComponentSystem::EntitySystem::Get<ControllerSystem>();
        if (0 == controllerSystem) {
            return false;
        }
        ControllerSystem::Handle controllerComponent = EntityComponentSystem::EntitySystem::GetDerivedComponent<Candera::ControllerComponent, CameraLookAtNodeController>(this);
        if (0 != lookAtNode) {
            if (controllerComponent.IsNullHandle()) {
                controllerComponent = controllerSystem->CreateComponent<CameraLookAtNodeController>();
                if (controllerComponent.IsNullHandle()) {
                    return false;
                }
                bool success = controllerSystem->AttachComponent(controllerComponent, this);
                if (!success) {
                    return false;
                }
            }
            return SetValue(CdaDynamicPropertyInstance(LookAtNode), lookAtNode);
        }
        else {
            bool success = SetValue(CdaDynamicPropertyInstance(LookAtNode), lookAtNode);
            success = controllerSystem->DestroyComponent(controllerComponent) && success;
            return success;
        }
    }

} // namespace Candera
