//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BITMAPTEXTURE_IMAGE_H)
#define CANDERA_BITMAPTEXTURE_IMAGE_H

#include <Candera/Engine3D/Core/TextureImage.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera {
    /** @addtogroup Core3D
     *  @{
     */

    class Texture;

    /**
    * @brief  TextureImage encapsulates a VRAM handle which is retrieved when uploading an image to the vram.
    *         An instance holds a associated pointer to a bitmap, which holds pixel data and meta information about this data.
    *         The TextureImage implements a reference counting mechanism for handling multiple upload and unloads. Only
    *         If the reference is 0 the it will upload, and only if it is 0 and upload it will unload.
    *
    *         Multiple textures can share a textureImage, in this case multiple uploads will be prevented.
    *         This can be used to implement a specific texture upload strategy. The user can create instances
    *         of textureImages objects and upload it manually by calling Upload.
    *         If a texture is uploaded it also triggers the upload of the associated textureImage.
    *
    *         Keep following in mind:
    *         1. Only share TextureImages with the same mip mapping setting is used.
    *         2. When you render on a textureImage in vram all textures with use this textureImage will get a changed textureaImage.
    *            Only share if you want the changes to be shared too.
    *         3. The associated bitmap pointer can get null, the memory of this is not handled by the textureImage. If a texture image is unloaded
    *            and has to be restored in vram memory, this pointer must be valid so that an upload can happen.
     */
    class BitmapTextureImage : public TextureImage, public FeatStd::Diagnostics::Measurable {
            FEATSTD_TYPEDEF_BASE(TextureImage);

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(BitmapTextureImage);

            /**
             *  Creates an instance of this class.
             *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
             */
            static BitmapTextureImage::SharedPointer Create();

            /**
             *  Destructor
             */
            virtual ~BitmapTextureImage() override;

            /**
             *  Updates data within an already uploaded TextureImage directly in VRAM.
             *  The source data is taken from the attached Bitmap in the TextureImage.
             *  All mip map levels are updated (if any).
             *  @return True if updating this BitmapTextureImage and setting every texture sub image succeeded. False otherwise.
             */
            bool Update() const;

            /**
             *  Updates data within an already uploaded TextureImage directly in VRAM.
             *  @param xOffset     The x-Offset inside the Texture.
             *  @param yOffset     The y-Offset inside the Texture.
             *  @param width       The width to update. The width relates also to the width of the source data.
             *  @param height      The height to update. The height relates also to the height of the source data.
             *  @param data        The source data to update. The source data must be in continuous memory.
             *  @param level       The mip map level to update (in case predefined mip mapping is used), default is first.
             *  @param unit        The texture unit to update.
             *  @param compressedSize The size of the data array in bytes, needed for updating compressed images.
             *                     Note: Function will fail if image is compressed and size is wrong.
             *  @return            True if updating every texture sub image succeeded. False if updating failed, texture isn't uploaded,
             *                     no data is passed. Also fails when mip mapping is disabled and a level > 0 is passed.
             */
            bool Update(Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt level = 0, UInt unit = 0, UInt compressedSize = 0) const;

            /**
             *  Copy content from the current frame buffer to the TextureImage in VRAM.
             *  @param xTex               The x-Offset inside the Texture.
             *  @param yTex               The y-Offset inside the Texture.
             *  @param xFramebuffer       The x-Offset inside the frame buffer.
             *  @param yFramebuffer       The y-Offset inside the frame buffer.
             *  @param width              The width to copy.
             *  @param height             The height to copy.
             *  @return                   True if current frame buffer was copied to the TextureImage in VRAM. False otherwise.
             */
            bool CopyFromFramebuffer(Int32 xTex, Int32 yTex, Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height) const;

            /**
             *  Sets the Bitmap and the corresponding Disposer for the Bitmap.
             *  Call this function before Upload. A bitmap must be set.
             *  It also also possible to pass a list of bitmaps for predefined mip-mapping.
             *  @param bitmap             The Bitmap or list of Bitmaps which is uploaded by the TextureImage.
             *                            If a list of Bitmaps is passed then pass here the head of the list.
             *  @return                   False if BitmapTextureImage is already uploaded, modification of Bitmaps after uploading is not allowed.
             *                            True if Bitmap is set.
             */
            bool SetBitmap(const Bitmap::SharedPointer& bitmap);

            /**
             *  Retrieves bitmap used by this Texture object.
             *  @return Pointer to bitmap used by this BitmapTextureImage.
             */
            const Bitmap::SharedPointer& GetBitmap() const { return m_bitmap; }

            /**
             *  Retrieves the Bitmap used by this Texture object if it is mutable.
             *  @return  The Bitmap if it is mutable (Bitmap::IsMutable() == true), otherwise NULL.
             */
            Bitmap::SharedPointer GetMutableBitmap();

            /**
             *  Retrieve texture handle of graphic device, associated with the active resource pool.
             *  @return the texture handle of graphic device associated with the active pool.
             *  @return Texture handle of graphic device.
             */
            virtual Handle GetVideoMemoryHandle() const override;

            /**
             *  Defines if MipMap-chain shall be attached to TextureImage.
             *  If MipMapping is enabled with one single Bitmap applied only, the MipMap-chain is generated by driver automatically.
             *  If MipMapping is enabled with more than one Bitmap applied (See Bitmap::SetNext), then the MipMap-chain
             *  includes the Bitmaps of the predefined Bitmap chain.
             *  Call this function before Texture::Upload() to achieve desired effect, see return param for further details.
             *  @param   enableMipMapping defines if MipMap chain will be generated or not if Texture will be uploaded.
             *  @return  true, if @param enableMipMapping is applied when the Texture will be uploaded.
             *  @return  false, if @param enableMipMapping is ignored, as Texture has already been uploaded.
             */
            bool SetMipMappingEnabled(bool enableMipMapping);

            /**
             *  Retrieve if MipMapping is enabled or not
             *  @return   true, if mipmapping is enabled and mipmap filters are evaluated.
             *  @return   false, if mipmapping is disabled and mipmap filters are ignored.
             */
            virtual bool IsMipMappingEnabled() const override {
                return m_isMipMappingEnabled;
            }

            /**
             *  Retrieves ImageSource3D of this BitmapTextureImage. ImageSource3D contains videohandle and if mipmapping is enabled.
             *  @return ImageSource3D of this BitmapTextureImage.
             */
            virtual ImageSource3D* ToImageSource3D() override;

            /**
             * Retrieves the texture target type of this TextureImage.
             * @return TextureTargetType is in case of BitmapTextureImages always Texture2D.
             */
            virtual TextureTargetType GetTextureTargetType() const override{
                return TextureImage::Texture2D;
            }

            /**
             *  Overridden function of Measurable.
             *  @return estimate of the video memory used by the bitmap.
             */
            virtual UInt GetSize() const override;

            FEATSTD_RTTI_DECLARATION();

        protected:
            /**
             *  overridden from DeviceObject
             *  @return True if uploading of TextureImage succeeded. False otherwise.
             */
            virtual bool UploadInternal(LoadingHint loadingHint) override;
            /**
             *  overridden from DeviceObject
             *  @return True if unloading of TextureImage succeeded. False otherwise.
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) override;

            // override DeviceObject::DisposeInternal
            virtual void DisposeInternal() override;

        private:
            friend class RenderDevice;
            friend class Renderer;

            /**
             *  Wrapper class which is exposed by ToImageSource3D function.
             */
            class ImageSource3DInstance : public ImageSource3D {
                    friend class BitmapTextureImage;

                public:
                    ImageSource3DInstance(): m_bitmapTextureImage(0) { }
                    ~ImageSource3DInstance() {
                        m_bitmapTextureImage = 0;
                    }

                    virtual Handle GetVideoMemoryHandle() const override{
                        return (m_bitmapTextureImage != 0) ? m_bitmapTextureImage->GetVideoMemoryHandle() : 0;
                    }
                    virtual bool IsMipMappingEnabled() const override{
                        return (m_bitmapTextureImage != 0) ? m_bitmapTextureImage->IsMipMappingEnabled() : false;
                    }
                    virtual Int GetHeight() const override{
                        return (m_bitmapTextureImage != 0) ? static_cast<Int>(m_bitmapTextureImage->GetBitmap()->GetHeight()) : 0;
                    }
                    virtual Int GetWidth() const override{
                        return (m_bitmapTextureImage != 0) ? static_cast<Int>(m_bitmapTextureImage->GetBitmap()->GetWidth()) : 0;
                    }
                    virtual void Sync() override {}
                    virtual void WaitSync() override {}

                private:
                    const BitmapTextureImage* m_bitmapTextureImage;

                    void SetBitmapTextureImage(const BitmapTextureImage* image) {
                        m_bitmapTextureImage = image;
                    }
            };

            bool m_isMipMappingEnabled; // Defines if mipmapping is enabled or not. Only the value that was set before upload is considered.
            Bitmap::SharedPointer m_bitmap; // Bitmap that stores the image data for this TextureImage object.
            Handle m_videoMemoryHandle[CANDERA_MAX_CONTEXT_COUNT]; // Texture handle of graphics device.
            // ImageSource3DInstance is exposed by ToImageSource3D function.
            ImageSource3DInstance m_imageSource3DInstance;

            // Private because only ::Create() should be used to create an instance of this class.
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::BitmapTextureImage::BitmapTextureImage, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
            BitmapTextureImage();
            BitmapTextureImage(const BitmapTextureImage& material);
            BitmapTextureImage& operator = (const BitmapTextureImage& material);

            /**
             *  Sets the texture handle of graphic device.
             *  @param handle the texture handle
             */
            void SetVideoMemoryHandle(Handle handle);
    };
    /** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_TEXTURE_IMAGE_H
