//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapTextureImage.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <Candera/Engine3D/Core/Renderer.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    BitmapTextureImage::BitmapTextureImage() :
        Base(),
        m_isMipMappingEnabled(false),
        m_bitmap(0)
    {
        MemoryPlatform::Set(m_videoMemoryHandle, 0, sizeof(m_videoMemoryHandle));
        m_imageSource3DInstance.SetBitmapTextureImage(this);
    }

    SharedPointer<BitmapTextureImage> BitmapTextureImage::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        BitmapTextureImage* ptr = FEATSTD_NEW(BitmapTextureImage)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("Texture image create failed, out of memory.");
        }
        BitmapTextureImage::SharedPointer sharedPointer(ptr);
        return sharedPointer;
    }

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1579, "bitmap data is disposed by the disposer; the disposer has external lifetime")
    BitmapTextureImage::~BitmapTextureImage()
    {
        static_cast<void>(Unload(ForceAll));
    }

    bool BitmapTextureImage::SetMipMappingEnabled(bool enableMipMapping)
    {
        bool isNotUploaded = !IsUploaded();
        if (isNotUploaded) {
            m_isMipMappingEnabled = enableMipMapping;
        }
        return isNotUploaded;
    }

    bool BitmapTextureImage::Update() const
    {
        bool isSuccessful = IsUploaded() && (m_bitmap != 0);
        if (isSuccessful) {
            Bitmap* current = m_bitmap.GetPointerToSharedInstance();
            UInt level = 0;
            while (current != 0) {
                Bitmap* next = current->GetNext().GetPointerToSharedInstance();
                Bitmap::PixelsResource pixelsResource(current->GetPixelsResourceHandle());
                const UInt8* data = pixelsResource.GetData();
                isSuccessful = isSuccessful && (data != 0) && RenderDevice::SetTextureSubimage(*this, level, 0, 0, current->GetWidth(), current->GetHeight(), data);
                ++level;
                current = next;
            }
        }
        return isSuccessful;
    }

    bool BitmapTextureImage::Update(Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data, UInt level, UInt unit, UInt compressedSize) const
    {
        bool result = false;
        if ((!m_isMipMappingEnabled) && (level > 0)) {
            return result;
        }
        if (IsUploaded() && (data != 0)) {
            result = RenderDevice::SetTextureSubimage(*this, level, xOffset, yOffset, width, height, data, unit, compressedSize);
        }
        return result;
    }

    bool BitmapTextureImage::CopyFromFramebuffer(Int32 xTex, Int32 yTex, Int32 xFramebuffer, Int32 yFramebuffer, Int32 width, Int32 height) const
    {
        //xxx todo what if width, height are not powers of two (which actual implementations may require)? then several copy operations are needed
        return RenderDevice::CopyFromFramebufferToTexture(*this, xTex, yTex, xFramebuffer, yFramebuffer, width, height);
    }

    bool BitmapTextureImage::SetBitmap(const Bitmap::SharedPointer& bitmap)
    {
        if (IsUploaded()) {
            return false; // bitmap data is already uploaded; modification after upload not allowed
        }
        m_bitmap = bitmap;
        return true;
    }

    bool BitmapTextureImage::UploadInternal(LoadingHint loadingHint)
    {
        return Renderer::UploadBitmapTextureImage(*this, 0, loadingHint);
    }

    bool BitmapTextureImage::UnloadInternal(LoadingHint loadingHint)
    {
        return Renderer::UnloadBitmapTextureImage(*this, loadingHint);
    }

    void BitmapTextureImage::DisposeInternal()
    {
        // Dispose a possible MipMap chain.
        Bitmap* current = m_bitmap.GetPointerToSharedInstance();
        while (current != 0) {
            current->DisposePixels();
            current = current->GetNext().GetPointerToSharedInstance();
        }
    }

    Bitmap::SharedPointer BitmapTextureImage::GetMutableBitmap()
    {
        return ((m_bitmap != 0) && (m_bitmap->GetPixelsResourceHandle().m_isMutable == 1)) ? m_bitmap : Bitmap::SharedPointer();
    }

    ImageSource3D* BitmapTextureImage::ToImageSource3D()
    {
        return &m_imageSource3DInstance;
    }

    UInt BitmapTextureImage::GetSize() const
    {
        UInt32 textureImageSize = 0;
        if (m_bitmap != 0) {
            //Calculate bytes per color channel (texel) in bytes based on type. (All texel formats are assembled into an RGBA element by GL.)
            textureImageSize = RenderDevice::GetSize(m_bitmap.GetPointerToSharedInstance(), GetTextureMemoryPool());

            if (IsMipMappingEnabled()) {
                if (m_bitmap->HasNext() != 0) {
                    Bitmap::SharedPointer bitmap = m_bitmap->GetNext();
                    while (bitmap != 0) {
                        textureImageSize += RenderDevice::GetSize(m_bitmap.GetPointerToSharedInstance(), GetTextureMemoryPool());
                        bitmap = bitmap->GetNext();
                    }
                }
                else {
                    if ((!m_bitmap->IsCompressed()) && (textureImageSize > 0)) {
                        // A mipmap chain must conclude with a 1x1 sized level in OpenGL ES and has max. 1/3 more memory consumption than the greatest image.
                        textureImageSize = ((4 * textureImageSize) - 1) / 3;
                    }
                }
            }
        }

        return static_cast<UInt>(textureImageSize);
    }

    Handle BitmapTextureImage::GetVideoMemoryHandle() const
    {
        return m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
    }

    void BitmapTextureImage::SetVideoMemoryHandle(Handle handle)
    {
        m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
    }

    FEATSTD_RTTI_DEFINITION(BitmapTextureImage, Base)
} // namespace Candera
