//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Billboard_H)
#define CANDERA_Billboard_H

#include <Candera/Engine3D/Core/Renderable.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/Environment.h>
#include <Candera/Macros.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   The class Billboard represents a Plane that can align towards camera automatically, according to the alignment type set.
 *          Thus, Billboards are quite often used as "imposters" pretending to be a 3D geometry by showing a 2D image that is always facing the camera.
 *          In order to relief distinction between Billboard and PointSprite see following comparison:
 *            + Billboards support rectangular dimension and non-uniform scale,
 *            + Billboards support different rotation techniques (see Alignment), whereas PointSprites are aligned according to CenterEyeAlignment, exclusively.
 *            + Billboards take local rotation and scale into account (see class Transformable), whereas PointSprites ignore Transformable parameters others than position.
 *            + PointSprites have a performance advantage due to less geometry (one instead of 4 vertices).
 *          Overall recommendation: Use PointSprites for spherical shapes like particles, lens flares, sparkles, dust which are screen aligned (see Billboard::CameraUpAlignment).
 *          Further, use Billboards for world up oriented clouds, text, or yaw axis aligned trees, and signs, etc.
 *          Note: The Billboard - because of absent normals - does not support lighting.
 *          Belows sketch depicts the layout of a Billboard.
 *                         W
 *              U:0,V:1 +-----+ U:1,V:1
 *                    H |  X  |           Legend: texture coordinates: U,V; Width: W, Height: H, Local Center: X
 *              U:0,V:0 +-----+ U:1,V:0
 */

class Billboard : public Renderable
{

    FEATSTD_TYPEDEF_BASE(Renderable);

    public:
        /**
         *  The enumerator Alignment defines how the Billboard object is aligned when facing the camera.
         */
        enum Alignment {
            CameraUpAlignment,   ///< Regardless of camera rotation, the billboard is always in a horizontal position on the screen, looking at the camera (like flares).
            WorldUpAlignment,    ///< Regardless of camera rotation, the billboard does not roll but is looking at the camera (clouds, hovering text).
            YawAxisAlignment,    ///< Regardless of camera rotation, the billboard does not roll or pitch, the UpVector is fixed, but the billboard is looking at the camera (trees).
            FixedAlignment       ///< Billboard does not align to the camera. It behaves like a planar mesh.
        };

        /**
         *  Creates a Billboard object. The center is in the middle of an Billboard (width/2, height/2).
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @param   width       The width of the Billboard in screen coordinates.
         *  @param   height      The height of the Billboard in screen coordinates.
         *  @return              Pointer to the created Billboard object.
         */
        static Billboard* Create(const Float width, const Float height);

        /**
         *  Destructs a Billboard object.
         */
        virtual ~Billboard() override;

        /**
         *  Clones this Billboard only.
         *  Attached Node resources (except VertexBuffer) are not deep-copied but referenced.
         *  VertexBuffer is copied, so if the width, height or a texture coordinate is changed, the change
         *  is not reflected in the original Billboard too.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual Billboard* Clone() const override;

        /**
         *  Sets width of Billboard object.
         *  @param width Width of the Billboard object
         */
        void SetWidth(Float width);

        /**
         *  Retrieves the width of the Billboard object.
         *  @return The width of the Billboard object
         */
        Float GetWidth() const { return m_vertices[TopRight].x - m_vertices[TopLeft].x; }

        /**
         *  Sets height of Billboard object.
         *  @param height Height of the Billboard object.
         */
        void SetHeight(Float height);

        /**
         *  Retrieves the height of this Billboard object.
         *  @return The height of this Billboard object.
         */
        Float GetHeight() const { return m_vertices[TopLeft].y - m_vertices[BottomLeft].y; }

        /**
         *  Sets alignment.
         *  @param alignment defines the alignment of this Billboard object. Default alignment is FixedAlignment.
         */
        void SetAlignment(Alignment alignment) { m_alignment = alignment; }

        /**
         *  Retrieves the allignment of this Billboard object.
         *  @return The allignment of this Billboard object.
         */
        Alignment GetAlignment() const { return m_alignment; }

        /**
         *  Set texture coordinates for top left corner.
         *  @param u u-Parameter of the texture coordinate.
         *  @param v v-Parameter of the texture coordinate.
         */
        void SetTextureCoordinateTopLeft(Float u, Float v);

        /**
         *  Set texture coordinates for bottom left corner.
         *  @param u u-Parameter of the texture coordinate.
         *  @param v v-Parameter of the texture coordinate.
         */
        void SetTextureCoordinateBottomLeft(Float u, Float v);

        /**
         *  Set texture coordinates for top right corner.
         *  @param u u-Parameter of the texture coordinate.
         *  @param v v-Parameter of the texture coordinate.
         */
        void SetTextureCoordinateTopRight(Float u, Float v);

        /**
         *  Set texture coordinates for bottom right corner.
         *  @param u u-Parameter of the texture coordinate.
         *  @param v v-Parameter of the texture coordinate.
         */
        void SetTextureCoordinateBottomRight(Float u, Float v);

        /**
         *  Retrieves the texture coordinate for the top left corner.
         *  @return The texture coordinate as a Vector.
         */
        Vector2 GetTextureCoordinateTopLeft() const { return Vector2(m_vertices[TopLeft].u, m_vertices[TopLeft].v); }

        /**
         *  Retrieves the texture coordinate for the bottom left corner.
         *  @return The texture coordinate as a Vector.
         */
        Vector2 GetTextureCoordinateBottomLeft() const { return Vector2(m_vertices[BottomLeft].u, m_vertices[BottomLeft].v); }

        /**
         *  Retrieves the texture coordinate for the top right corner.
         *  @return The texture coordinate as a Vector.
         */
        Vector2 GetTextureCoordinateTopRight() const { return Vector2(m_vertices[TopRight].u, m_vertices[TopRight].v); }

        /**
         *  Retrieves the texture coordinate for the bottom right corner.
         *  @return The texture coordinate as a Vector.
         */
        Vector2 GetTextureCoordinateBottomRight() const { return Vector2(m_vertices[BottomRight].u, m_vertices[BottomRight].v); }

        /**
         *  Tests if geometry of this billboard intersects with the given Line. The billboard is aligned with the
         *  current active camera if any, otherwise billboard is not camera aligned (FixedAlignment).
         *  @param line               The intersection Line.
         *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
         *                           If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this billboard, otherwise false.
         */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

        /**
         *  Retrieves the VertexBuffer assigned to this Billboard, which defines the geometry attributes.
         *  @return VertexBuffer defining the geometry of this Billboard.
         */
        MemoryManagement::SharedPointer<VertexBuffer> GetVertexBuffer() const { return m_vertexBuffer; }

        /**
         *  Upload asset data attached to this Node to video memory.
         *  @return False if Upload of vertex buffer or appearance failed. True otherwise.
         */
        virtual bool Upload() override;

        /**
         *  Unload asset data attached to this Node from video memory.
         *  @return False if Unload of vertex buffer or appearance failed. True otherwise.
         */
        virtual bool Unload() override;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Retrieves the world transform matrix of this Billboard aligned to the active camera.
         *  Multiplies local transformation with transformations of all ancestors to this Node.
         *  On top of the Node's world transform, another rotation is added
         *  so the billboard is correctly aligned with the active camera.
         *  To retrieve the world transform matrix of a custom camera aligned Billboard, the application
         *  must first activate the desired camera with a call to RenderDevice::SetActiveCamera(..).
         *  If no active camera is set, the Billboard's alignment will be considered fixed.
         *  @return World transform matrix of this Billboard aligned to the active camera.
         */
        virtual const Matrix4& GetWorldTransform() const override;

    protected:
        /**
         *  Renders this Billboard. This method is overridden from Node.
         */
        virtual void Render() override;

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Computes axis-aligned bounding box (AABB) in object coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
         *  Tests if geometry of this billboard intersects with a pick in absolute Viewport coordinate in pixels.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera)
         *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this billboard, otherwise false.
         */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;

    private:
        // Explicit private Constructor and Copy-Constructor, use Create() to create an instance of this object.
        // Assigning to a node is prohibited, since it was proved error prone.
        // Clone instead.
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Billboard);
        Billboard();

        /**
         *  The enumerator defines the four corners of the billboard, each one represented by a vertex
         */
        enum {
            BottomLeft = 0,     ///< Bottom left corner.
            BottomRight = 1,    ///< Bottom right corner.
            TopLeft = 2,        ///< Top left corner.
            TopRight = 3        ///< Top right corner.
        };

        struct Vertex
        {
            Float x;
            Float y;
            Float z;
            Float u;
            Float v;
            Vertex():x(0.0F),y(0.0F),z(0.0F),u(0.0F),v(0.0F){}
        };

        bool m_isVertexBufferValid;
        mutable Matrix4 m_billboardTransform;
        Vertex m_vertices[4];
        Alignment m_alignment;

        void InvalidateVertexBuffer() { m_isVertexBufferValid = false; }
        bool IsVertexBufferValid() const { return m_isVertexBufferValid; }

        /**
         *  Creates and initializes a Billboard vertex buffer.
         *  @return False if creation of vertex geometry or corresponding vertex buffer failed. True otherwise.
         */
        bool InitializeVertexBuffer();

        void UpdateVertexWidth(Float width);
        void UpdateVertexHeight(Float height);

        CdaDynamicProperties(Candera::Billboard, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
}

#endif  // CANDERA_Billboard_H
