//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Billboard.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    static inline MemoryManagement::SharedPointer<GenericShaderParamSetter> CreateGenericShaderParamSetterInstance()
    {
        MemoryManagement::SharedPointer<GenericShaderParamSetter> gsps = GenericShaderParamSetter::Create();
        gsps->SetLightActivationEnabled(false);
        return gsps;
    }

    static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, s_gsps, CreateGenericShaderParamSetterInstance());
        return s_gsps;
    }

    static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

    Billboard::Billboard() :
        Base(),
        m_isVertexBufferValid(false),
        m_alignment(FixedAlignment)
    {
    }

    Billboard::Billboard(const Billboard& rhs) :
        Base(rhs),
        m_isVertexBufferValid(false),
        m_alignment(rhs.m_alignment)
    {
    }

    Billboard::~Billboard()
    {
    }

    Billboard* Billboard::Create(const Float width, const Float height)
    {
        Billboard* const billboard = FEATSTD_NEW(Billboard);
        if (billboard == 0) {
            return 0;
        }

        if (!billboard->InitializeVertexBuffer()) {
            FEATSTD_DELETE(billboard);
            return 0;
        }

        billboard->UpdateVertexWidth(width);
        billboard->UpdateVertexHeight(height);
        billboard->SetTextureCoordinateBottomLeft(0.0F, 0.0F);
        billboard->SetTextureCoordinateBottomRight(1.0F, 0.0F);
        billboard->SetTextureCoordinateTopLeft(0.0F, 1.0F);
        billboard->SetTextureCoordinateTopRight(1.0F, 1.0F);

        if (!billboard->ComputeBoundingSphere()) {
            FEATSTD_LOG_WARN("ComputeBoundingSphere failed.");
        }
        if (!billboard->ComputeBoundingBox()) {
            FEATSTD_LOG_WARN("ComputeBoundingBox failed.");
        }
        billboard->InvalidateVertexBuffer();

        return billboard;
    }

    Billboard* Billboard::Clone() const
    {
        Billboard* const billboard = FEATSTD_NEW(Billboard)(*this);
        if (billboard == 0) {
            return 0;
        }
        if (!billboard->InitializeVertexBuffer()) {
            FEATSTD_DELETE(billboard);
            return 0;
        }

        billboard->UpdateVertexWidth(GetWidth());
        billboard->UpdateVertexHeight(GetHeight());
        Vector2 bottomLeft = GetTextureCoordinateBottomLeft();
        Vector2 bottomRight = GetTextureCoordinateBottomRight();
        Vector2 topLeft = GetTextureCoordinateTopLeft();
        Vector2 topRight = GetTextureCoordinateTopRight();
        billboard->SetTextureCoordinateBottomLeft(
            bottomLeft.GetX(), bottomLeft.GetY());
        billboard->SetTextureCoordinateBottomRight(
            bottomRight.GetX(), bottomRight.GetY());
        billboard->SetTextureCoordinateTopLeft(
            topLeft.GetX(), topLeft.GetY());
        billboard->SetTextureCoordinateTopRight(
            topRight.GetX(), topRight.GetY());

        // Bounding box and bounding sphere are copied from
        // the current billboard.
        billboard->InvalidateVertexBuffer();

        return billboard;
    }

    void Billboard::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    bool Billboard::Upload()
    {
        // Vertex buffer is considered valid if modifications are done before upload to video memory.
        if (! m_vertexBuffer->IsUploaded()) {
            m_isVertexBufferValid = true;
        }

        // Upload vertex buffer.
        bool success = m_vertexBuffer->Upload();
        if (! success) {
            FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        }

        // Upload appearance if not null.
        MemoryManagement::SharedPointer<Appearance> appearance = GetAppearance();
        if (appearance != 0) {
            if (!appearance->Upload()) {
                FEATSTD_LOG_ERROR("Upload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        return success;
    }

    bool Billboard::Unload()
    {
        // Unload mandatory vertex buffer.
        bool success = m_vertexBuffer->Unload();
        if (! success) {
            FEATSTD_LOG_ERROR("Unload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        }

        // Unload appearance if not null.
        MemoryManagement::SharedPointer<Appearance> appearance = GetAppearance();
        if (appearance != 0) {
            if (!appearance->Unload()) {
                FEATSTD_LOG_ERROR("Unload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        return success;
    }

    bool Billboard::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
    {
        return Math3D::ComputeBoundingBox(m_vertexBuffer, minBounds, maxBounds);
    }

    static NodeRenderSequence& GetNodeRenderSequence()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
        return renderSequence;
    }

    static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

    void Billboard::Render()
    {
        NodeRenderSequence& renderSequence = GetNodeRenderSequence();

        if ( !IsRenderPrerequisiteFulfilled() ) {
            FEATSTD_LOG_ERROR("Render precondition is not fulfilled for Node:\"%s\"", (GetName() == 0) ? "" : GetName());
            return;
        }

        if (!IsVertexBufferValid()) {
            if (m_vertexBuffer->Update(0, 4)) {
                m_isVertexBufferValid = true;
            }
        }

        renderSequence.Render(*this, m_vertexBuffer);
    }

    bool Billboard::IsLineIntersectingGeometry(const Line& line, Float& distance) const
    {
        if (!IsIntersectionTestEnabled()) {
            return false;
        }

        Vector3 triangleVertex1(m_vertices[BottomLeft].x, m_vertices[BottomLeft].y, m_vertices[BottomLeft].z);
        Vector3 triangleVertex2(m_vertices[BottomRight].x, m_vertices[BottomRight].y, m_vertices[BottomRight].z);
        Vector3 triangleVertex3(m_vertices[TopLeft].x, m_vertices[TopLeft].y, m_vertices[TopLeft].z);
        Vector3 triangleVertex4(m_vertices[TopRight].x, m_vertices[TopRight].y, m_vertices[TopRight].z);

        const Matrix4& worldTransform = GetWorldTransform();
        triangleVertex1.TransformCoordinate(worldTransform);
        triangleVertex2.TransformCoordinate(worldTransform);
        triangleVertex3.TransformCoordinate(worldTransform);
        triangleVertex4.TransformCoordinate(worldTransform);

        Vector3 hitPosition;
        if (Math3D::TriangleLineIntersection(triangleVertex1,
            triangleVertex2,
            triangleVertex3,
            line,
            hitPosition)) {
                distance = line.GetStart().GetDistanceTo(hitPosition);
                return true;
        }

        if (Math3D::TriangleLineIntersection(triangleVertex4,
            triangleVertex2,
            triangleVertex3,
            line,
            hitPosition)) {
                distance = line.GetStart().GetDistanceTo(hitPosition);
                return true;
        }

        return false;
    }

    bool Billboard::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
    {

        const Camera* activeCamera = RenderDevice::GetActiveCamera();
        RenderDevice::SetActiveCamera(&camera);//For better understanding: Temporarily the active camera is set to the camera parameter,
        //as it is needed to calculate the Billboards WorldTransform,
        //using it's camera alignment. WorldTransform is used in IsLineIntersectingGeometry.
        
        bool result = Math3D::IsPickIntersectingGeometry(*this, camera, x, y, distance);
        RenderDevice::SetActiveCamera(activeCamera);

        return result;
    }

    FEATSTD_RTTI_DEFINITION(Billboard, Renderable)

    void Billboard::SetWidth(Float width)
    {
        UpdateVertexWidth(width);

        if (!ComputeBoundingSphere()) {
            FEATSTD_LOG_WARN("ComputeBoundingSphere failed.");
        }
        if (!ComputeBoundingBox()) {
            FEATSTD_LOG_WARN("ComputeBoundingBox failed.");
        }
        InvalidateVertexBuffer();
    }

    void Billboard::SetHeight(Float height)
    {
        UpdateVertexHeight(height);

        if (!ComputeBoundingSphere()) {
            FEATSTD_LOG_WARN("ComputeBoundingSphere failed.");
        }
        if (!ComputeBoundingBox()) {
            FEATSTD_LOG_WARN("ComputeBoundingBox failed.");
        }
        InvalidateVertexBuffer();
    }

    void Billboard::SetTextureCoordinateTopLeft(Float u, Float v)
    {
        m_vertices[TopLeft].u = u;
        m_vertices[TopLeft].v = v;
        InvalidateVertexBuffer();
    }

    void Billboard::SetTextureCoordinateBottomLeft(Float u, Float v)
    {
        m_vertices[BottomLeft].u = u;
        m_vertices[BottomLeft].v = v;
        InvalidateVertexBuffer();
    }

    void Billboard::SetTextureCoordinateTopRight(Float u, Float v)
    {
        m_vertices[TopRight].u = u;
        m_vertices[TopRight].v = v;
        InvalidateVertexBuffer();
    }

    void Billboard::SetTextureCoordinateBottomRight(Float u, Float v)
    {
        m_vertices[BottomRight].u = u;
        m_vertices[BottomRight].v = v;
        InvalidateVertexBuffer();
    }

    const Matrix4& Billboard::GetWorldTransform() const
    {
        const Camera* camera = RenderDevice::GetActiveCamera();
        if ((m_alignment == FixedAlignment) || (camera == 0)) {
            return Base::GetWorldTransform();
        }

        Vector3 lookat;
        Vector3 up;
        Vector3 right;

        Vector3 worldScale;
        Matrix4 worldRotation;
        Vector3 worldPosition;
        if (!Base::GetWorldTransform().Decompose(worldScale, worldRotation, worldPosition))
        {
            worldScale.SetZero();
            worldPosition.SetZero();
        }

        switch (m_alignment) {
        case CameraUpAlignment:
            lookat = -camera->GetWorldLookAtVector();
            up = camera->GetWorldUpVector();
            right = camera->GetWorldRightVector();
            break;
        case WorldUpAlignment:
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(864, Vector3 constructor copies it's arguments and therefore doesn't change the passed values.)
                up = Vector3(worldRotation(1,0), worldRotation(1,1), worldRotation(1,2));
            lookat = -camera->GetWorldLookAtVector();
            right = up.GetCrossProduct(lookat);
            static_cast<void>(right.Normalize());
            up = lookat.GetCrossProduct(right);
            break;
        case YawAxisAlignment:
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(864, Vector3 constructor copies it's arguments and therefore doesn't change the passed values.)
                up = Vector3(worldRotation(1,0), worldRotation(1,1), worldRotation(1,2));
            lookat = -camera->GetWorldLookAtVector();
            right = up.GetCrossProduct(lookat);
            static_cast<void>(right.Normalize());
            lookat = right.GetCrossProduct(up);
            break;
        default:
            FEATSTD_LOG_ERROR("BillboardGetWorldTransform - Unknown alignment.");
            // Return FixedAlignment:
            return Base::GetWorldTransform();
        }

        m_billboardTransform = Matrix4(right.GetX() * worldScale.GetX(),  right.GetY() * worldScale.GetX(),  right.GetZ() * worldScale.GetX(),  0.0F,
            up.GetX() * worldScale.GetY(),     up.GetY() * worldScale.GetY(),     up.GetZ() * worldScale.GetY(),     0.0F,
            lookat.GetX() * worldScale.GetZ(), lookat.GetY() * worldScale.GetZ(), lookat.GetZ() * worldScale.GetZ(), 0.0F,
            worldPosition.GetX(),              worldPosition.GetY(),              worldPosition.GetZ(),              1.0F);
        return m_billboardTransform;
    }

    bool Billboard::InitializeVertexBuffer()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_INSTANCESOBTAINABLE);
        static VertexGeometry::VertexElementFormat s_vertexFormat[] = {
            {  0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 },
            { 12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 }
        };

        VertexGeometry* vertexGeometry = FEATSTD_NEW(VertexGeometry)(m_vertices, 0, s_vertexFormat, 0, 0, 0, 4, sizeof (Vertex), 2, 0,
            VertexGeometry::VideoMemory, VertexGeometry::ArrayBuffer, VertexGeometry::StaticWrite);

        if(vertexGeometry == 0) {
            return false;
        }

        m_vertexBuffer = VertexBuffer::Create();
        if ((m_vertexBuffer == 0) || (!m_vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose))) {
            FEATSTD_DELETE(vertexGeometry);
            return false;
        }

        m_vertexBuffer->SetPrimitiveType(VertexBuffer::TriangleStrip);

        return true;
    }

    void Billboard::UpdateVertexWidth(Float width)
    {
        if (width < 0.0F) {
            width = 0.0F;
        }

        const Float widthHalf = width / 2.0F;

        m_vertices[BottomLeft].x = -widthHalf;
        m_vertices[BottomRight].x = widthHalf;
        m_vertices[TopLeft].x = -widthHalf;
        m_vertices[TopRight].x = widthHalf;
    }

    void Billboard::UpdateVertexHeight(Float height)
    {
        if (height < 0.0F) {
            height = 0.0F;
        }

        const Float heightHalf = height / 2.0F;
        m_vertices[BottomLeft].y = -heightHalf;
        m_vertices[BottomRight].y = -heightHalf;
        m_vertices[TopLeft].y = heightHalf;
        m_vertices[TopRight].y = heightHalf;
    }
}
