//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_APPEARANCE_H)
#define CANDERA_APPEARANCE_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/InstanceId.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/Engine3D/Core/Texture.h>
#include <Candera/Engine3D/ShaderParamSetters/AbstractShaderParamSetter.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Macros.h>
#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

//forward declarations
class Texture;
class Shader;

/**
 * @brief   The class Appearance groups following render attributes: Material, Textures, RenderMode, and Shader.
 *            Render attributes define the distinctive visualization of a geometry like Mesh, Billboard, and PointSprite.
 *            Further, render attributes can be shared across multiple objects, which    conserves memory and enables sharing of appearance characteristics.
 *          An Appearance object is mandatory for any object in order to get rendered.
 *            Per default all render attributes are initialized to null, which is a valid setting. If the Appearance is activated, all render
 *            attributes that are set become activated. If no RenderMode is defined (null), then the default RenderMode is used instead;
 */
class Appearance: public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Appearance);

        friend class Node;
        friend class RenderOrder;

        /**
        *  Creates an instance of this class.
        *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
        */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~Appearance() override;

        /**
        *  Create a shallow copy of this appearance. All objects are shared
        *  between the source appearance and the copy. For chains the chain is
        *  broken.
        *  @return   A Shared pointer to the copy.
        */
        virtual SharedPointer Clone() const;

        /**
         *  Sets the material for this Appearance. If no Material for this Appearance is set,
         *  then no material is used and lighting calculations in associated shaders can not be applied.
         *  @param material Material to be applied to this Appearance.
         */
        void SetMaterial(Material::SharedPointer material) { m_material = material; }

        /**
         *  Retrieves the current material of this Appearance.
         *  @return The current material.
         */
        Material::SharedPointer GetMaterial() const { return m_material; }

        /**
         *  Sets the texture related to the given texture unit for this Appearance.
         *  The unit must not be higher then the max. number of texture units allowed.
         *  @param texture Texture to be applied to this Appearance.
         *  @param unit Texture unit used for this texture. The unit must not be higher then the max. number of texture units allowed.
         *  @return True if a valid texture unit is used. False if unit is higher than the max number of texture units.
         */
        bool SetTexture(MemoryManagement::SharedPointer<Texture> texture, UInt unit = 0);

        /**
         *  Retrieves the current texture for a given unit.
         *  @param unit Texture unit used for getting the texture default = 0.
         *  @return A MemoryManagement::SharedPointer to the Texture.
         */
        MemoryManagement::SharedPointer<Texture> GetTexture(UInt unit = 0) const;

        /**
         *  Sets the render mode for this Appearance. If no render mode for this Appearance is set, then the default render mode is used.
         *  For details how to set the default RenderMode see: RenderDevice::SetDefaultRenderMode(RenderMode::SharedPointer renderMode).
         *  If a Camera has a RenderMode set, then the Camera's RenderMode overrules the DefaultRenderMode during its render pass.
         *  @param renderMode Render mode to be set for this Appearance.
         */
        void SetRenderMode(RenderMode::SharedPointer renderMode) { m_renderMode = renderMode; }

        /**
         *  Retrieves the current render mode for this Appearance.
         *  @return The current render mode.
         */
        RenderMode::SharedPointer GetRenderMode() const { return m_renderMode; }

        /**
         *  Sets the Shader for this Appearance. If no Shader is set, rendering can not be performed.
         *  @param shader Shader to be set for this Appearance.
         */
        void SetShader(MemoryManagement::SharedPointer<Shader> shader) { m_shader = shader; }

        /**
         *  Retrieves the current shader of this Appearance.
         *  @return The current shader.
         */
        MemoryManagement::SharedPointer<Shader> GetShader() const { return m_shader; }

        /**
         *  Set shader parameters associated to this Appearance.
         *  @param shaderParamSetter Shader parameter setter to be set to this Appearance.
         */
        void SetShaderParamSetter(AbstractShaderParamSetter::SharedPointer shaderParamSetter) { m_shaderParamSetter = shaderParamSetter; }
        
        /**
         *  Retrieves the shader parameters associated to this Appearance.
         *  @return The current shader parameters.
         */
        AbstractShaderParamSetter::SharedPointer GetShaderParamSetter() const { return m_shaderParamSetter; }

        /**
         *  Activates the Appearance and therefore the appropriate RenderMode.  Other render attributes, such as Material, Textures and the Shader are activated .
         *  If no RenderMode is set (null), then the default RenderMode is used instead.
         *  @param node specifies a Node the Appearance is associated to. If Node is given and not null, Node's
         *  properties like the effective alpha value are taken into account, when Appearance gets activated.
         *  @return False if shader is not set or material, texture units or render mode couldn't be activated. True otherwise.
         */
        bool Activate(const Node* node = 0);

        /**
         *  Upload Textures and Shader to render-device memory (VRAM).
         *  @return False if shader is 0 or any associated texture or shader couldn't be uploaded to VRAM. True otherwise.
         */
        virtual bool Upload();

        /**
         *  Unload Textures and Shader from render-device memory (VRAM).
         *  @return False if shader is 0 or any associated texture or shader couldn't be unloaded from VRAM. True otherwise.
         */
        virtual bool Unload();

        /**
         *  Gets the next render pass of an appearance, if there is any. In case of class Appearance, this is always a SharedPointer to null.
         *  See derived class MultiPassAppearance if node-based multi-pass rendering shall be supported.
         *  @return SharedPointer to null as Appearance represents a single pass and must not point to next render pass.
         */
        virtual SharedPointer GetNextPass() const { return SharedPointer(0); }

        /**
         *  Returns the Id of this vertex buffer instance.
         *  @return The Id of this vertex buffer instance.
         */
        Int GetInstanceId() const { return m_instanceId; }

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Copies the properties of the given source appearance to this appearance.
         *  @param sourceAppearance  The appearance to copy the properties for this appearance from.
         */
        void CopyProperties(const Appearance& sourceAppearance);

    protected:
        // Protected because only ::Create() should be used to create an instance of this class, but subclasses should be able to invoke constructor.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Appearance::Appearance, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        Appearance();
        Appearance(const Appearance& appearance);

    private:
        Internal::InstanceId<Appearance> m_instanceId;

        // to bypass expensive SharedPointer getters
        friend class BatchOrderCriterion;
        friend class Renderer;

        enum {
            TextureUnitCount = CANDERA_MAX_TEXTURE_UNIT_COUNT    ///< Number of texture units supported by Candera.
        };

        Material::SharedPointer m_material;
        RenderMode::SharedPointer m_renderMode;
        MemoryManagement::SharedPointer<Shader> m_shader;
        AbstractShaderParamSetter::SharedPointer m_shaderParamSetter; // Custom shader params associated to this appearance.
        Texture::SharedPointer m_textureUnits[TextureUnitCount];

        // Private because only ::Create() should be used to create an instance of this class.
        Appearance& operator = (const Appearance& appearance);

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

};

/** @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_APPEARANCE_H

