//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_CANVASTEXT_BATCH_H
#define CANDERA_CANVASTEXT_BATCH_H

#include <Candera/Engine3D/Core/Mesh.h>

namespace Candera {
namespace Internal {

class CanvasTextBatch : public Mesh
{
public:
    /**
     *  Creates an instance of this class.
     *  Use Dispose() to delete the instance and possible children, if any.
     *  @return Pointer to the created CanvasTextBatch object.
     */
    static CanvasTextBatch* Create();

    /**
     *  Destructor
     */
    virtual ~CanvasTextBatch() override;

    /**
     *  Upload asset data attached to this Node to video memory.
     *  Note: The nodes data is only uploaded into the currently active context.
     *  @return true if upload succeeded and false if upload failed.
     */
    virtual bool Upload() { return true; }

    /**
     *  Unload asset data attached to this Node from video memory.
     *  Note: The nodes data is only unloaded from the currently active context.
     *  @return true if unload succeeded and false if unload failed.
     */
    virtual bool Unload() { return true; }

    /**
     *  Tests if geometry of this mesh intersects with the given Line.
     *  @param line               The intersection Line.
     *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
     *                            If no intersection is found then this param stays unchanged.
     *  @return                   True if an intersection is found with this mesh, otherwise false.
     */
    virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

    FEATSTD_RTTI_DECLARATION();

protected:
    // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
    CanvasTextBatch();
    FEATSTD_MAKE_CLASS_UNCOPYABLE(CanvasTextBatch);

    /**
     *  Computes axis-aligned bounding box (AABB) in object coordinate space.
     *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
     *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
     *  @return true if bounding box was computed successfully and false if not because of missing geometry.
     */
    virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

    /**
     *  Tests if geometry of this mesh intersects with a pick in absolute Viewport coordinate in pixels.
     *  @remark                   This interface is usually used for object selection in scene editors.
     *  @param camera             The view camera. (i.e. the current camera)
     *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
     *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
     *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
     *                            If no intersection is found then this param stays unchanged.
     *  @return                   Always false, because this internally used class is not pickable.
     */
    virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;
};

}
}

#endif
