//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanvasSprite.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/System/Mathematics/Line.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);


    /*
    * This vertex array creates the following vertex buffer:
    * C---D
    * | \ |
    * |  \|
    * A---B
    * x y z u v
    */
    const Float c_vertices[] = {
            0.0F, 0.0F, 0.0F, 0.0F, 1.0F, //A
            0.0F, 1.0F, 0.0F, 0.0F, 0.0F, //C
            1.0F, 0.0F, 0.0F, 1.0F, 1.0F, //B
            1.0F, 1.0F, 0.0F, 1.0F, 0.0F  //D
    };

    static inline MemoryManagement::SharedPointer<GenericShaderParamSetter> CreateGenericShaderParamSetterInstance() {
        MemoryManagement::SharedPointer<GenericShaderParamSetter> gsps = GenericShaderParamSetter::Create();
        gsps->SetLightActivationEnabled(false);
        gsps->SetCanvasActivationEnabled(true);
        return gsps;
    }

    static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance() {
        FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, s_gsps, CreateGenericShaderParamSetterInstance());
        return s_gsps;
    }

    static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

    CanvasSprite::CanvasSprite() :
        Base()
    {

    }

    bool CanvasSprite::IsBoundingBoxValid() const
    {
        // Check if the Canvas scale (determined by vertex rectangle or (for negative values) by texture size) has changed
        if (m_vertexScaleCache != GetVertexScale())
        {
            return false;
        }

        return Base::IsBoundingBoxValid();
    }

    CanvasSprite::CanvasSprite(const Candera::CanvasSprite &other) :
        Base(other)
    {
    }

    CanvasSprite* CanvasSprite::Create()
    {

        CanvasSprite* canvasRenderNode = FEATSTD_NEW(CanvasSprite);
        if ((canvasRenderNode != 0) && (!canvasRenderNode->Initialize())) {
            FEATSTD_DELETE(canvasRenderNode);
            canvasRenderNode = 0;
        }
        return canvasRenderNode;
    }

    void CanvasSprite::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    CanvasSprite::~CanvasSprite()
    {
       
    }

    CanvasSprite* CanvasSprite::Clone() const {
        CanvasSprite* canvasRenderNode = FEATSTD_NEW(CanvasSprite)(*this);
        if ((canvasRenderNode != 0) && (!canvasRenderNode->Initialize())) {
            FEATSTD_DELETE(canvasRenderNode);
            canvasRenderNode = 0;
        }
        return canvasRenderNode;
    }

    Vector2 CanvasSprite::GetActualDimension(const Appearance& appearance) const {
        
        Vector2 dimension = m_vertexRectangle.GetSize();
        Vector2 imageSourceDimension(0.0F, 0.0F);
        
        if ((dimension.GetX() < 0.0F) || (dimension.GetY() < 0.0F)) {
            TextureImage* textureImage = 0;

            if (m_targetTexture < CANDERA_MAX_TEXTURE_UNIT_COUNT) {
                Texture::SharedPointer texture = appearance.GetTexture(m_targetTexture);

                if (!texture.PointsToNull()) {
                    SharedPointer<TextureImage> ti = texture->GetTextureImage();
                    textureImage = ti.GetPointerToSharedInstance();
                }
            }

            if (textureImage != 0) {
                ImageSource3D* imageSource = textureImage->ToImageSource3D();

                if (imageSource != 0) {
                    imageSourceDimension.SetX(static_cast<Float>(imageSource->GetWidth()));
                    imageSourceDimension.SetY(static_cast<Float>(imageSource->GetHeight()));
                }
            }   
        }

        dimension.GetX() < 0.0F ? 
            dimension.SetX(imageSourceDimension.GetX()) :
            dimension.SetX(dimension.GetX());

        dimension.GetY() < 0.0F ?
            dimension.SetY(imageSourceDimension.GetY()) :
            dimension.SetY(dimension.GetY());

        return dimension;
    }

    bool CanvasSprite::Upload()
    {
        bool success = true;

        VertexBuffer::SharedPointer vertexBuffer = GetSpriteVertexBuffer();
        FEATSTD_DEBUG_ASSERT(vertexBuffer != 0); // Sprite vertex buffer is always created at instantiation via Create()
        if (!vertexBuffer->Upload()) {
            FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }

        // Upload vertex buffer if not null.
        if ((m_vertexBuffer != 0) && (m_vertexBuffer != vertexBuffer)) {
            if (!m_vertexBuffer->Upload()) {
                FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        // Upload appearance if not null.
        MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
        if (!app.PointsToNull()) {
            if (!app->Upload()) {
                FEATSTD_LOG_ERROR("Upload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        return success;
    }

    bool CanvasSprite::Unload()
    {
        bool success = true;

        // Unload vertex buffer if not null.
        if (m_vertexBuffer != 0) {
            if (!m_vertexBuffer->Unload()) {
                FEATSTD_LOG_ERROR("Unload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        // Unload appearance if not null.
        MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
        if (!app.PointsToNull()) {
            if (!app->Unload()) {
                FEATSTD_LOG_ERROR("Unload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        return success;
    }

    static NodeRenderSequence& GetNodeRenderSequence()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
        return renderSequence;
    }

    static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

    void CanvasSprite::Render()
    {
        Vector2 actualSize;
        AbstractNodePointer abstractNodePointer(this);
        if ((m_ninePatchBitmap != 0) && (m_ninePatchBitmap->IsNinePatch())) {
#ifdef CANDERA_LAYOUT_ENABLED
            actualSize = Layouter::GetActualSize(abstractNodePointer);

            if (actualSize == Vector2(-1.F, -1.F)) {
                actualSize.SetX(Layouter::GetSize(*(abstractNodePointer.ToCanderaObject())).GetX());
                if (Math::FloatAlmostEqual(actualSize.GetX(), -1.F)) {
                    actualSize.SetX(static_cast<Float>(m_ninePatchBitmap->GetWidth()));
                }
                actualSize.SetY(Layouter::GetSize(*(abstractNodePointer.ToCanderaObject())).GetY());
                if (Math::FloatAlmostEqual(actualSize.GetY(), -1.F)) {
                    actualSize.SetY(static_cast<Float>(m_ninePatchBitmap->GetHeight()));
                }
            }
#else
            actualSize.SetX(static_cast<Float>(m_ninePatchBitmap->GetWidth()));
            actualSize.SetY(static_cast<Float>(m_ninePatchBitmap->GetHeight()));
#endif
            // If the unit of measure changes from 1.0F to some other value,
            // the vertex rectangle needs to be scaled accordingly.
            SetVertexRectangle(Candera::Rectangle(0.0F, 0.0F, actualSize.GetX(), actualSize.GetY()));

            Float scaledWidth = actualSize.GetX();
            Float scaledHeight = actualSize.GetY();

            Bitmap::NinePatchProperties npp = *(m_ninePatchBitmap->GetNinePatchProperties());

            Float horizontalBordersSum = static_cast<Float>(npp.left + npp.right);
            Float verticalBordersSum = static_cast<Float>(npp.top + npp.bottom);
            Float leftRight = static_cast<Float>(m_ninePatchBitmap->GetWidth()) - horizontalBordersSum;
            Float topBottom = static_cast<Float>(m_ninePatchBitmap->GetHeight()) - verticalBordersSum;

            Float scaledLeftRight = scaledWidth - horizontalBordersSum;
            Float scaledTopBottom = scaledHeight - verticalBordersSum;

            Vector2 toScale;
            bool xScale = false;
            bool yScale = false;

            if (scaledLeftRight <= .0F) {
                xScale = true;
                toScale.SetX(scaledWidth / horizontalBordersSum);
                FEATSTD_DEBUG_ASSERT(toScale.GetX() <= 1.F);
                scaledWidth = horizontalBordersSum * toScale.GetX();
            }
            else {
                toScale.SetX(scaledLeftRight / leftRight);
            }

            if (scaledTopBottom <= .0F) {
                yScale = true;
                toScale.SetY(scaledHeight / verticalBordersSum);
                FEATSTD_DEBUG_ASSERT(toScale.GetY() <= 1.F);
                scaledHeight = verticalBordersSum * toScale.GetY();
            }
            else {
                toScale.SetY(scaledTopBottom / topBottom);
            }

            Rectangle boundingRectangle(0.0F, 0.0F, scaledWidth, scaledHeight);
            GetCanvasTransformable().SetBoundingRectangle(boundingRectangle);

            PrepareNinePatchVertexGeometry(scaledWidth, scaledHeight, npp, toScale, xScale, yScale);
        }

        NodeRenderSequence& renderSequence = GetNodeRenderSequence();

        if (!IsRenderPrerequisiteFulfilled()) {
            FEATSTD_LOG_ERROR("Render precondition is not fulfilled for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            return;
        }

        renderSequence.Render(*this, m_vertexBuffer);
    }

    Vector3 CanvasSprite::GetVertexScale() const {
        Vector2 scale(0.0F, 0.0F);

        if ((m_vertexRectangle.GetSize().GetX() < 0.0F) || (m_vertexRectangle.GetSize().GetY() < 0.0F)) {
            if (!GetAppearance().PointsToNull()) {
                scale = GetActualDimension(GetAppearance().GetSharedInstance());
            }
        }
        else {
            scale = m_vertexRectangle.GetSize();
        }
        return Vector3(scale.GetX(), scale.GetY(), 1.0F);
    }

    bool CanvasSprite::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
    {
        // The vertex buffer of a nine patch is only valid after a Layouter has set the actual size, and
        // the CanvasSprite has used that size to generated the vertices of the nine patch accordingly.
        // Only then the bounding box can provide correct values. However the Layouter needs the bounding
        // box values before it can calculate the actual size. This circular dependency is resolved by
        // returning an arbitrary bounding box value as long as the vertices of a nine patch are not valid.
        if ((m_ninePatchBitmap != 0) && (m_ninePatchBitmap->IsNinePatch())) {
            if (!m_ninePatchVertexBufferProperties[ContextResourcePool::GetActive().GetIndex()].IsVertexBufferValid()) {
                minBounds.SetZero();
                maxBounds = Vector3(1.0F, 1.0F, 0.0F);
                return true;
            }
        }

        bool result = Math3D::ComputeBoundingBox(m_vertexBuffer, minBounds, maxBounds);

        Vector3 offset(m_vertexRectangle.GetPosition().GetX(), m_vertexRectangle.GetPosition().GetY(), 0.0F);

        if (result) {
            m_vertexScaleCache = GetVertexScale();

            minBounds = (minBounds * m_vertexScaleCache) + offset;
            maxBounds = (maxBounds * m_vertexScaleCache) + offset;
        }
        else {
            minBounds = offset;
            maxBounds = offset;
        }
        return result;
    }

    bool CanvasSprite::IsLineIntersectingGeometry(const Line& line, Float& distance /*out*/) const
    {
        if (!IsIntersectionTestEnabled()) {
            return false;
        }

        Vector3 vertexPositions[4];
        vertexPositions[0] = Vector3(c_vertices[0], c_vertices[1], c_vertices[2]);
        vertexPositions[1] = Vector3(c_vertices[5], c_vertices[6], c_vertices[7]);
        vertexPositions[2] = Vector3(c_vertices[10], c_vertices[11], c_vertices[12]);
        vertexPositions[3] = Vector3(c_vertices[15], c_vertices[16], c_vertices[17]);

        Vector3 offset(m_vertexRectangle.GetPosition().GetX(), m_vertexRectangle.GetPosition().GetY(), 0.0F);
        Vector3 scale = GetVertexScale();
        const Matrix4& worldTransform = GetWorldTransform();
        for (UInt8 i = 0; i < 4; ++i) {
            vertexPositions[i] = (vertexPositions[i] * scale) + offset;
            vertexPositions[i].TransformCoordinate(worldTransform);
        }

        Vector3 hitPosition;
        if (Math3D::TriangleLineIntersection(vertexPositions[0],
            vertexPositions[1],
            vertexPositions[2],
            line,
            hitPosition)) {
            distance = line.GetStart().GetDistanceTo(hitPosition);
            return true;
        }

        if (Math3D::TriangleLineIntersection(vertexPositions[3],
            vertexPositions[1],
            vertexPositions[2],
            line,
            hitPosition)) {
            distance = line.GetStart().GetDistanceTo(hitPosition);
            return true;
        }

        return false;
    }

    bool CanvasSprite::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
    {
        return Math3D::IsPickIntersectingGeometry(*this, camera, x, y, distance);
    }

    struct NinePatchVertex {
        Float x;
        Float y;
        Float z;
        Float u;
        Float v;
    };

    static const VertexGeometry::VertexElementFormat NinePatchVertexFormat[] = {
        { 0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 },
        { 12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 }
    };

    static void PrepareNinePatchIndices(UInt16* indexBuffer)
    {
        SizeType idx = 0;
        // Upper row
        indexBuffer[idx++] = 0; indexBuffer[idx++] = 4; indexBuffer[idx++] = 1;
        indexBuffer[idx++] = 4; indexBuffer[idx++] = 5; indexBuffer[idx++] = 1;
        indexBuffer[idx++] = 1; indexBuffer[idx++] = 5; indexBuffer[idx++] = 2;
        indexBuffer[idx++] = 5; indexBuffer[idx++] = 6; indexBuffer[idx++] = 2;
        indexBuffer[idx++] = 2; indexBuffer[idx++] = 6; indexBuffer[idx++] = 3;
        indexBuffer[idx++] = 6; indexBuffer[idx++] = 7; indexBuffer[idx++] = 3;

        // Middle row
        indexBuffer[idx++] = 4; indexBuffer[idx++] = 8; indexBuffer[idx++] = 5;
        indexBuffer[idx++] = 8; indexBuffer[idx++] = 9; indexBuffer[idx++] = 5;
        indexBuffer[idx++] = 5; indexBuffer[idx++] = 9; indexBuffer[idx++] = 6;
        indexBuffer[idx++] = 9; indexBuffer[idx++] = 10; indexBuffer[idx++] = 6;
        indexBuffer[idx++] = 6; indexBuffer[idx++] = 10; indexBuffer[idx++] = 7;
        indexBuffer[idx++] = 10; indexBuffer[idx++] = 11; indexBuffer[idx++] = 7;

        // Lower row
        indexBuffer[idx++] = 8; indexBuffer[idx++] = 12; indexBuffer[idx++] = 9;
        indexBuffer[idx++] = 12; indexBuffer[idx++] = 13; indexBuffer[idx++] = 9;
        indexBuffer[idx++] = 9; indexBuffer[idx++] = 13; indexBuffer[idx++] = 10;
        indexBuffer[idx++] = 13; indexBuffer[idx++] = 14; indexBuffer[idx++] = 10;
        indexBuffer[idx++] = 10; indexBuffer[idx++] = 14; indexBuffer[idx++] = 11;
        indexBuffer[idx++] = 14; indexBuffer[idx++] = 15; indexBuffer[idx++] = 11;
    }

    static VertexBuffer::SharedPointer CreateNinePatchVertexBuffer()
    {
        using namespace Candera::Internal;

        VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
        if (vertexBuffer == 0) {
            return VertexBuffer::SharedPointer(0);
        }

        const SizeType vertexCount = 16;
        NinePatchVertex* vertices = CANDERA_NEW_ARRAY(NinePatchVertex, vertexCount);
        if (0 == vertices) {
            return VertexBuffer::SharedPointer(0);
        }

        MemoryPlatform::Set(vertices, 0, sizeof(NinePatchVertex)*vertexCount);

        const SizeType indexCount = 18 * 3;
        UInt16* indexBuffer = CANDERA_NEW_ARRAY(UInt16, indexCount);
        if (0 == indexBuffer) {
            CANDERA_DELETE_ARRAY(vertices);
            return VertexBuffer::SharedPointer(0);
        }

        PrepareNinePatchIndices(indexBuffer);

        VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(
            vertices, MemoryManagement::AdaptedArrayDisposer<const void*, const NinePatchVertex*>::Dispose,
            NinePatchVertexFormat, 0,
            indexBuffer, MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*>::Dispose,
            static_cast<UInt32>(vertexCount), sizeof(NinePatchVertex), 2, static_cast<UInt32>(indexCount),
            VertexGeometry::VideoMemory,
            VertexGeometry::UInt16IndexedArrayBuffer,
            VertexGeometry::DynamicWrite);

        static_cast<void>(vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
        vertexBuffer->SetPrimitiveType(VertexBuffer::Triangles);

        return vertexBuffer;
    }

    static void SetVertex(SizeType offset, Float* vertex, Float x, Float y, Float u, Float v)
    {
        vertex[offset] = x;
        vertex[offset + 1] = y;
        vertex[offset + 2] = 0.0F;
        vertex[offset + 3] = u;
        vertex[offset + 4] = v;
    }

    void CanvasSprite::PrepareNinePatchVertexGeometry(const Float width, const Float height,
        const Bitmap::NinePatchProperties& ninePatchProperties, const Vector2& toScale, const bool scaleX, const bool scaleY)
    {
        if (m_ninePatchVertexBufferProperties[ContextResourcePool::GetActive().GetIndex()].IsEqual(width, height,
            ninePatchProperties, toScale, scaleX, scaleY)) {
            return;
        }

        if (m_vertexBuffer != 0) {
            VertexGeometry::VertexArrayResource vertexResource(m_vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
            Float* vertices = const_cast<Float*>(FeatStd::Internal::PointerToPointer<const Float*>(vertexResource.GetData()));

            Float originalWidth = static_cast<Float>(m_ninePatchBitmap->GetWidth());
            Float originalHeight = static_cast<Float>(m_ninePatchBitmap->GetHeight());

            Float top = static_cast<Float>(ninePatchProperties.top);
            Float bottom = static_cast<Float>(ninePatchProperties.bottom);
            Float left = static_cast<Float>(ninePatchProperties.left);
            Float right = static_cast<Float>(ninePatchProperties.right);

            Float leftRight = originalWidth - (right + left);
            Float topBottom = originalHeight - (bottom + top);

            SizeType idx = 0;

            Float h1;
            Float h2;
            Float h3;

            Float v1;
            Float v2;
            Float v3;

            if (!scaleX) {
                h1 = left / width;
                h2 = h1 + (leftRight * toScale.GetX()) / width;
                h3 = 1.0F;
            }
            else {
                h1 = (left * toScale.GetX()) / width;
                h2 = h1;
                h3 = 1.0F;
            }

            if (!scaleY) {
                v1 = top / height;
                v2 = v1 + (topBottom * toScale.GetY()) / height;
                v3 = 1.0F;
            }
            else {
                v1 = (top * toScale.GetY()) / height;
                v2 = v1;
                v3 = 1.0F;
            }

            SetVertex(idx, vertices, 0.0F, 0.0F, 0.0F, 1.0F);
            idx += 5;
            SetVertex(idx, vertices, h1, 0.0F, left / originalWidth, 1.0F);
            idx += 5;
            SetVertex(idx, vertices, h2, 0.0F, 1.0F - (right / originalWidth), 1.0F);
            idx += 5;
            SetVertex(idx, vertices, h3, 0.0F, 1.0F, 1.0F);
            idx += 5;

            SetVertex(idx, vertices, 0.0F, v1, 0.0F, 1.0F - (top / originalHeight));
            idx += 5;
            SetVertex(idx, vertices, h1, v1, left / originalWidth, 1.0F - (top / originalHeight));
            idx += 5;
            SetVertex(idx, vertices, h2, v1, 1.0F - (right / originalWidth), 1.0F - (top / originalHeight));
            idx += 5;
            SetVertex(idx, vertices, h3, v1, 1.0F, 1.0F - (top / originalHeight));
            idx += 5;

            SetVertex(idx, vertices, 0.0F, v2, 0.0F, (bottom / originalHeight));
            idx += 5;
            SetVertex(idx, vertices, h1, v2, left / originalWidth, bottom / originalHeight);
            idx += 5;
            SetVertex(idx, vertices, h2, v2, 1.0F - (right / originalWidth), bottom / originalHeight);
            idx += 5;
            SetVertex(idx, vertices, h3, v2, 1.0F, bottom / originalHeight);
            idx += 5;

            SetVertex(idx, vertices, 0.0F, v3, 0.0F, 0.0F);
            idx += 5;
            SetVertex(idx, vertices, h1, v3, left / originalWidth, 0.0F);
            idx += 5;
            SetVertex(idx, vertices, h2, v3, 1.0F - (right / originalWidth), 0.0F);
            idx += 5;
            SetVertex(idx, vertices, h3, v3, 1.0F, 0.0F);

            if (!m_vertexBuffer->IsUploaded()) {
                if (!m_vertexBuffer->Upload(DeviceObject::NoHint)) {
                    return;
                }
            }
            else {
                if (m_vertexBuffer->Update(0, 16)) {
                    m_vertexBuffer->SetElementCount(54);
                }
                else {
                    return;
                }
            }

            m_ninePatchVertexBufferProperties[ContextResourcePool::GetActive().GetIndex()].Set(width, height,
                ninePatchProperties, toScale, scaleX, scaleY);

            if (!ComputeBoundingBox()) {
                FEATSTD_LOG_WARN("Bounding box of Canvas Sprite could not be computed!");
            }
            if (!ComputeBoundingSphere()) {
            FEATSTD_LOG_WARN("Bounding sphere of Canvas Sprite could not be computed!");
            }
        }
    }

    bool CanvasSprite::Initialize()
    {
        VertexBuffer::SharedPointer vertexBuffer = GetSpriteVertexBuffer();
        if ((m_ninePatchBitmap != 0) && (m_ninePatchBitmap->IsNinePatch())) {
            m_vertexBuffer = CreateNinePatchVertexBuffer();
            for(SizeType i = 0; i < CANDERA_MAX_CONTEXT_COUNT; i++) {
                m_ninePatchVertexBufferProperties[i] = NinePatchVertexBufferProperties();
            }
        }
        else {
            m_vertexBuffer = vertexBuffer;
        }

        return (0 != m_vertexBuffer.GetPointerToSharedInstance());
    }

    VertexBuffer::SharedPointer CanvasSprite::GetSpriteVertexBuffer()
    {
        FEATSTD_SYNCED_STATIC_OBJECT(VertexBuffer::SharedPointer, s_vertexBuffer, CreateVertexBuffer());
        return s_vertexBuffer;
    }

    VertexBuffer::SharedPointer CanvasSprite::CreateVertexBuffer()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_INSTANCESOBTAINABLE);
        static VertexGeometry::VertexElementFormat s_vertexFormat[] = {
            {0, VertexGeometry::Float32_3, VertexGeometry::Position, 0},
            {12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 },
        };

        static VertexGeometry s_vertexGeometry(c_vertices, 0, s_vertexFormat,
            0, 0, 0, 4, 5 * sizeof(Float), 2, 0, 
            VertexGeometry::VideoMemory, VertexGeometry::ArrayBuffer, VertexGeometry::StaticWrite);

        VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
        if (!vertexBuffer.PointsToNull()) {
            if (!vertexBuffer->SetVertexGeometry(&s_vertexGeometry, 0))
            {
                return VertexBuffer::SharedPointer(0);
            }
            vertexBuffer->SetPrimitiveType(VertexBuffer::TriangleStrip);
        }

        return vertexBuffer;
    }


    void CanvasSprite::GetBasePoint(Vector2& basePoint) const
    {
        Rectangle boundingRect;
        m_canvasTransformable.GetEffectiveBoundingRectangle(boundingRect);

        basePoint = boundingRect.GetPosition() + (boundingRect.GetSize() * 0.5F);
    }

    void CanvasSprite::GetBasePoint(Vector3& basePoint) const
    {
        Vector2 basePoint2D;
        GetBasePoint(basePoint2D);

        basePoint.SetX(basePoint2D.GetX());
        basePoint.SetY(basePoint2D.GetY());
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1058, "False positive: init a non const reference!")
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "False positive: Type missmatch between Vector2 and Vector3. Lint resolves the GetPosition wrongly.")
            const Vector3& currPos = GetPosition();
        basePoint.SetZ(currPos.GetZ());
    }


    void CanvasSprite::GetComputedBoundingRectangle(Rectangle& boundingRectangle) const
    {
        Vector2 position = m_vertexRectangle.GetPosition();
        Vector3 size = GetVertexScale();
        boundingRectangle.SetPosition(position);
        boundingRectangle.SetSize(Vector2(size.GetX(), size.GetY()));
    }

    void CanvasSprite::SetNinePatchBitmap(const Bitmap::SharedPointer& bitmap)
    {
        m_ninePatchBitmap = bitmap;

        if (!Initialize()) {
            FEATSTD_LOG_ERROR("Failed to initialize the vertex buffer.");
        }
    }

    bool CanvasSprite::IsRenderPrerequisiteFulfilled() const
    {
        // Skip Renderable::IsRenderPrerequisiteFulfilled(), but call Renderable's Base (i.e. Node)
        bool isRenderPrerequisiteFulfilled = Node::IsRenderPrerequisiteFulfilled();
        if ((m_ninePatchBitmap != 0) && (m_ninePatchBitmap->IsNinePatch())) {
            return isRenderPrerequisiteFulfilled;
        }

        return isRenderPrerequisiteFulfilled && (m_vertexBuffer != 0) && (m_vertexBuffer->IsUploaded());
    }

    FEATSTD_RTTI_DEFINITION(CanvasSprite, CanvasRenderable)
}
