//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_RENDERABLE_H)
#define CANDERA_CANVAS_RENDERABLE_H

#include <Candera/Engine3D/Canvas/CanvasTransformable.h>
#include <Candera/Engine3D/Core/Renderable.h>
#include <Candera/System/Mathematics/Rectangle.h>

namespace Candera {

    /**
     * @brief   CanvasRenderable is used as base class for renderable canvas objects.
     *          The canvas renderable sets up the parameters needed to render a canvas object.
     */
    class CanvasRenderable : public Renderable, public CanvasInterface {

    public:
        FEATSTD_TYPEDEF_BASE(Renderable);

        virtual ~CanvasRenderable() override {}


        /**
         * Sets the vertex rectangle used for rendering a canvas object.
         * The vertex rectangle's position determines the offset that is given to each vertex.
         * With this mechanism a pivot can be applied to a canvas renderable.
         * The vertex rectangle's size determines the scale that is applied to each vertex.
         * If a cameras projection is orthographic, and it's width and height parameters correspond to the render targets dimensions, and if 
         * no transformation scale is applied then the rectangle's size correspond to concrete pixel dimensions.
         * If width or height is set to a value < 0.0F (e.g. -1.0F), then the dimension of the currently rendered texture is taken. If no texture is available, 
         * it will be set to 0.0F.
         * @param rectangle The rectangle to set pivot and size of the canvas object.
         */
        void SetVertexRectangle(const Rectangle& rectangle) { m_vertexRectangle = rectangle; }

        /**
        * Gets the vertex rectangle used for rendering a canvas object.
        * The vertex rectangle's position determines the offset that is given to each vertex.
        * With this mechanism a pivot can be applied to a canvas renderable.
        * The vertex rectangle's size determines the scale that is applied to each vertex.
        * If a cameras projection is orthographic, and it's width and height parameters correspond to the render targets dimensions, and if
        * no transformation scale is applied then the rectangle's size correspond to concrete pixel dimensions.
        * If width or height is set to a value < 0.0F (e.g. -1.0F), then the dimension of the currently rendered texture is taken. If no texture is available,
        * it will be set to 0.0F.
        * @return The current rectangle that determines pivot and size of the rendered canvas object.
        */
        Rectangle GetVertexRectangle() const { return m_vertexRectangle; }

        /**
         * Sets the texture's index (inside an appearance) that is used to determine the dimension of the canvas object, 
         * if a vertex rectangle's size component is smaller than 0.0F.
         * @param texture The texture's index to determine the dimension of the canvas object.
         */
        void SetTargetTexture(UInt8 texture) { m_targetTexture = texture; }

        /**
        * Gets the texture's index (inside an appearance) that is used to determine the dimension of the canvas object,
        * if a vertex rectangle's size component is smaller than 0.0F.
        * @return The texture's index to determine the dimension of the canvas object.
        */
        UInt8 GetTargetTexture() const { return m_targetTexture; }

        //Needed for bounding box and in case of image also for rendering. Text is different.
        /**
         * Queries the current actual dimension, considering the vertex rectangles size and a texture's dimension if needed.
         */
        virtual Vector2 GetActualDimension(const Appearance& appearance) const = 0; 

        //Canvas Interface.

        /**
        *  Gets the CanvasTransformable of this Canvas node.
        *  @return  This nodes CanvasTransformable.
        */
        virtual CanvasTransformable& GetCanvasTransformable() override;

        /**
        * Override from Node.
        * Invalidates parent canvas.
        */
        virtual void OnAncestorAdded(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas();  }

        /**
        * Override from Node.
        * Invalidates parent canvas.
        */
        virtual void OnAncestorRemoved(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas(); }

        FEATSTD_RTTI_DECLARATION();

    protected:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(CanvasRenderable);

        /**
         * Constructor.
         */
        CanvasRenderable();

        /**
        *  Called when one of the transforms (translation, rotation, scale, general transform) has changed.
        *  Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
        */
        virtual void OnCompositeTransformChanged() override;

        Rectangle m_vertexRectangle;
        UInt8 m_targetTexture;
#ifdef CANDERA_LAYOUT_ENABLED
        /**
        *  Returns the "offset" coordinate which is required to position 3D objects correctly when using layout.
        *  Normally, 3D objects are centered but the layouter thinks its position is the "upper-left" corner.
        *  This will be replaced as soon as there is a more appropriate handling (e.g. own pivot point)
        *  @return  layout start position
        */
        virtual const Vector3 GetLayoutStartPosition() const override;

#endif
        CanvasTransformable m_canvasTransformable;

    private:
        
        CdaDynamicProperties(Candera::CanvasRenderable, Candera::Node);
        CdaDynamicPropertiesEnd();
    };

}


#endif  // CANDERA_CANVAS_RENDERABLE_H
