//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanvasGroup.h"

#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#include <Candera/Engine3D/Canvas/Canvas.h>

namespace Candera {

    Internal::BoundingRectangleTraverser::BoundingRectangleTraverser(const Candera::Node& root) :
        m_root(root)
    {
        m_inverseRootWorldTransform = root.GetWorldTransform();
        m_inverseRootWorldTransform.Inverse();
    }

    void Internal::BoundingRectangleTraverser::GetBoundingRectangle(Rectangle& rect) const
    {
        rect = m_boundingRectangle;
    }


    ConstTreeTraverser::TraverserAction Internal::BoundingRectangleTraverser::ProcessNode(const Node& node)
    {
        if (node.IsTypeOf(CanvasRenderable::GetTypeId())) {

            Rectangle boundingRect;
            CanvasTransformable& canvasTransformable = static_cast<CanvasRenderable&>(const_cast<Node&>(node)).GetCanvasTransformable();
            canvasTransformable.GetEffectiveBoundingRectangle(boundingRect);

            Canvas* parentCanvas = canvasTransformable.GetParentCanvas();

            if (parentCanvas != 0) {
                Matrix4 canvasGroupLocalTransform = node.GetWorldTransform() * m_inverseRootWorldTransform;
                CanvasTransformable::ComputeLocalXYAlignedBoundingRectangle(canvasGroupLocalTransform, boundingRect);
            }
            m_boundingRectangle.Union(boundingRect);
        }

        return ConstTreeTraverser::ProceedTraversing;
    }

    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CanvasGroup::CanvasGroup() :
        Base(),
        m_canvasTransformable(*this, *this)
    {

    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    CanvasGroup* CanvasGroup::Create()
    {
        return FEATSTD_NEW(CanvasGroup);
    }

    void CanvasGroup::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    CanvasGroup* CanvasGroup::Clone() const
    {
        return FEATSTD_NEW(CanvasGroup)(*this);
    }

    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CanvasGroup::CanvasGroup(const Candera::CanvasGroup &other) :
        Base(other),
        m_canvasTransformable(*this, *this)
    {
        if (other.m_canvasTransformable.IsBoundingRectangleSet()) {
            m_canvasTransformable.SetBoundingRectangle(other.m_canvasTransformable.GetBoundingRectangle());
        }
    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    void CanvasGroup::TranslatePivot2D(const Vector2& translation)
    {
        Base::TranslatePivot(Vector3(translation.GetX(), translation.GetY(), 0.0F));
    }

    void CanvasGroup::GetBasePoint(Vector2& basePoint) const
    {
        Rectangle boundingRect;
        m_canvasTransformable.GetEffectiveBoundingRectangle(boundingRect);

        basePoint = boundingRect.GetPosition() + (boundingRect.GetSize() * 0.5F);
    }

#ifdef CANDERA_LAYOUT_ENABLED
    void CanvasGroup::GetBasePoint(Vector3& basePoint) const
    {
        Vector2 basePoint2D;
        GetBasePoint(basePoint2D);

        basePoint.SetX(basePoint2D.GetX());
        basePoint.SetY(basePoint2D.GetY());
        basePoint.SetZ(GetPosition().GetZ());
    }
#endif


    void CanvasGroup::GetComputedBoundingRectangle(Rectangle& boundingRectangle) const
    {
        Internal::BoundingRectangleTraverser traverser(*this);
        traverser.Traverse(*this);
        traverser.GetBoundingRectangle(boundingRectangle);
    }

    CanvasTransformable& CanvasGroup::GetCanvasTransformable()
    {
        return m_canvasTransformable;
    }

    void CanvasGroup::OnCompositeTransformChanged()
    {
        Base::OnCompositeTransformChanged();
        Canvas* parentalCanvas = m_canvasTransformable.GetParentCanvas();
        if (0 != parentalCanvas) {
            parentalCanvas->SetChanged();
        }
    }

#if defined(CANDERA_LAYOUT_ENABLED)
    const Candera::Vector3 CanvasGroup::GetLayoutStartPosition() const
    {
        return m_canvasTransformable.GetLayoutStartPosition();
    }
#endif

    FEATSTD_RTTI_DEFINITION(CanvasGroup, Base)

}
