//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CANDERA_VERTEX_GEOMETRY_2D_H)
#define CANDERA_VERTEX_GEOMETRY_2D_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {
/** @addtogroup Core2D
*  @{
*/

    /**
    *  @brief   The VertexGeometry2D holds all vertex data of a 2D mesh.
    *           The VertexGeometry2D may be used for multiple VertexBuffer2D.
    *           It supports 2D position and 2D texture coordinate vertex attributes,
    *           which can be used in various render APIs.
    *           This class is designed platform agnostic and represents typical data for 2D vertex buffers.
    *           It is possible that certain elements are not supported by the target platform.
    *           During Upload of a VertexBuffer2D this is checked. An Upload will fail if the specified data is not supported by the device.
    *           An according error message will be attached to the log.
    *           Note: 2D has a defined coordinate system with Y-axis pointing down, such that screen coordinates are simulated.
    */
    class VertexGeometry2D : public CanderaObject {
        FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:

        /**
         * Define which memory pool to use.
         */
        enum MemoryPool {
            VideoMemory = 0,   ///< Use video memory.
            ClientMemory = 1   ///< Use target defined memory.
        };

        /**
        *  Define how the buffer data will be used.
        *  Note: On some devices this setting has no effect.
        */
        enum BufferUsage
        {
            StaticWrite = 0,    ///< The buffer data will be specified once and used many times to draw.
            DynamicWrite = 1,   ///< The buffer data will be specified repeatedly and used many times to draw.
            InfrequentDraw = 2  ///< The buffer data will be specified once and used infrequently to draw.
        };

       /**
        * Buffer Type.
        * Note: Some devices only support "ArrayBuffer".
        */
        enum BufferType
        {
            ArrayBuffer = 0,                                ///< The VertexGeometry2D holds only an array of 2D vertices.
            UInt16IndexedArrayBuffer = 1,                   ///< The VertexGeometry2D holds an array of vertices and an index array describing a list of primitives.
            IndexedArrayBuffer = UInt16IndexedArrayBuffer   ///< Is a UInt16IndexedArrayBuffer.
        };

        /**
         * The description of one vertex.
         */
        struct Vertex2D {
            Vector2 Position;
            Vector2 TextureCoordinate;

            Vertex2D() :
                Position(0.0F, 0.0F),
                TextureCoordinate(0.0F, 0.0F)
            {}

            Vertex2D(Vector2 position, Vector2 textureCoordinate) {
                Position = position;
                TextureCoordinate = textureCoordinate;
            }
        };


        /**
        * VertexArray2DResource is a ResourceObject that provides access to the 2D vertex data.
        *
        * To create a resource handle, call VertexArray2DResource::CreateHandle with the following parameters:
        * - data: Pointer to the 2D vertex data.
        * - disposer: The function which defines the dispose method for the data.
        *    The function will be called on VertexGeometry2D::DisposeVertex2DData call or on VertexGeometry2D destruction.
        * - size: size of data in bytes.
        *
        * @see ResourceObject
        */
        typedef ResourceObject<const Vertex2D> VertexArray2DResource;
        typedef MemoryManagement::ArrayDisposer<const Vertex2D*> VertexGeometry2DDisposer;
        typedef VertexArray2DResource::DisposerFunction VertexGeometry2DDisposerFn;


        /**
        * IndexArrayResource is a ResourceObject that provides access to the index data.
        *
        * To create a resource handle, call IndexArrayResource::CreateHandle with the following parameters:
        * - data: Pointer to the index data.
        * - disposer: The function which defines the dispose method for the data.
        *    The function will be called on VertexGeometry::DisposeVertexData call or on VertexGeometry destruction.
        * - size: size of data in bytes.
        *
        * @see ResourceObject
        */
        typedef ResourceObject<const void> IndexArrayResource;
        typedef MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*> IndexBufferDisposer;
        typedef IndexArrayResource::DisposerFunction IndexBufferDisposerFn;


        /**
        *  Constructor for immutable const data.
        *  Takes data and a corresponding Disposer for the data.
        *  The method Disposer::Dispose will be called on destruction.
        *  If a Disposer is set to null the corresponding data will not be disposed on destruction.
        *  @param vertexArray2D                    The Vertex2D geometry array.
        *  @param vertexGeometry2DDisposerFn       The function which defines the dispose method for the 2D vertex geometry.
        *  @param indexBuffer                      The index buffer.
        *  @param indexBufferDisposerFn            The function which defines the dispose method for the index buffer.
        *  @param vertexCount                      The count of vertices in VertexGeometry2D.
        *  @param indexCount                       The count of indices in indexBuffer.
        *  @param memoryPool                       The memory pool of the VertexGeometry2D.
        *  @param bufferType                       The buffer type.
        *  @param bufferUsage                      The buffer usage.
        */
        VertexGeometry2D(const Vertex2D* vertexArray2D,
            VertexGeometry2DDisposerFn vertexGeometry2DDisposerFn,
            const UInt16* indexBuffer,
            IndexBufferDisposerFn indexBufferDisposerFn,
            UInt32 vertexCount,
            UInt32 indexCount,
            MemoryPool memoryPool,
            BufferType bufferType,
            BufferUsage bufferUsage);

        /**
        *  Constructor for mutable  data.
        *  Takes data and a corresponding Disposer for the data.
        *  The method Disposer::Dispose will be called on destruction.
        *  If a Disposer is set to null the corresponding data will not be disposed on destruction.
        *  @param vertexArray2D                    The Vertex2D geometry array.
        *  @param vertexGeometry2DDisposerFn       The function which defines the dispose method for the 2D vertex geometry.
        *  @param indexBuffer                      The index buffer.
        *  @param indexBufferDisposerFn            The function which defines the dispose method for the index buffer.
        *  @param vertexCount                      The count of vertices in VertexGeometry2D.
        *  @param indexCount                       The count of indices in indexBuffer.
        *  @param memoryPool                       The memory pool of the VertexGeometry2D.
        *  @param bufferType                       The buffer type.
        *  @param bufferUsage                      The buffer usage.
        */
        VertexGeometry2D(Vertex2D* vertexArray2D,
            VertexGeometry2DDisposerFn vertexGeometry2DDisposerFn,
            UInt16* indexBuffer,
            IndexBufferDisposerFn indexBufferDisposerFn,
            UInt32 vertexCount,
            UInt32 indexCount,
            MemoryPool memoryPool,
            BufferType bufferType,
            BufferUsage bufferUsage);

        /**
        *  Constructor that takes ResourceDataHandles for accessing Vertex2D and index arrays.
        *  @param vertexArray2DResourceHandle      ResourceDataHandle that references Vertex2D data.
        *  @param indexArrayResourceHandle         ResourceDataHandle that references index data.
        *  @param memoryPool                       The memory pool of the vertexArray.
        *  @param bufferType                       The buffer type.
        *  @param bufferUsage                      The buffer usage.
        */
        VertexGeometry2D(const ResourceDataHandle& vertexArray2DResourceHandle,
            const ResourceDataHandle& indexArrayResourceHandle,
            MemoryPool memoryPool,
            BufferType bufferType,
            BufferUsage bufferUsage);


        /**
        *  Destructor
        */
        virtual ~VertexGeometry2D() override;

        /**
         * Clones a 2D vertex geometry. Note: In case of mutable data,
         * ownership of the cloned data is always passed to the new VertexGeometry2D instance.
         * @return The cloned VertexGeometry2D or 0 if it couldn't be cloned.
         */
        VertexGeometry2D* Clone() const;

        /**
        * @return Constant VertexArray2DResource handle.
        * Create a VertexArray2DResource for accessing the constant 2D vertex array data.
        */
        const ResourceDataHandle& GetVertexArray2DResourceHandle() const { return m_vertexArray2DResourceHandle; }

        /**
        * @return Mutable VertexArray2DResource handle.
        * Create a VertexArray2DResource for accessing the constant/mutable 2D vertex array data.
        */
        ResourceDataHandle& GetVertexArray2DResourceHandle() { return m_vertexArray2DResourceHandle; }

        /**
        * @return Constant IndexArrayResource handle.
        * Create a IndexArrayResource for accessing the constant index array data.
        */
        const ResourceDataHandle& GetIndexArrayResourceHandle() const { return m_indexArrayResourceHandle; }

        /**
        * @return Mutable VertexArrayResource handle.
        * Create a IndexArrayResource for accessing the constant/mutable index array data.
        */
        ResourceDataHandle& GetIndexArrayResourceHandle() { return m_indexArrayResourceHandle; }

        /**
        *  Retrieve the vertex count.
        *  @return The vertex count.
        */
        UInt32 GetVertexCount() const { return m_vertexArray2DResourceHandle.m_size / sizeof(Vertex2D); }


        /**
        *  Retrieve the memory pool.
        *  @return The memory pool.
        */
        VertexGeometry2D::MemoryPool GetMemoryPool() const { return m_memoryPool; }

        /**
        *  Retrieve the buffer type.
        *  @return The buffer type.
        */
        VertexGeometry2D::BufferType GetBufferType() const { return m_bufferType; }

        /**
        *  Retrieve the index count.
        *  @return The index count.
        */
        UInt32 GetIndexCount() const;

        /**
        *  Retrieve the buffer usage.
        *  @return The buffer usage.
        */
        VertexGeometry2D::BufferUsage GetBufferUsage() const { return m_bufferUsage; }


        void DisposeVertex2DData();

        private:
            //FEATSTD_MAKE_CLASS_UNCOPYABLE(VertexGeometry2D);
            VertexGeometry2D& operator=(const VertexGeometry2D&);
            VertexGeometry2D(const VertexGeometry2D& rhs);

            VertexGeometry2D::MemoryPool m_memoryPool; //Specifies the pool in memory.
            VertexGeometry2D::BufferType m_bufferType; //Specifies the type of the VertexGeometry2D.
            VertexGeometry2D::BufferUsage m_bufferUsage; //Gives a hint how the buffer data will be used.
            ResourceDataHandle m_vertexArray2DResourceHandle;
            ResourceDataHandle m_indexArrayResourceHandle;
    };
/** @} */ // end of Core2D
}
#endif //CANDERA_VERTEX_GEOMETRY_2D_H
