//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexGeometry2D.h"

#include <Candera/System/Diagnostics/Log.h>

namespace Candera {

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);

    //Ensure that Vertex2D data only contains the float data in order to be compatible with genereic vertex buffers.
    void CheckVertex2DSize() {
        VertexGeometry2D::Vertex2D toCheck[2];
        FEATSTD_COMPILETIME_ASSERT(sizeof(toCheck) == (8 * sizeof(Float)));
    }

    VertexGeometry2D::VertexGeometry2D(const Vertex2D* vertexArray2D,
        VertexGeometry2DDisposerFn vertexGeometry2DDisposerFn,
        const UInt16* indexBuffer,
        IndexBufferDisposerFn indexBufferDisposerFn,
        UInt32 vertexCount,
        UInt32 indexCount,
        MemoryPool memoryPool,
        BufferType bufferType,
        BufferUsage bufferUsage) :
        Base(),
        m_memoryPool(memoryPool),
        m_bufferType(bufferType),
        m_bufferUsage(bufferUsage),
        m_vertexArray2DResourceHandle(VertexArray2DResource::CreateHandle(vertexArray2D, vertexGeometry2DDisposerFn, vertexCount * sizeof(Vertex2D))),
        m_indexArrayResourceHandle(IndexArrayResource::CreateHandle(indexBuffer, indexBufferDisposerFn, indexCount * sizeof(UInt16)))
    {

    }

    VertexGeometry2D::VertexGeometry2D(Vertex2D* vertexArray2D,
        VertexGeometry2DDisposerFn vertexGeometry2DDisposerFn,
        UInt16* indexBuffer,
        IndexBufferDisposerFn indexBufferDisposerFn,
        UInt32 vertexCount,
        UInt32 indexCount,
        MemoryPool memoryPool,
        BufferType bufferType,
        BufferUsage bufferUsage) :
        Base(),
        m_memoryPool(memoryPool),
        m_bufferType(bufferType),
        m_bufferUsage(bufferUsage),
        m_vertexArray2DResourceHandle(VertexArray2DResource::CreateHandle(vertexArray2D, vertexGeometry2DDisposerFn, vertexCount * sizeof(Vertex2D))),
        m_indexArrayResourceHandle(IndexArrayResource::CreateHandle(indexBuffer, indexBufferDisposerFn, indexCount * sizeof(UInt16)))
    {
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, indexBuffer, CANDERA_LINT_REASON_NONCONST)
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, vertexArray2D, CANDERA_LINT_REASON_NONCONST)
    }


    VertexGeometry2D::VertexGeometry2D(const ResourceDataHandle& vertexArray2DResourceHandle,
        const ResourceDataHandle& indexArrayResourceHandle,
        MemoryPool memoryPool,
        BufferType bufferType,
        BufferUsage bufferUsage) :
        Base(),
        m_memoryPool(memoryPool),
        m_bufferType(bufferType),
        m_bufferUsage(bufferUsage),
        m_vertexArray2DResourceHandle(vertexArray2DResourceHandle),
        m_indexArrayResourceHandle(indexArrayResourceHandle)
    {

    }


    //Implement private Copy constructor to enable cloning.
    VertexGeometry2D::VertexGeometry2D(const VertexGeometry2D& rhs):
        Base(rhs),
        m_memoryPool(rhs.m_memoryPool),
        m_bufferType(rhs.m_bufferType),
        m_bufferUsage(rhs.m_bufferUsage),
        m_vertexArray2DResourceHandle(),
        m_indexArrayResourceHandle()
    {
        ResourceDataHandle vertexArray2DHandle = rhs.GetVertexArray2DResourceHandle();
        VertexGeometry2D::VertexArray2DResource vertexArray2DResource(vertexArray2DHandle);

        ResourceDataHandle indexArrayHandle = rhs.GetIndexArrayResourceHandle();
        VertexGeometry2D::IndexArrayResource indexArrayResource(indexArrayHandle);

        if ((vertexArray2DResource.GetMutableData() == 0) || (vertexArray2DHandle.m_size == 0)) {
            //immutable data, simply create new handle.
            m_vertexArray2DResourceHandle = VertexArray2DResource::CreateHandle(vertexArray2DResource.GetData(), 0, vertexArray2DHandle.m_size);
        }
        else {
            Vertex2D* copiedData = FEATSTD_NEW_ARRAY(Vertex2D, (vertexArray2DHandle.m_size / sizeof(Vertex2D)));
            if (copiedData != 0) {
                static_cast<void>(vertexArray2DResource.CopyData(copiedData, vertexArray2DHandle, 0, vertexArray2DHandle.m_size));
                m_vertexArray2DResourceHandle = VertexArray2DResource::CreateHandle(copiedData, VertexGeometry2DDisposer::Dispose, vertexArray2DHandle.m_size);
            }
            else {
                FEATSTD_LOG_ERROR("Copy of VertexArray2D data could not be allocated. VertexGeometry2D could not be copied.");
            }
        }

        if ((indexArrayResource.GetMutableData() == 0) || (indexArrayHandle.m_size == 0)) {
            //immutable data, simply create new handle.
            m_indexArrayResourceHandle = IndexArrayResource::CreateHandle(indexArrayResource.GetData(), 0, indexArrayHandle.m_size);
        }
        else {
            UInt8* copiedData = FEATSTD_NEW_ARRAY(UInt8, (indexArrayHandle.m_size / sizeof(UInt8)));
            if (copiedData != 0) {
                static_cast<void>(indexArrayResource.CopyData(copiedData, indexArrayHandle, 0, indexArrayHandle.m_size));
                m_indexArrayResourceHandle = IndexArrayResource::CreateHandle(copiedData, IndexBufferDisposer::Dispose, indexArrayHandle.m_size);
            }
            else {
                FEATSTD_LOG_ERROR("Copy of IndexBuffer data could not be allocated. VertexGeometry2D could not be copied.");
            }
        }

    }

    VertexGeometry2D* VertexGeometry2D::Clone() const
    {
        return FEATSTD_NEW(VertexGeometry2D)(*this);
    }


    VertexGeometry2D::~VertexGeometry2D()
    {
        DisposeVertex2DData();
    }


    UInt32 Candera::VertexGeometry2D::GetIndexCount() const
    {
        UInt32 result = 0;

        switch (m_bufferType) {
        case Candera::VertexGeometry2D::UInt16IndexedArrayBuffer:
            result = GetIndexArrayResourceHandle().m_size / sizeof(UInt16);
            break;
        default:
            break;
        }

        return result;
    }


    void VertexGeometry2D::DisposeVertex2DData()
    {
        VertexArray2DResource::DisposeData(m_vertexArray2DResourceHandle);
        IndexArrayResource::DisposeData(m_indexArrayResourceHandle);
    }

} // namespace Candera
